package dev.tazer.clutternomore.common.registry;

import dev.tazer.clutternomore.ClutterNoMore;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.Items;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.block.*;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static dev.tazer.clutternomore.ClutterNoMore.MODID;

public class BlockSetRegistry {

    public static ShapeSet getBlockTypeOf(Item item) {
        return ShapeSetRegistry.items.get(item);
    }

    public static class ShapeSetRegistry {
        public static final Map<Item, ShapeSet> items = new LinkedHashMap<>();

        public static void register(Item block, ShapeSet shapeSet) {
            items.put(block, shapeSet);
        }

        public static void detectTypeFromBlock(Block block, ResourceLocation blockId) {
            if (block.asItem() != Items.AIR) {
                if (isParentBlock(blockId)) {
                    new ShapeSet(blockId, block);
                }
            }

        }

        private static boolean has(ResourceLocation block) {
            return BuiltInRegistries.BLOCK.containsKey(block);
        }

        private static boolean isParentBlock(ResourceLocation block) {
            List<String> namespaces = List.of(block.getNamespace(), "minecraft");
            String path = block.getPath();

            List<String> parentSuffixes = List.of("log", "planks", "block");
            List<String> ignoredSuffixes = List.of("block");
            Map<String, String> replacements = Map.of("log", "wood");
            List<String> prefixes = List.of("spiked");
            List<String> ignoredPrefixes = List.of("stripped");
            List<String> suffixes = List.of("stairs", "slab", "wall");

            for (String namespace : namespaces) {
                ResourceLocation base = ClutterNoMore.location(namespace, path);

                String suffixBase = path;
                for (String ignored : ignoredSuffixes) {
                    suffixBase = stripSuffix(suffixBase, ignored);
                }

                for (String parent : parentSuffixes) {
                    String suffixed = suffixBase + "_" + parent;
                    if (!suffixed.equals(path)) {
                        ResourceLocation candidate = base.withPath(p -> suffixed);
                        if (has(candidate)) return false;
                    }
                }

                for (Map.Entry<String, String> replacement : replacements.entrySet()) {
                    String replaced = path.replace(replacement.getKey(), replacement.getValue());
                    if (!replaced.equals(path)) {
                        ResourceLocation candidate = base.withPath(p -> replaced);
                        if (has(candidate)) return true;
                    }
                }

                String prefixBase = path;
                for (String ignored : ignoredPrefixes) {
                    prefixBase = stripPrefix(prefixBase, ignored);
                }

                for (String prefix : prefixes) {
                    String prefixed = prefix + "_" + prefixBase;
                    if (!prefixed.equals(path)) {
                        ResourceLocation candidate = base.withPath(p -> prefixed);
                        if (has(candidate)) return true;
                    }
                }

                for (String parent : parentSuffixes) {
                    suffixBase = stripSuffix(suffixBase, parent);
                }

                for (String suffix : suffixes) {
                    if (suffixBase.endsWith("s")) {
                        String trimmed  = suffixBase.substring(0, suffixBase.length() - 1);
                        String suffixed = trimmed + "_" + suffix;
                        if (!suffixed.equals(path)) {
                            ResourceLocation candidate = base.withPath(p -> suffixed);
                            if (has(candidate)) return true;
                        }
                    }

                    String suffixed = suffixBase + "_" + suffix;
                    if (!suffixed.equals(path)) {
                        ResourceLocation candidate = base.withPath(p -> suffixed);
                        if (has(candidate)) return true;
                    }
                }
            }

            return false;
        }
    }

    public static class ShapeSet {
        private final ResourceLocation id;
        private final String type;
        private final Block block;
        private final Map<String, ItemLike> items = new LinkedHashMap<>();

        ShapeSet(ResourceLocation id, Block block) {
            this.id = id;
            this.block = block;
            this.type = id.getPath().replace("_block", "").replace("_planks", "");
            initializeChildrenBlocks();
            ShapeSetRegistry.register(block.asItem(), this);

        }

        protected void initializeChildrenBlocks() {
            addChild("block", block);
            if (id.getPath().contains("log")) {
                addChild("wood", getWood());
                if (id.getPath().contains("stripped")) {
                    addChild("hollow_log",ClutterNoMore.location("wilderwild", id.getPath().replace("stripped_", "stripped_hollowed_")));
                } else {
                    addChild("hollow_log", ClutterNoMore.location("wilderwild", "hollowed_"+ id.getPath()));
                }
            }
            addChild("slab", findShape("slab"));
            addChild("stairs", findShape("stairs"));
            addChild("wall", findShape("wall"));
            addChild("vertical_slab", findShape(MODID,id.getNamespace() + "/vertical", "slab"));
            addChild("vertical_slab", findShape(MODID,"vertical", "slab"));
            addChild("vertical_slab", findShape("vertical", "slab"));
            addChild("step", findShape(MODID, id.getNamespace() + "/","step", false));
            addChild("step", findShape(MODID, "","step"));
            addChild("step", findShape("", "step"));
            addChild("spiked", findShape("spiked", ""));
        }

        private void addChild(String block, ItemLike block1) {
            if (block1 != null && !items.containsKey(block))
                items.put(block, block1);
        }

        private void addChild(String block, ResourceLocation id) {
            BuiltInRegistries.ITEM.getOptional(id).ifPresent(block1 -> items.put(block, block1));
        }

        public String getVariantId(String prefix, String postfix) {
            String name = prefix + (prefix.isEmpty() ? "" : "_") + getTypeName();

            List<String> suffixes = List.of("_block", "_planks", "s");

            for (String suffix : suffixes) {
                if (name.endsWith(suffix)) {
                    name = name.substring(0, name.length() - suffix.length());
                    break;
                }
            }

            return name + (postfix.isEmpty() ? "" : "_") + postfix;
        }

        private String getTypeName() {
            return type;
        }

        protected @Nullable Item findShape(String prefix, String postfix) {
            return findShape(id.getNamespace(), prefix, postfix);
        }

        protected @Nullable Item findShape(String namespace, String prefix, String postfix) {
            return findShape(namespace, prefix, postfix, true);
        }

            protected @Nullable Item findShape(String namespace, String prefix, String postfix, boolean addUnderscore) {
            String basePath = id.getPath();

            List<String> parentSuffixes = List.of("block", "planks");
            for (String parent : parentSuffixes) {
                basePath = stripSuffix(basePath, parent);
            }

            List<String> ignoredPrefixes = List.of("stripped");
            String reapplyPrefix = "";
            for (String ignored : ignoredPrefixes) {
                if (id.getPath().startsWith(ignored + "_")) {
                    reapplyPrefix = ignored + "_";
                    basePath = stripPrefix(basePath, ignored);
                }
            }

            String prefixPart = prefix.isEmpty() ? "" : prefix + (addUnderscore ? "_" : "");
            String postfixPart = postfix.isEmpty() ? "" : "_" + postfix;

            List<String> candidates = new ArrayList<>();
            if (basePath.endsWith("s")) {
                candidates.add(basePath.substring(0, basePath.length() - 1));
            }
            candidates.add(basePath);

            for (String stem : candidates) {
                String candidatePath = reapplyPrefix + prefixPart + stem + postfixPart;
                ResourceLocation candidateId = ClutterNoMore.location(namespace, candidatePath);
                Optional<Item> found = BuiltInRegistries.ITEM.getOptional(candidateId);
                if (found.isPresent()) return found.get();
            }

            return null;
        }

        protected @Nullable Item findShape(String postfix) {
            return findShape("", postfix);
        }

        protected @Nullable Item getWood() {
            String path = id.getPath();
            if (path.endsWith("log")) {
                String stem = path.substring(0, path.length() - 3);
                ResourceLocation woodId = id.withPath(p -> stem + "wood");
                return BuiltInRegistries.ITEM.getOptional(woodId).orElse(null);
            }

            return null;
        }

        public Block mainChild() {
            return block;
        }

        public Collection<ItemLike> getChildren() {
            return items.values();
        }
    }

    private static String stripSuffix(String path, String suffix) {
        if (path.endsWith("_" + suffix)) return path.substring(0, path.length() - suffix.length() - 1);
        if (path.endsWith(suffix)) return path.substring(0, path.length() - suffix.length());
        return path;
    }

    private static String stripPrefix(String path, String prefix) {
        if (path.startsWith(prefix + "_")) return path.substring(prefix.length() + 1);
        if (path.startsWith(prefix)) return path.substring(prefix.length());
        return path;
    }
}