/*
 * Decompiled with CFR 0.152.
 */
package de.pianoman911.mapengine.core.colors;

import com.google.common.io.ByteArrayDataInput;
import com.google.common.io.ByteArrayDataOutput;
import com.google.common.io.ByteStreams;
import de.pianoman911.mapengine.api.colors.IMapColors;
import de.pianoman911.mapengine.api.util.ColorBuffer;
import de.pianoman911.mapengine.api.util.Converter;
import de.pianoman911.mapengine.api.util.FullSpacedColorBuffer;
import de.pianoman911.mapengine.core.MapEnginePlugin;
import de.pianoman911.mapengine.core.colors.dithering.FloydSteinbergDithering;
import java.awt.Color;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashSet;
import java.util.concurrent.CompletableFuture;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.bukkit.Bukkit;
import org.bukkit.map.MapPalette;
import org.bukkit.plugin.Plugin;

public class ColorPalette
implements IMapColors {
    protected final CompletableFuture<ColorPalette> loadFuture = new CompletableFuture();
    private final MapEnginePlugin plugin;
    private final File file;
    protected byte[] colors;
    protected byte[] available;
    protected int[] rgb;
    protected int[] reverseColors;
    protected int retries = 0;

    public ColorPalette(MapEnginePlugin plugin) {
        this.plugin = plugin;
        this.file = new File(plugin.getDataFolder(), "colors.bin");
        this.load();
    }

    @Override
    public final byte color(Color color) {
        return this.color(color.getRGB());
    }

    @Override
    public final byte color(int rgb) {
        if ((rgb >> 24 & 0xFF) < 128) {
            return 0;
        }
        return this.colors[rgb & 0xFFFFFF];
    }

    @Override
    public byte[] colors(int[] rgb, int threads) {
        byte[] result = new byte[rgb.length];
        CompletableFuture[] futures = new CompletableFuture[threads];
        int size = rgb.length / threads;
        for (int i = 0; i < threads; ++i) {
            int finalI = i;
            futures[i] = CompletableFuture.runAsync(() -> {
                int start = finalI * size;
                int end = (finalI + 1) * size;
                if (finalI == threads - 1) {
                    end = rgb.length;
                }
                for (int j = start; j < end; ++j) {
                    int color = rgb[j];
                    if ((color >> 24 & 0xFF) < 128) continue;
                    result[j] = this.color(rgb[j]);
                }
            });
        }
        CompletableFuture.allOf(futures).join();
        return result;
    }

    @Override
    public byte[] convertImage(BufferedImage image) {
        int[] rgb = new int[image.getWidth() * image.getHeight()];
        image.getRGB(0, 0, image.getWidth(), image.getHeight(), rgb, 0, image.getWidth());
        return this.colors(rgb);
    }

    @Override
    public FullSpacedColorBuffer adjustColors(FullSpacedColorBuffer buffer, Converter converter) {
        return switch (converter) {
            default -> throw new IncompatibleClassChangeError();
            case Converter.DIRECT -> {
                ColorBuffer mc = this.convertDirect(buffer);
                yield new FullSpacedColorBuffer(this.plugin.colorPalette().toRGBs(mc.data()), mc.width(), mc.height());
            }
            case Converter.FLOYD_STEINBERG -> {
                ColorBuffer mc = FloydSteinbergDithering.dither(buffer, this.plugin.colorPalette(), buffer.height() / 128 + 1);
                yield new FullSpacedColorBuffer(this.plugin.colorPalette().toRGBs(mc.data()), mc.width(), mc.height());
            }
        };
    }

    @Override
    public final Color toColor(byte color) {
        return new Color(this.toRGB(color));
    }

    @Override
    public final byte color(int r, int g, int b) {
        return this.colors[this.dataIndex(r, g, b)];
    }

    private final int dataIndex(int r, int g, int b) {
        return (r & 0xFF) << 16 | (g & 0xFF) << 8 | b & 0xFF;
    }

    private void generateColors() {
        MapPalette.matchColor((int)0, (int)0, (int)0);
        Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            long start;
            this.plugin.getLogger().info("Generating palette...");
            this.colors = new byte[0x1000000];
            this.reverseColors = new int[0x1000000];
            long last = start = System.currentTimeMillis();
            HashSet usedColors = new HashSet();
            for (int red = 0; red < 256; ++red) {
                CompletableFuture[] futures = new CompletableFuture[256];
                for (int green = 0; green < 256; ++green) {
                    int finalRed = red;
                    int finalGreen = green;
                    futures[green] = CompletableFuture.supplyAsync(() -> {
                        for (int blue = 0; blue < 256; ++blue) {
                            byte color = MapPalette.matchColor((int)finalRed, (int)finalGreen, (int)blue);
                            int index = this.dataIndex(finalRed, finalGreen, blue);
                            this.colors[index] = color;
                            this.reverseColors[index] = index;
                            usedColors.add(color);
                        }
                        return null;
                    });
                }
                if (last + 250L < System.currentTimeMillis() || red == 255) {
                    this.plugin.getLogger().info("Generating palette... " + String.format("%.2f", (double)(red * 100) / 255.0) + "% - Working threads: " + futures.length);
                    last = System.currentTimeMillis();
                }
                CompletableFuture.allOf(futures).join();
            }
            this.available = new byte[usedColors.size()];
            this.rgb = new int[255];
            int i = 0;
            for (Byte color : usedColors) {
                this.available[i++] = color;
                this.rgb[color.byteValue() >= 0 ? color.byteValue() : color.byteValue() + 256] = MapPalette.getColor((byte)color).getRGB();
            }
            this.plugin.getLogger().info("Palette generated! Took " + (System.currentTimeMillis() - start) + "ms");
            this.save();
            this.loadFuture.complete(this);
            this.checkRetry();
        });
    }

    private void save() {
        this.plugin.getLogger().info("Saving palette...");
        ByteArrayDataOutput output = ByteStreams.newDataOutput();
        output.writeInt(Bukkit.getUnsafe().getDataVersion());
        output.writeInt(this.colors.length);
        for (byte color : this.colors) {
            output.writeByte((int)color);
        }
        try (FileOutputStream fileOutputStream = new FileOutputStream(this.file);
             GZIPOutputStream gzip = new GZIPOutputStream(fileOutputStream);){
            gzip.write(output.toByteArray());
        }
        catch (IOException exception) {
            throw new RuntimeException(exception);
        }
        this.plugin.getLogger().info("Palette saved!");
    }

    private void load() {
        block19: {
            if (!this.file.exists()) {
                this.file.getParentFile().mkdir();
                this.generateColors();
                return;
            }
            try (FileInputStream fileInput = new FileInputStream(this.file);){
                int i;
                GZIPInputStream gzipInput = new GZIPInputStream(fileInput);
                ByteArrayDataInput input = ByteStreams.newDataInput((byte[])gzipInput.readAllBytes());
                if (input.readInt() != Bukkit.getUnsafe().getDataVersion()) break block19;
                HashSet<Byte> usedColors = new HashSet<Byte>();
                this.colors = new byte[input.readInt()];
                this.reverseColors = new int[this.colors.length];
                for (i = 0; i < this.colors.length; ++i) {
                    this.colors[i] = input.readByte();
                    this.reverseColors[i] = MapPalette.getColor((byte)this.colors[i]).getRGB();
                    usedColors.add(this.colors[i]);
                }
                this.available = new byte[usedColors.size()];
                this.rgb = new int[255];
                i = 0;
                for (Byte color : usedColors) {
                    this.available[i++] = color;
                    this.rgb[color.byteValue() >= 0 ? color.byteValue() : color.byteValue() + 256] = MapPalette.getColor((byte)color).getRGB();
                }
                if (!this.checkValidity()) {
                    this.plugin.getLogger().warning("Incompatible color palette saved, regenerating...");
                    this.generateColors();
                    return;
                }
                this.plugin.getLogger().info("Loaded color palette");
                this.loadFuture.complete(this);
                return;
                finally {
                    gzipInput.close();
                }
            }
            catch (IOException exception) {
                throw new RuntimeException(exception);
            }
        }
        this.plugin.getLogger().info("Incompatible color palette saved");
        this.generateColors();
    }

    @Override
    public final int toRGB(byte color) {
        return this.rgb[color >= 0 ? color : color + 256];
    }

    @Override
    public final int[] toRGBs(byte[] colors) {
        int[] rgb = new int[colors.length];
        for (int i = 0; i < colors.length; ++i) {
            rgb[i] = this.toRGB(colors[i]);
        }
        return rgb;
    }

    public final int closestColor(int rgb) {
        int alpha = rgb >> 24 & 0xFF;
        if (alpha == 0) {
            return 0;
        }
        int ret = this.reverseColors[rgb & 0xFFFFFF];
        return ret & 0xFFFFFF | alpha << 24;
    }

    private void checkRetry() {
        this.loadFuture.thenAccept($ -> {
            boolean valid = this.checkValidity();
            if (valid) {
                this.plugin.getLogger().info("Color palette is valid!" + (String)(this.retries > 0 ? " (Retried " + this.retries + " times)" : ""));
                return;
            }
            this.plugin.getLogger().warning("Color palette is invalid!" + (String)(this.retries > 0 ? " (Retried " + this.retries + " times)" : ""));
            if (this.retries < 10) {
                ++this.retries;
                this.plugin.getLogger().warning("Retrying... (" + this.retries + "/10)");
                this.generateColors();
            } else {
                this.plugin.getLogger().warning("Failed to load color palette!");
                Bukkit.getPluginManager().disablePlugin((Plugin)this.plugin);
            }
        });
    }

    private boolean checkValidity() {
        boolean valid = true;
        try {
            for (byte i = -128; i < 127; i = (byte)(i + 1)) {
                int bukkit;
                int engine = this.toRGB(i);
                if (engine == (bukkit = MapPalette.getColor((byte)i).getRGB())) continue;
                this.plugin.getLogger().warning("Color " + i + " is invalid! MapEngine: " + engine + " Bukkit: " + bukkit);
                valid = false;
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        return valid;
    }
}

