/*
 * Decompiled with CFR 0.152.
 */
package fr.siroz.cariboustonks.core.data.hypixel.fetcher;

import com.google.gson.annotations.SerializedName;
import fr.siroz.cariboustonks.CaribouStonks;
import fr.siroz.cariboustonks.core.data.hypixel.HypixelDataSource;
import fr.siroz.cariboustonks.core.data.hypixel.bazaar.BazaarItemAnalytics;
import fr.siroz.cariboustonks.core.data.hypixel.bazaar.BazaarProduct;
import fr.siroz.cariboustonks.core.json.GsonProvider;
import fr.siroz.cariboustonks.core.scheduler.AsyncScheduler;
import fr.siroz.cariboustonks.core.scheduler.TickScheduler;
import fr.siroz.cariboustonks.util.StonksUtils;
import fr.siroz.cariboustonks.util.http.Http;
import fr.siroz.cariboustonks.util.http.HttpResponse;
import java.time.Instant;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BooleanSupplier;
import org.apache.http.client.HttpResponseException;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;

@ApiStatus.Internal
public final class BazaarFetcher {
    private static final String BAZAAR_URL = "https://api.hypixel.net/v2/skyblock/bazaar";
    private final HypixelDataSource hypixelDataSource;
    private final int intervalInMinutes;
    private final BooleanSupplier shouldFetch;
    private final AtomicBoolean fetchInProgress;
    private final AtomicBoolean lastFetchSuccessful;
    private final AtomicReference<Map<String, BazaarProduct>> bazaarCache;
    private volatile Instant lastBazaarUpdate;
    private volatile boolean firstBazaarUpdated;

    public BazaarFetcher(HypixelDataSource hypixelDataSource, int intervalInMinutes, @NotNull BooleanSupplier shouldFetch) {
        this.hypixelDataSource = hypixelDataSource;
        this.intervalInMinutes = intervalInMinutes;
        this.shouldFetch = shouldFetch;
        this.fetchInProgress = new AtomicBoolean(false);
        this.lastFetchSuccessful = new AtomicBoolean(false);
        this.bazaarCache = new AtomicReference(Map.of());
        this.lastBazaarUpdate = null;
        this.firstBazaarUpdated = false;
        CaribouStonks.LOGGER.info("[BazaarFetcher] Bazaar products will be updated every {} minute", (Object)intervalInMinutes);
    }

    public void start() {
        TickScheduler.getInstance().runRepeating(() -> {
            if (this.shouldFetch.getAsBoolean()) {
                this.triggerFetch().thenRun(() -> {
                    if (this.lastFetchSuccessful.get()) {
                        CaribouStonks.LOGGER.info("[BazaarFetcher] {} products updated. (last update: {})", (Object)this.bazaarCache.get().size(), (Object)(this.lastBazaarUpdate == null ? "n/a" : this.lastBazaarUpdate.toString()));
                    }
                    if (!this.firstBazaarUpdated) {
                        this.firstBazaarUpdated = true;
                        this.hypixelDataSource.fixSkyBlockItems();
                    }
                });
            }
        }, this.intervalInMinutes, TimeUnit.MINUTES);
    }

    public Map<String, BazaarProduct> getBazaarSnapshot() {
        return this.bazaarCache.get();
    }

    public boolean isFirstBazaarUpdated() {
        return this.firstBazaarUpdated;
    }

    public boolean isFetching() {
        return this.fetchInProgress.get();
    }

    @NotNull
    private CompletableFuture<Void> triggerFetch() {
        this.fetchInProgress.set(true);
        this.lastFetchSuccessful.set(false);
        CompletionStage<Void> promise = CompletableFuture.runAsync(this::executeFetch, AsyncScheduler.getInstance().blockingExecutor());
        promise = ((CompletableFuture)promise).exceptionallyCompose(throwable -> {
            Throwable cause = throwable instanceof CompletionException ? throwable.getCause() : throwable;
            CaribouStonks.LOGGER.error("[BazaarFetcher] Fetch Bazaar failed.", cause);
            return CompletableFuture.completedFuture(null);
        });
        promise = ((CompletableFuture)promise).whenComplete((v, t) -> this.fetchInProgress.set(false));
        return promise;
    }

    private void executeFetch() {
        try (HttpResponse response = Http.request(BAZAAR_URL);){
            if (!response.success()) {
                throw new HttpResponseException(response.statusCode(), response.content());
            }
            SkyBlockBazaarReply reply = (SkyBlockBazaarReply)GsonProvider.prettyPrinting().fromJson(response.content(), SkyBlockBazaarReply.class);
            if (reply == null) {
                throw new IllegalStateException("SkyBlock API Bazaar returned null reply");
            }
            if (!reply.success) {
                String cause = reply.cause != null ? reply.cause : "?";
                throw new RuntimeException("SkyBlock API Bazaar failed. Cause: " + cause);
            }
            if (reply.products == null || reply.products.isEmpty()) {
                throw new RuntimeException("SkyBlock API Bazaar returned empty products");
            }
            this.bazaarCache.set(this.computeAndConsumeProducts(reply.products));
            this.lastBazaarUpdate = reply.lastUpdated > 0L ? Instant.ofEpochSecond(reply.lastUpdated) : Instant.MIN;
            this.lastFetchSuccessful.set(true);
            reply.products = null;
        }
        catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    @NotNull
    private Map<String, BazaarProduct> computeAndConsumeProducts(@NotNull Map<String, HypixelProduct> products) {
        HashMap<String, BazaarProduct> result = new HashMap<String, BazaarProduct>(Math.max(16, products.size()));
        Iterator<Map.Entry<String, HypixelProduct>> it = products.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, HypixelProduct> entry = it.next();
            result.put(entry.getKey(), this.computeProduct(entry.getValue()));
            it.remove();
        }
        return result;
    }

    @Contract(value="_ -> new")
    @NotNull
    private BazaarProduct computeProduct(@NotNull HypixelProduct product) {
        Status qs = product.quickStatus();
        List<Double> buyPrices = product.buySummary().stream().map(Summary::pricePerUnit).toList();
        List<Double> sellPrices = product.sellSummary().stream().map(Summary::pricePerUnit).toList();
        double buyPrice = BazaarItemAnalytics.buyPrice(buyPrices);
        double sellPrice = BazaarItemAnalytics.sellPrice(sellPrices);
        double spread = BazaarItemAnalytics.spread(buyPrice, sellPrice);
        double spreadPercentage = BazaarItemAnalytics.spreadPercentage(buyPrice, sellPrice);
        double buyMedianPrice = StonksUtils.calculateMedian(buyPrices);
        double sellMedianPrice = StonksUtils.calculateMedian(sellPrices);
        double buyPriceStdDev = BazaarItemAnalytics.standardDeviation(buyPrices);
        double sellPriceStdDev = BazaarItemAnalytics.standardDeviation(sellPrices);
        double buyVelocity = BazaarItemAnalytics.priceVelocity(qs.buyVolume(), qs.buyMovingWeek());
        double sellVelocity = BazaarItemAnalytics.priceVelocity(qs.sellVolume(), qs.sellMovingWeek());
        return new BazaarProduct(product.productId(), buyPrice, sellPrice, qs.buyPrice(), qs.sellPrice(), qs.buyVolume(), qs.sellVolume(), qs.buyMovingWeek(), qs.sellMovingWeek(), qs.buyOrders(), qs.sellOrders(), spread, spreadPercentage, buyMedianPrice, sellMedianPrice, buyPriceStdDev, sellPriceStdDev, buyVelocity, sellVelocity);
    }

    private static class SkyBlockBazaarReply {
        public boolean success;
        public String cause;
        public long lastUpdated;
        public Map<String, HypixelProduct> products;

        private SkyBlockBazaarReply() {
        }
    }

    private record HypixelProduct(@SerializedName(value="product_id") String productId, @SerializedName(value="sell_summary") List<Summary> sellSummary, @SerializedName(value="buy_summary") List<Summary> buySummary, @SerializedName(value="quick_status") Status quickStatus) {
    }

    private record Status(double sellPrice, long sellVolume, long sellMovingWeek, long sellOrders, double buyPrice, long buyVolume, long buyMovingWeek, long buyOrders) {
    }

    private record Summary(long amount, double pricePerUnit, long orders) {
    }
}

