// "Sampling Visible GGX Normals with Spherical Caps" 
// https://arxiv.org/abs/2306.05044

// Sampling the visible hemisphere as half vectors (our method)
vec3 SampleVndf_Hemisphere(vec2 u, vec3 wi) {
    // sample a spherical cap in (-wi.z, 1]
    float phi = 2.0f * PI * u.x;
    float z = fma((1.0f - u.y), (1.0f + wi.z), -wi.z);
    float sinTheta = sqrt(clamp(1.0f - z * z, 0.0f, 1.0f));
    float x = sinTheta * cos(phi);
    float y = sinTheta * sin(phi);
    vec3 c = vec3(x, y, z);
    // compute halfway direction;
    vec3 h = c + wi;
    // return without normalization (as this is done later)
    return h;
}

vec3 SampleVndf_GGX(vec2 u, vec3 wi, vec2 alpha) {
    // warp to the hemisphere configuration
    vec3 wiStd = normalize(vec3(wi.xy * alpha, wi.z));
    // sample the hemisphere (see implementation 2 or 3)
    vec3 wmStd = SampleVndf_Hemisphere(u, wiStd);
    // warp back to the ellipsoid configuration
    vec3 wm = normalize(vec3(wmStd.xy * alpha, wmStd.z));
    // return final normal
    return wm;
}

bool raytrace(vec3 ScreenPos, vec3 ViewPos, vec3 Dir, bool IsDH, float Dither, out vec3 RayPos) {
    int Steps = SSR_STEPS;
    vec3 Offset = normalize(view_screen(ViewPos + Dir, IsDH, true) - ScreenPos);
    vec3 Len = (step(0, Offset) - ScreenPos) / Offset;
    float MinLen = min(Len.x, min(Len.y, Len.z)) / Steps;
    Offset *= MinLen;

    RayPos = ScreenPos + Offset * Dither;
    for (int i = 1; i <= Steps; i++) {
        float RealDepth = get_depth_solid(RayPos.xy, IsDH);
        if (RealDepth < 0.56) {
            break;
        }
        if (RayPos.z > RealDepth) {
            // Depth based rejection
            if (RayPos.z - RealDepth > abs(Offset.z) * 4) break;

            // Binary refinement
            for (int i = 1; i <= Steps / 6; i++) {
                Offset /= 2;
                vec3 EPos1 = RayPos - Offset;
                float RDepth1 = get_depth_solid(RayPos.xy, IsDH);
                if (EPos1.z > RDepth1) {
                    RayPos = EPos1;
                }
            }
            return true;
        }
        RayPos += Offset;
    }
    return false;
}

bool flipped_image_ref(vec3 ViewPos, vec3 RVec, inout bool IsDH, out vec3 RayPos) {
    #ifdef DISTANT_HORIZONS
    float Offset = min(1000, 50 + dhRenderDistance / 4);
    #else
    float Offset = 50 + far / 4;
    #endif

    RayPos = view_screen(ViewPos + RVec * Offset, IsDH, true);
    if(RayPos.xy == clamp(RayPos.xy, 0, 1)) {
        float RealDepth = get_depth_solid(RayPos.xy, IsDH);
        if(RayPos.z < 1 && RayPos.z > 0.56 && RealDepth < RayPos.z) {
            RayPos.z = RealDepth;
            vec3 ViewPosReal = screen_view(RayPos, IsDH, true);
            if(len2(ViewPosReal) + 25 > len2(ViewPos)) {
                return true;
            }
        }
    }
    return false;
}

vec3 ssr(vec3 Normal, Positions Pos, bool IsDH, float LightmapSky, float Dither) {
    vec3 Dir = reflect(Pos.ViewN, Normal);

    vec3 RayPos; 
    bool Hit = raytrace(Pos.Screen, Pos.View, Dir, IsDH, Dither, RayPos);
    #ifdef DISTANT_HORIZONS
        if(!Hit) {
            Hit = flipped_image_ref(Pos.View, Dir, IsDH, RayPos);
        }
    #endif

    // if(Hit) return vec3(100, 0,0 );

    if(Hit) {
        vec3 SkyColor = get_sky(Dir, false, 1);

        vec3 TerrainColor = texture(colortex0, RayPos.xy).rgb;
        vec3 StartPos = Pos.Player;
        vec3 EndPos = view_player(screen_view(RayPos, IsDH, true));
        vec3 RefColor = do_vl(StartPos, EndPos, view_player(Dir), TerrainColor, Pos.Screen, IsDH, 3, false, false, false);

        float Dist = length(EndPos);
        #ifdef DISTANT_HORIZONS
            Dist /= dhRenderDistance;
        #else
            Dist /= far;
        #endif
        return get_border_fog(Dist, RefColor, SkyColor);
    } else {
        if (LightmapSky < 0.1) return vec3(0);

        vec3 StartPos = Pos.Player;
        vec3 EndPos = view_player(Dir);
        vec3 SkyColor = get_sky(Dir, false, EndPos.y);
        #if (defined CLOUDS) && (defined DIMENSION_OVERWORLD) 
            vec3 P = EndPos;
            vec2 CloudPos = vec2(P.x, P.z) / (1 + P.y);
            SkyColor += texture(colortex3, (CloudPos * 0.5 + 0.5) * resolutionInv * CLOUD_TEX_SIZE).rgb;
        #endif

        //SkyColor += get_stars(EndPos); // Not bright enough to be noticed
        return do_vl(StartPos, vec3(0, 10000, 0), EndPos, SkyColor, Pos.Screen, IsDH, 3, false, false, false) * LightmapSky;
    }
}
