/*
 * Decompiled with CFR 0.152.
 */
package de.melanx.simplebackups;

import de.melanx.simplebackups.BackupData;
import de.melanx.simplebackups.SimpleBackups;
import de.melanx.simplebackups.StorageSize;
import de.melanx.simplebackups.compat.Mc2DiscordCompat;
import de.melanx.simplebackups.config.BackupType;
import de.melanx.simplebackups.config.CommonConfig;
import de.melanx.simplebackups.config.ServerConfig;
import de.melanx.simplebackups.network.Pause;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.SignStyle;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Objects;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import net.minecraft.ChatFormatting;
import net.minecraft.DefaultUncaughtExceptionHandler;
import net.minecraft.FileUtil;
import net.minecraft.network.Connection;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.chat.Style;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.level.storage.LevelStorageSource;
import net.neoforged.fml.i18n.FMLTranslations;
import net.neoforged.neoforge.network.registration.NetworkRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupThread
extends Thread {
    private static final DateTimeFormatter FORMATTER = new DateTimeFormatterBuilder().appendValue(ChronoField.YEAR, 4, 10, SignStyle.EXCEEDS_PAD).appendLiteral('-').appendValue(ChronoField.MONTH_OF_YEAR, 2).appendLiteral('-').appendValue(ChronoField.DAY_OF_MONTH, 2).appendLiteral('_').appendValue(ChronoField.HOUR_OF_DAY, 2).appendLiteral('-').appendValue(ChronoField.MINUTE_OF_HOUR, 2).appendLiteral('-').appendValue(ChronoField.SECOND_OF_MINUTE, 2).toFormatter();
    public static final Logger LOGGER = LoggerFactory.getLogger(BackupThread.class);
    private final MinecraftServer server;
    private final boolean quiet;
    private final long lastSaved;
    private final boolean fullBackup;
    private final LevelStorageSource.LevelStorageAccess storageSource;
    private final Path backupPath;

    private BackupThread(@Nonnull MinecraftServer server, boolean quiet, BackupData backupData) {
        this.server = server;
        this.storageSource = server.storageSource;
        this.quiet = quiet;
        if (backupData == null) {
            this.lastSaved = 0L;
            this.fullBackup = true;
        } else {
            long l = this.lastSaved = CommonConfig.backupType() == BackupType.MODIFIED_SINCE_LAST ? backupData.getLastSaved() : backupData.getLastFullBackup();
            this.fullBackup = CommonConfig.backupType() == BackupType.FULL_BACKUPS || (CommonConfig.useTickCounter() ? server.overworld().getGameTime() : System.currentTimeMillis()) - (long)CommonConfig.getFullBackupTimer() > backupData.getLastFullBackup();
        }
        this.setName("SimpleBackups");
        this.setUncaughtExceptionHandler((Thread.UncaughtExceptionHandler)new DefaultUncaughtExceptionHandler(LOGGER));
        this.backupPath = CommonConfig.getOutputPath(this.storageSource.levelId);
    }

    public static boolean tryCreateBackup(MinecraftServer server) {
        BackupData backupData = BackupData.get(server);
        if (BackupThread.shouldRunBackup(server)) {
            BackupThread thread = new BackupThread(server, false, backupData);
            thread.start();
            long currentTime = CommonConfig.useTickCounter() ? server.overworld().getGameTime() : System.currentTimeMillis();
            backupData.updateSaveTime(currentTime);
            if (thread.fullBackup) {
                backupData.updateFullBackupTime(currentTime);
            }
            return true;
        }
        return false;
    }

    public static boolean shouldRunBackup(MinecraftServer server) {
        BackupData backupData = BackupData.get(server);
        if (!CommonConfig.isEnabled() || backupData.isPaused()) {
            return false;
        }
        if (CommonConfig.useTickCounter()) {
            int timer;
            long lastSaved;
            long gameTime = server.overworld().getGameTime();
            return gameTime - (lastSaved = backupData.getLastSaved()) >= (long)(timer = CommonConfig.getTimer() * 20 * 60);
        }
        return System.currentTimeMillis() - (long)CommonConfig.getTimer() > backupData.getLastSaved();
    }

    public static void createBackup(MinecraftServer server, boolean quiet) {
        BackupThread thread = new BackupThread(server, quiet, null);
        thread.start();
    }

    public void deleteFiles() {
        ArrayList<File> files;
        File backups = this.backupPath.toFile();
        if (backups.isDirectory() && (files = new ArrayList<File>(Arrays.stream(Objects.requireNonNull(backups.listFiles())).filter(File::isFile).toList())).size() >= CommonConfig.getBackupsToKeep()) {
            files.sort(Comparator.comparingLong(File::lastModified));
            while (files.size() >= CommonConfig.getBackupsToKeep()) {
                boolean deleted = ((File)files.getFirst()).delete();
                String name = ((File)files.getFirst()).getName();
                if (!deleted) continue;
                files.removeFirst();
                LOGGER.info("Successfully deleted \"{}\"", (Object)name);
            }
        }
    }

    public void saveStorageSize() {
        try {
            while (this.getOutputFolderSize() > CommonConfig.getMaxDiskSize()) {
                File[] files = this.backupPath.toFile().listFiles();
                if (Objects.requireNonNull(files).length == 1) {
                    LOGGER.error("Cannot delete old files to save disk space. Only one backup file left!");
                    return;
                }
                Arrays.sort(Objects.requireNonNull(files), Comparator.comparingLong(File::lastModified));
                File file = files[0];
                String name = file.getName();
                if (!file.delete()) continue;
                LOGGER.info("Successfully deleted \"{}\"", (Object)name);
            }
        }
        catch (IOException | NullPointerException e) {
            LOGGER.error("Cannot delete old files to save disk space", (Throwable)e);
        }
    }

    @Override
    public void run() {
        try {
            this.deleteFiles();
            Files.createDirectories(this.backupPath, new FileAttribute[0]);
            long start = System.currentTimeMillis();
            this.broadcast("simplebackups.backup_started", Style.EMPTY.withColor(ChatFormatting.GOLD), new Object[0]);
            long size = this.makeWorldBackup();
            long end = System.currentTimeMillis();
            String time = Timer.getTimer(end - start);
            this.saveStorageSize();
            this.broadcast("simplebackups.backup_finished", Style.EMPTY.withColor(ChatFormatting.GOLD), time, StorageSize.getFormattedSize(size), StorageSize.getFormattedSize(this.getOutputFolderSize()));
        }
        catch (IOException e) {
            SimpleBackups.LOGGER.error("Error backing up", (Throwable)e);
        }
    }

    private long getOutputFolderSize() throws IOException {
        File[] files = this.backupPath.toFile().listFiles();
        long size = 0L;
        try {
            for (File file : Objects.requireNonNull(files)) {
                size += Files.size(file.toPath());
            }
        }
        catch (NullPointerException e) {
            return 0L;
        }
        return size;
    }

    private void broadcast(String message, Style style, Object ... parameters) {
        if (CommonConfig.sendMessages() && !this.quiet) {
            this.server.execute(() -> this.server.getPlayerList().getPlayers().forEach(player -> {
                if (ServerConfig.messagesForEveryone() || player.hasPermissions(2)) {
                    player.sendSystemMessage((Component)BackupThread.component(player, message, parameters).withStyle(style));
                }
            }));
            if (Mc2DiscordCompat.isLoaded() && CommonConfig.mc2discord()) {
                Mc2DiscordCompat.announce((Component)BackupThread.component(null, message, parameters));
            }
        }
    }

    public static MutableComponent component(@Nullable ServerPlayer player, String key, Object ... parameters) {
        if (player != null && NetworkRegistry.hasChannel((Connection)player.connection.connection, null, (ResourceLocation)Pause.ID)) {
            return Component.translatable((String)key, (Object[])parameters);
        }
        return Component.literal((String)String.format(FMLTranslations.getPattern((String)key, () -> key), parameters));
    }

    private long makeWorldBackup() throws IOException {
        this.storageSource.checkLock();
        String fileName = this.storageSource.levelId + "_" + LocalDateTime.now().format(FORMATTER);
        Path path = CommonConfig.getOutputPath(this.storageSource.levelId);
        try {
            Files.createDirectories(Files.exists(path, new LinkOption[0]) ? path.toRealPath(new LinkOption[0]) : path, new FileAttribute[0]);
        }
        catch (IOException ioexception) {
            throw new RuntimeException(ioexception);
        }
        Path outputFile = path.resolve(FileUtil.findAvailableName((Path)path, (String)fileName, (String)".zip"));
        try (final ZipOutputStream zipStream = new ZipOutputStream(new BufferedOutputStream(Files.newOutputStream(outputFile, new OpenOption[0])));){
            zipStream.setLevel(CommonConfig.getCompressionLevel());
            final Path levelName = Paths.get(this.storageSource.levelId, new String[0]);
            final Path levelPath = this.storageSource.getWorldDir().resolve(this.storageSource.levelId).toRealPath(new LinkOption[0]);
            Files.walkFileTree(levelPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                @Nonnull
                public FileVisitResult visitFile(Path file, @Nonnull BasicFileAttributes attrs) throws IOException {
                    if (file.endsWith("session.lock")) {
                        return FileVisitResult.CONTINUE;
                    }
                    long lastModified = file.toFile().lastModified();
                    if (BackupThread.this.fullBackup || lastModified - BackupThread.this.lastSaved > 0L) {
                        String completePath = levelName.resolve(levelPath.relativize(file)).toString().replace('\\', '/');
                        ZipEntry zipentry = new ZipEntry(completePath);
                        zipStream.putNextEntry(zipentry);
                        com.google.common.io.Files.asByteSource((File)file.toFile()).copyTo((OutputStream)zipStream);
                        zipStream.closeEntry();
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        return Files.size(outputFile);
    }

    private static class Timer {
        private static final SimpleDateFormat SECONDS = new SimpleDateFormat("s.SSS");
        private static final SimpleDateFormat MINUTES = new SimpleDateFormat("mm:ss");
        private static final SimpleDateFormat HOURS = new SimpleDateFormat("HH:mm");

        private Timer() {
        }

        public static String getTimer(long milliseconds) {
            Date date = new Date(milliseconds);
            double seconds = (double)milliseconds / 1000.0;
            if (seconds < 60.0) {
                return SECONDS.format(date) + "s";
            }
            if (seconds < 3600.0) {
                return MINUTES.format(date) + "min";
            }
            return HOURS.format(date) + "h";
        }
    }
}

