#version 150

// Import Minecraft-specific GLSL libraries
#moj_import <minecraft:light.glsl>        // Lighting calculations and functions
#moj_import <minecraft:fog.glsl>          // Fog distance calculations and application
#moj_import <minecraft:dynamictransforms.glsl> // Dynamic transformation utilities
#moj_import <minecraft:projection.glsl>   // Projection matrix utilities

// ============================================================================
// INPUT VERTEX ATTRIBUTES
// ============================================================================
in vec3 Position;    // Vertex position in model space
in vec4 Color;       // Vertex color (tint, team color, etc.)
in vec2 UV0;         // Main texture coordinates
in ivec2 UV1;        // Overlay texture coordinates (integer coordinates for texelFetch)
in ivec2 UV2;        // Lightmap texture coordinates (integer coordinates for texelFetch)
in vec3 Normal;      // Vertex normal for lighting calculations

// ============================================================================
// UNIFORM SAMPLERS
// ============================================================================
uniform sampler2D Sampler1; // Overlay texture sampler (team colors, effects, etc.)
uniform sampler2D Sampler2; // Lightmap texture sampler (block light + sky light)

// ============================================================================
// OUTPUT VARIABLES TO FRAGMENT SHADER
// ============================================================================
out float sphericalVertexDistance;    // Distance calculated using spherical fog model
out float cylindricalVertexDistance;  // Distance calculated using cylindrical fog model
out vec4 vertexColor;                 // Vertex color passed through
out vec4 lightColor;                  // Computed directional lighting color
out vec4 lightMapColor;               // Ambient lighting from lightmap
out vec4 overlayColor;                // Overlay texture color
out vec2 texCoord0;                   // Transformed texture coordinates

// ============================================================================
// MAIN VERTEX SHADER FUNCTION
// ============================================================================
void main() {
    // Transform vertex position from model space to clip space
    // ModelViewMat and ProjMat are provided by Minecraft as uniform blocks
    gl_Position = ProjMat * ModelViewMat * vec4(Position, 1.0);

    // Calculate fog distances using both spherical and cylindrical models
    // This allows the fragment shader to choose the appropriate fog calculation
    sphericalVertexDistance = fog_spherical_distance(Position);
    cylindricalVertexDistance = fog_cylindrical_distance(Position);

    // Pass through the vertex color (can be used for tinting, team colors, etc.)
    vertexColor = Color;

    // Initialize light color to white (no lighting modification)
    lightColor = vec4(1.0);
    
    // Apply cardinal lighting (directional light) unless disabled
    // This calculates lighting based on normal direction and light directions
    #ifndef NO_CARDINAL_LIGHTING
    lightColor *= minecraft_mix_light(Light0_Direction, Light1_Direction, Normal, vec4(1));
    #endif
    
    // Apply ambient lighting from lightmap unless this is an emissive material
    // Emissive materials (like glowstone, end rods) don't use lightmap lighting
    #ifndef EMISSIVE
    lightColor *= texelFetch(Sampler2, UV2 / 16, 0);
    #endif

    // Sample overlay texture (used for team colors, potion effects, etc.)
    // UV1 coordinates are typically entity-specific overlay coordinates
    overlayColor = texelFetch(Sampler1, UV1, 0);

    // Handle texture coordinates
    texCoord0 = UV0; // Default to original UV coordinates
    
    // Apply texture matrix transformation if enabled
    // Used for animated textures, scrolling effects, etc.
    #ifdef APPLY_TEXTURE_MATRIX
    texCoord0 = (TextureMat * vec4(UV0, 0.0, 1.0)).xy;
    #endif
}