package net.mt1006.mocap.mocap.actions;

import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.serialization.DynamicOps;
import net.minecraft.class_1297;
import net.minecraft.class_1304;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2487;
import net.minecraft.class_2497;
import net.minecraft.class_2509;
import net.minecraft.class_2519;
import net.minecraft.class_2520;
import net.minecraft.class_7923;
import net.minecraft.nbt.*;
import net.mt1006.mocap.api.v1.extension.MocapRecordingData;
import net.mt1006.mocap.api.v1.extension.actions.MocapActionContext;
import net.mt1006.mocap.api.v1.extension.actions.MocapStateAction;
import net.mt1006.mocap.mixin.fields.LivingEntityFields;
import net.mt1006.mocap.utils.Utils;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

public class ChangeItem implements MocapStateAction
{
	private static final int ITEM_COUNT_LEGACY = 6;
	private static final int ITEM_COUNT = 8;
	private final byte itemCount;
	private final List<ItemData> items = new ArrayList<>();

	public static @Nullable ChangeItem fromEntity(class_1297 entity)
	{
		return (entity instanceof class_1309 livingEntity) ? new ChangeItem(livingEntity) : null;
	}

	private ChangeItem(class_1309 entity)
	{
		DynamicOps<class_2520> ops = entity.method_56673().method_57093(class_2509.field_11560);

		addItem(entity.method_6047(), ops);
		addItem(entity.method_6079(), ops);
		addItem(entity.method_6118(class_1304.field_6166), ops);
		addItem(entity.method_6118(class_1304.field_6172), ops);
		addItem(entity.method_6118(class_1304.field_6174), ops);
		addItem(entity.method_6118(class_1304.field_6169), ops);
		addItem(entity.method_6118(class_1304.field_48824), ops);
		addItem(entity.method_6118(class_1304.field_55946), ops);

		int itemCounter = 0;
		for (int i = 0; i < ITEM_COUNT; i++)
		{
			if (items.get(i).type != ItemDataType.NO_ITEM) { itemCounter = i + 1; }
		}
		itemCount = (byte)itemCounter;
	}

	public ChangeItem(Reader reader, MocapRecordingData data)
	{
		byte firstByte = reader.readByte();

		if (firstByte >= 0)
		{
			// backwards compatibility
			reader.shift(-1);
			itemCount = ITEM_COUNT_LEGACY;
		}
		else
		{
			itemCount = (byte)(firstByte != Byte.MIN_VALUE ? -firstByte : 0);
		}

		for (int i = 0; i < itemCount; i++) { items.add(new ItemData(reader, data)); }
	}

	private void addItem(@Nullable class_1799 itemStack, DynamicOps<class_2520> ops)
	{
		items.add(ItemData.get(itemStack, ops));
	}

	private void setEntityItems(class_1309 entity)
	{
		DynamicOps<class_2520> ops = entity.method_56673().method_57093(class_2509.field_11560);
		for (int i = 0; i < ITEM_COUNT; i++)
		{
			ItemData item = i < itemCount ? items.get(i) : ItemData.EMPTY;
			class_1799 itemStack = item.getItemStack(ops);

			switch (i)
			{
				case 0 -> entity.method_5673(class_1304.field_6173, itemStack);
				case 1 -> entity.method_5673(class_1304.field_6171, itemStack);
				case 2 -> entity.method_5673(class_1304.field_6166, itemStack);
				case 3 -> entity.method_5673(class_1304.field_6172, itemStack);
				case 4 -> entity.method_5673(class_1304.field_6174, itemStack);
				case 5 -> entity.method_5673(class_1304.field_6169, itemStack);
				case 6 -> entity.method_5673(class_1304.field_48824, itemStack);
				case 7 -> entity.method_5673(class_1304.field_55946, itemStack);
			}

			// for non-player living entities it's detected in their "tick" method
			if (entity instanceof class_1657) { ((LivingEntityFields)entity).callDetectEquipmentUpdates(); }
		}
	}

	@Override public boolean differs(MocapStateAction previousAction)
	{
		if (itemCount != ((ChangeItem)previousAction).itemCount) { return true; }

		for (int i = 0; i < itemCount; i++)
		{
			ItemData item1 = items.get(i);
			ItemData item2 = ((ChangeItem)previousAction).items.get(i);
			if (item1.differs(item2)) { return true; }
		}
		return false;
	}

	@Override public boolean shouldBeInitialized()
	{
		return itemCount != 0;
	}

	@Override public void prepareWrite(MocapRecordingData data)
	{
		if (itemCount > items.size()) { throw new RuntimeException(); }

		for (int i = 0; i < itemCount; i++)
		{
			items.get(i).prepareWrite(data);
		}
	}

	@Override public void write(Writer writer, MocapRecordingData data)
	{
		writer.addByte(itemCount > 0 ? (byte)(-itemCount) : Byte.MIN_VALUE);

		for (int i = 0; i < itemCount; i++)
		{
			items.get(i).write(writer);
		}
	}

	@Override public Result execute(MocapActionContext ctx)
	{
		class_1309 livingEntity = ctx.getLivingEntityOrDummyPlayer();
		if (livingEntity == null) { return Result.IGNORED; }

		setEntityItems(livingEntity);
		return Result.OK;
	}

	private enum ItemDataType
	{
		NO_ITEM(0, false, false),
		ID_ONLY(1, true, false),
		ID_AND_NBT(2, true, true),
		ID_AND_COMPONENTS(3, true, true);

		private static final ItemDataType[] VALUES = values();
		public final byte id;
		public final boolean hasId;
		public final boolean hasData;

		ItemDataType(int id, boolean hasId, boolean hasData)
		{
			this.id = (byte)id;
			this.hasId = hasId;
			this.hasData = hasData;
		}

		public static ItemDataType get(byte id)
		{
			if (id < 0 || id >= VALUES.length) { return NO_ITEM; }

			ItemDataType type = VALUES[id];
			if (type.id != id) { throw new RuntimeException("ChangeItem.ItemDataType VALUES out of order!"); }
			return type;
		}
	}

	private static class ItemData
	{
		public static final ItemData EMPTY = new ItemData();
		public final ItemDataType type;
		public final class_1792 item;
		public final String data;
		private int idToWrite = -1;

		private ItemData()
		{
			type = ItemDataType.NO_ITEM;
			item = class_1802.field_8162;
			data = "";
		}

		private ItemData(class_1799 itemStack, DynamicOps<class_2520> ops)
		{
			item = itemStack.method_7909();
			class_2520 tag;
			try { tag = class_1799.field_24671.encodeStart(ops, itemStack).getOrThrow(); }
			catch (Exception exception) { tag = null; }

			if (!(tag instanceof class_2487) || !((class_2487)tag).method_10545("components"))
			{
				type = ItemDataType.ID_ONLY;
				data = "";
				return;
			}

			class_2520 componentsTag = ((class_2487)tag).method_10580("components");
			if (!(componentsTag instanceof class_2487))
			{
				type = ItemDataType.ID_ONLY;
				data = "";
				return;
			}

			type = ItemDataType.ID_AND_COMPONENTS;
			data = componentsTag.toString();
		}

		public ItemData(Reader reader, MocapRecordingData recordingData)
		{
			type = ItemDataType.get(reader.readByte());
			int itemId = type.hasId ? reader.readInt() : 0;
			data = type.hasData ? reader.readString() : "";

			if (recordingData == null)
			{
				item = class_1802.field_8162;
				return;
			}

			item = recordingData.itemFromId(itemId);
		}

		public static ItemData get(@Nullable class_1799 itemStack, DynamicOps<class_2520> ops)
		{
			return (itemStack == null || itemStack.method_7960()) ? EMPTY : new ItemData(itemStack, ops);
		}

		public boolean differs(ItemData itemData)
		{
			return type != itemData.type || item != itemData.item || !data.equals(itemData.data);
		}

		public void prepareWrite(MocapRecordingData recordingData)
		{
			idToWrite = recordingData.provideItemId(item);
		}

		public void write(Writer writer)
		{
			if (idToWrite == -1) { throw new RuntimeException("ItemData write wasn't prepared!"); }

			writer.addByte(type.id);
			if (type.hasId) { writer.addInt(idToWrite); }
			if (type.hasData) { writer.addString(data); }
		}

		public class_1799 getItemStack(DynamicOps<class_2520> ops)
		{
			switch (type)
			{
				case NO_ITEM:
					return class_1799.field_8037;

				case ID_ONLY:
				case ID_AND_NBT:
					return new class_1799(item);

				case ID_AND_COMPONENTS:
					class_2487 tag = tagFromIdAndComponents();
					if (tag == null) { return class_1799.field_8037; }
					try { return class_1799.field_24671.parse(ops, tag).getOrThrow(); }
					catch (Exception e) { return class_1799.field_8037; }
			}
			return null;
		}

		private @Nullable class_2487 tagFromIdAndComponents()
		{
			class_2487 tag = new class_2487();

			try { tag.method_10566("components", Utils.nbtFromString(data)); }
			catch (CommandSyntaxException e) { return null; }

			tag.method_10566("id", class_2519.method_23256(class_7923.field_41178.method_10221(item).toString()));
			tag.method_10566("count", class_2497.method_23247(1));
			return tag;
		}
	}
}
