package net.mt1006.mocap.mocap.files;

import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_5250;
import net.minecraft.class_5321;
import net.minecraft.class_7924;
import net.mt1006.mocap.MocapMod;
import net.mt1006.mocap.api.v1.controller.config.MocapDimensionSource;
import net.mt1006.mocap.api.v1.controller.playable.MocapRecordingFile;
import net.mt1006.mocap.api.v1.extension.actions.MocapAction;
import net.mt1006.mocap.api.v1.io.CommandInfo;
import net.mt1006.mocap.api.v1.io.CommandOutput;
import net.mt1006.mocap.command.CommandSuggestions;
import net.mt1006.mocap.mocap.playing.playable.RecordingFile;
import net.mt1006.mocap.mocap.settings.Settings;
import net.mt1006.mocap.utils.Utils;
import org.jetbrains.annotations.Nullable;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.*;

public class RecordingFiles
{
	public static final byte VERSION = MocapMod.RECORDING_FORMAT_VERSION;
	private static final int ALT_NAME_MAX_I = 128;
	public static final MocapAction.Reader DUMMY_READER = new DummyReader();

	public static boolean save(CommandOutput out, File recordingFile, String name, RecordingData data)
	{
		try
		{
			// double-check to make sure it won't override existing file
			if (recordingFile.exists())
			{
				out.sendFailure("recording.save.already_exists");
				return false;
			}

			BufferedOutputStream stream = new BufferedOutputStream(new FileOutputStream(recordingFile));
			data.save(stream);
			stream.close();
		}
		catch (IOException e)
		{
			out.sendException(e, "recording.save.error");
			return false;
		}

		CommandSuggestions.inputSet.add(name);
		return true;
	}

	public static boolean info(CommandInfo out, @Nullable RecordingFile file) //TODO: rename out
	{
		MocapRecordingFile.Info info = RecordingFile.Info.load(out, file);
		if (file == null || info == null) { return false; }

		out.sendSuccess("recordings.info.info");
		out.sendSuccess("file.info.name", file.getName());
		if (!Files.printVersionInfo(out, VERSION, info.version(), info.experimental(), info.experimentalSubversion())) { return true; }

		out.sendSuccess("recordings.info.length", String.format("%.2f", info.lengthInTicks() / 20.0), info.lengthInTicks());
		out.sendSuccess("recordings.info.size", String.format("%.2f", info.sizeInBytes() / 1024.0), info.sizeInOps());

		printPosInfo(out, info);

		if (info.assignedDimensionId() != null) { out.sendSuccess("recordings.info.dimension", info.assignedDimensionId().toString()); }
		else { out.sendSuccess("recordings.info.dimension.not_assigned"); }

		if (info.assignedPlayerName() != null) { out.sendSuccess("recordings.info.player_name_assigned.yes", info.assignedPlayerName()); }
		else { out.sendSuccess("recordings.info.player_name_assigned.no"); }

		out.sendSuccess(info.legacyEndsWithDeath() ? "recordings.info.dies.yes" : "recordings.info.dies.no");
		return true;
	}

	private static void printPosInfo(CommandInfo out, MocapRecordingFile.Info info) //TODO: rename out
	{
		class_2960 dimensionId = info.assignedDimensionId();
		boolean anotherDimension = (dimensionId != null && out.getLevel().method_27983() != class_5321.method_29179(class_7924.field_41223, dimensionId)
				&& (Settings.DIMENSION_SOURCE.val == MocapDimensionSource.ASSIGNED_OR_CURRENT
				|| Settings.DIMENSION_SOURCE.val == MocapDimensionSource.ASSIGNED_OR_OVERWORLD));

		String xStr = String.format(Locale.US, "%.2f", info.startPos().field_1352);
		String yStr = String.format(Locale.US, "%.2f", info.startPos().field_1351);
		String zStr = String.format(Locale.US, "%.2f", info.startPos().field_1350);
		String command = anotherDimension
				? String.format("/execute in %s run tp @p %s %s %s", dimensionId, xStr, yStr, zStr)
				: String.format("/tp @p %s %s %s", xStr, yStr, zStr);

		String text = String.format("%s %s %s", xStr, yStr, zStr);
		if (anotherDimension)
		{
			String dimensionIdStr = dimensionId.method_12836().equals("minecraft") ? dimensionId.method_12832() : dimensionId.toString();
			text += String.format(" (%s)", dimensionIdStr);
		}

		class_5250 tpSuggestionComponent = Utils.getSuggestCommandComponent(command,
				class_2561.method_43470(text)).method_27696(class_2583.field_24360.method_30938(true));
		out.sendSuccess("recordings.info.start_pos", tpSuggestionComponent);
	}

	public static @Nullable List<String> list()
	{
		if (!Files.initialized) { return null; }

		String[] fileList = Files.recordingsDirectory.list(Files::isRecordingFile);
		if (fileList == null) { return null; }

		List<String> recordings = new ArrayList<>(fileList.length);
		for (String filename : fileList)
		{
			recordings.add(filename.substring(0, filename.lastIndexOf('.')));
		}

		Collections.sort(recordings);
		return recordings;
	}

	public static @Nullable String findAlternativeName(String name)
	{
		if (name.isEmpty()) { return null; }

		int firstDigit = name.length();
		int lastDigit = name.length() - 1;
		for (int i = lastDigit; i >= 0; i--)
		{
			char ch = name.charAt(i);
			if (ch >= '0' && ch <= '9') { firstDigit = i; }
			else { break; }
		}

		if (firstDigit > lastDigit) { return null; }
		String prefix = name.substring(0, firstDigit);
		int suffix = Integer.parseInt(name.substring(firstDigit, lastDigit + 1));

		for (int i = suffix + 1; i <= suffix + ALT_NAME_MAX_I ; i++)
		{
			String possibleName = String.format("%s%d", prefix, i);
			if (!CommandSuggestions.inputSet.contains(possibleName)) { return possibleName; }
		}
		return null;
	}

	public static class Writer implements MocapAction.Writer
	{
		private final ArrayList<Byte> recording = new ArrayList<>();

		@Override public void addByte(byte val)
		{
			recording.add(val);
		}

		@Override public void addShort(short val)
		{
			recording.add((byte)(val >> 8));
			recording.add((byte)val);
		}

		@Override public void addInt(int val)
		{
			recording.add((byte)(val >> 24));
			recording.add((byte)(val >> 16));
			recording.add((byte)(val >> 8));
			recording.add((byte)val);
		}

		@Override public void addLong(long val)
		{
			recording.add((byte)(val >> 56));
			recording.add((byte)(val >> 48));
			recording.add((byte)(val >> 40));
			recording.add((byte)(val >> 32));
			recording.add((byte)(val >> 24));
			recording.add((byte)(val >> 16));
			recording.add((byte)(val >> 8));
			recording.add((byte)val);
		}

		@Override public void addFloat(float val)
		{
			for (byte b : floatToByteArray(val))
			{
				recording.add(b);
			}
		}

		@Override public void addDouble(double val)
		{
			for (byte b : doubleToByteArray(val))
			{
				recording.add(b);
			}
		}

		@Override public void addBoolean(boolean val)
		{
			recording.add(val ? (byte)1 : (byte)0);
		}

		@Override public void addString(String val)
		{
			byte[] bytes = val.getBytes(StandardCharsets.UTF_8);
			addPackedInt(bytes.length);
			for (byte b : bytes)
			{
				recording.add(b);
			}
		}

		@Override public void addUUID(UUID val)
		{
			addLong(val.getMostSignificantBits());
			addLong(val.getLeastSignificantBits());
		}

		@Override public void addVec3(class_243 vec)
		{
			addDouble(vec.field_1352);
			addDouble(vec.field_1351);
			addDouble(vec.field_1350);
		}

		@Override public void addBlockPos(class_2338 blockPos)
		{
			addInt(blockPos.method_10263());
			addInt(blockPos.method_10264());
			addInt(blockPos.method_10260());
		}

		@Override public void addPackedInt(int size)
		{
			if (size >= 0 && size < 255)
			{
				addByte((byte)size);
			}
			else
			{
				addByte((byte)255);
				addInt(size);
			}
		}

		public void copyToWriter(MocapAction.Writer writer)
		{
			recording.forEach(writer::addByte);
		}

		public int getSize()
		{
			return recording.size();
		}

		public byte[] toByteArray()
		{
			byte[] array = new byte[recording.size()];
			for (int i = 0; i < recording.size(); i++) { array[i] = recording.get(i); }
			return array;
		}

		private static byte[] floatToByteArray(float val)
		{
			int bits = Float.floatToIntBits(val);
			return new byte[] { (byte)(bits >> 24), (byte)(bits >> 16), (byte)(bits >> 8), (byte)bits };
		}

		private static byte[] doubleToByteArray(double val)
		{
			long bits = Double.doubleToLongBits(val);
			return new byte[] { (byte)(bits >> 56), (byte)(bits >> 48), (byte)(bits >> 40), (byte)(bits >> 32),
					(byte)(bits >> 24), (byte)(bits >> 16), (byte)(bits >> 8), (byte)bits };
		}
	}

	public static class FileReader implements MocapAction.Reader
	{
		private final byte[] recording;
		private boolean legacyString;
		public int offset = 0;
		public boolean convertStrings = false; //TODO: [CONVERTER] remove

		public FileReader(byte[] recording, boolean legacyString)
		{
			this.recording = recording;
			this.legacyString = legacyString;
		}

		@Override public byte readByte()
		{
			return recording[offset++];
		}

		@Override public short readShort()
		{
			short retVal = (short)(((recording[offset] & 0xFF) << 8) | (recording[offset + 1] & 0xFF));
			offset += 2;
			return retVal;
		}

		@Override public int readInt()
		{
			int retVal = ((recording[offset] & 0xFF) << 24) | ((recording[offset + 1] & 0xFF) << 16) |
					((recording[offset + 2] & 0xFF) << 8) | (recording[offset + 3] & 0xFF);
			offset += 4;
			return retVal;
		}

		@Override public long readLong()
		{
			long retVal = ((recording[offset] & 0xFFL) << 56) | ((recording[offset + 1] & 0xFFL) << 48) |
					((recording[offset + 2] & 0xFFL) << 40) | ((recording[offset + 3] & 0xFFL) << 32) |
					((recording[offset + 4] & 0xFFL) << 24) | ((recording[offset + 5] & 0xFFL) << 16) |
					((recording[offset + 6] & 0xFFL) << 8) | (recording[offset + 7] & 0xFFL);
			offset += 8;
			return retVal;
		}

		@Override public float readFloat()
		{
			float retVal = byteArrayToFloat(Arrays.copyOfRange(recording, offset, offset + 4));
			offset += 4;
			return retVal;
		}

		@Override public double readDouble()
		{
			double retVal = byteArrayToDouble(Arrays.copyOfRange(recording, offset, offset + 8));
			offset += 8;
			return retVal;
		}

		@Override public boolean readBoolean()
		{
			return recording[offset++] == 1;
		}

		@Override public String readString()
		{
			if (convertStrings && !legacyString) { return readAlphaString(); } //TODO: [CONVERTER] remove

			int len = legacyString ? readInt() : readPackedInt();
			String str = new String(recording, offset, len, StandardCharsets.UTF_8);
			offset += len;
			return str;
		}

		@Override public UUID readUUID()
		{
			return new UUID(readLong(), readLong());
		}

		//TODO: [CONVERTER] remove
		private String readAlphaString()
		{
			int termPos = -1;
			for (int i = offset; i < recording.length; i++)
			{
				if (recording[i] == 0)
				{
					termPos = i;
					break;
				}
			}

			int len = termPos - offset;
			String str = new String(recording, offset, len, StandardCharsets.UTF_8);
			offset += len + 1;
			return str;
		}

		@Override public class_243 readVec3()
		{
			return new class_243(readDouble(), readDouble(), readDouble());
		}

		@Override public class_2338 readBlockPos()
		{
			return new class_2338(readInt(), readInt(), readInt());
		}

		@Override public int readPackedInt()
		{
			int val = Byte.toUnsignedInt(readByte());
			return (val == 255) ? readInt() : val;
		}

		@Override public void shift(int val)
		{
			offset += val;
		}

		@Override public boolean isDummy()
		{
			return false;
		}

		public void setStringMode(boolean legacyString)
		{
			this.legacyString = legacyString;
		}

		public boolean canRead()
		{
			return recording.length > offset;
		}

		public int getSize()
		{
			return recording.length;
		}

		private static float byteArrayToFloat(byte[] bytes)
		{
			int bits = (((int)bytes[0] & 0xFF) << 24) | (((int)bytes[1] & 0xFF) << 16) | (((int)bytes[2] & 0xFF) << 8) | ((int)bytes[3] & 0xFF);
			return Float.intBitsToFloat(bits);
		}

		private static double byteArrayToDouble(byte[] bytes)
		{
			long bits = (((long)bytes[0] & 0xFF) << 56) | (((long)bytes[1] & 0xFF) << 48) | (((long)bytes[2] & 0xFF) << 40) | (((long)bytes[3] & 0xFF) << 32) |
					(((long)bytes[4] & 0xFF) << 24) | (((long)bytes[5] & 0xFF) << 16) | (((long)bytes[6] & 0xFF) << 8) | ((long)bytes[7] & 0xFF);
			return Double.longBitsToDouble(bits);
		}
	}

	private static class DummyReader implements MocapAction.Reader
	{
		private static final UUID UUID_ZERO = new UUID(0, 0);

		@Override public byte readByte() { return 0; }
		@Override public short readShort() { return 0; }
		@Override public int readInt() { return 0; }
		@Override public long readLong() { return 0; }
		@Override public float readFloat() { return 0.0f; }
		@Override public double readDouble() { return 0.0; }
		@Override public boolean readBoolean() { return false; }
		@Override public String readString() { return ""; }
		@Override public UUID readUUID() { return UUID_ZERO; }
		@Override public class_243 readVec3() { return class_243.field_1353; }
		@Override public class_2338 readBlockPos() { return class_2338.field_10980; }
		@Override public int readPackedInt() { return 0; }
		@Override public void shift(int val) {}
		@Override public boolean isDummy() { return true; }
	}
}
