package com.criticalrange.client;

import com.criticalrange.VulkanModExtra;
import com.criticalrange.client.config.VulkanModExtraClientConfig;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import net.minecraft.class_2561;

/**
 * Runtime integration with VulkanMod's GUI system
 * This class uses reflection to add VulkanMod Extra options to VulkanMod's settings screen
 */
public class VulkanModExtraIntegration {

    private static boolean integrationAttempted = false;
    private static boolean integrationSuccessful = false;

    // Track which screen instances have already been injected to prevent multiple injections
    private static final Map<Object, Boolean> injectedInstances = new WeakHashMap<>();

    /**
     * Attempt to integrate with VulkanMod's GUI system at runtime
     * This should be called when VulkanMod is loaded
     */
    public static void tryIntegrateWithVulkanMod() {
        if (integrationAttempted) {
            return; // Only attempt once
        }

        integrationAttempted = true;

        try {
            VulkanModExtra.LOGGER.info("Attempting to integrate VulkanMod Extra with VulkanMod GUI...");

            // Check if VulkanMod classes are available
            Class<?> vOptionScreenClass = findClass("net.vulkanmod.config.gui.VOptionScreen");
            if (vOptionScreenClass == null) {
                VulkanModExtra.LOGGER.warn("VulkanMod not found, skipping GUI integration");
                return;
            }

            // Try to set up the mixin integration
            // Note: The actual mixin will handle the GUI integration
            VulkanModExtra.LOGGER.info("VulkanMod found, GUI mixin will handle integration");

            integrationSuccessful = true;
            VulkanModExtra.LOGGER.info("Successfully integrated VulkanMod Extra with VulkanMod GUI!");

        } catch (Exception e) {
            VulkanModExtra.LOGGER.error("Failed to integrate with VulkanMod GUI", e);
            // Don't rethrow - let the mod continue without GUI integration
        }
    }



    /**
     * Safe method to inject pages into the currently active VulkanMod screen
     * Called by the mixin after screen initialization is complete
     */
    public static void injectPagesIntoCurrentScreen() {
        // Check if GUI integration is disabled
        if (!VulkanModExtra.CONFIG.extraSettings.enableVulkanModGuiIntegration) {
            return;
        }
        
        if (integrationAttempted && !integrationSuccessful) {
            return; // Don't retry if we already failed
        }

        try {
            // Get the current screen through Minecraft client
            Class<?> minecraftClass = findClass("net.minecraft.client.MinecraftClient");
            if (minecraftClass == null) {
                VulkanModExtra.LOGGER.warn("Could not find MinecraftClient class");
                return;
            }
            VulkanModExtra.LOGGER.debug("Found MinecraftClient class: {}", minecraftClass.getName());

            // Get the singleton Minecraft instance
            Object minecraft = getStaticFieldValue(minecraftClass, "INSTANCE");
            if (minecraft == null) {
                VulkanModExtra.LOGGER.warn("Could not get Minecraft instance");
                return;
            }
            VulkanModExtra.LOGGER.debug("Got Minecraft instance");

            Object currentScreen = getFieldValue(minecraft, "currentScreen");
            if (currentScreen == null) {
                VulkanModExtra.LOGGER.debug("No current screen found");
                return;
            }
            VulkanModExtra.LOGGER.debug("Found current screen: {}", currentScreen.getClass().getName());

            // Check if it's a VulkanMod options screen
            if (!isVulkanModOptionScreen(currentScreen)) {
                VulkanModExtra.LOGGER.debug("Current screen is not a VulkanMod options screen");
                return;
            }

            VulkanModExtra.LOGGER.info("Found VulkanMod options screen, creating extra pages...");

            // Try to add our pages to this screen
            List<Object> extraPages = createVulkanModExtraPages();
            if (extraPages != null && !extraPages.isEmpty()) {
                addPagesToScreen(currentScreen, extraPages);
                integrationSuccessful = true;
                VulkanModExtra.LOGGER.info("Successfully injected VulkanMod Extra pages into active screen!");
            }

        } catch (Exception e) {
            VulkanModExtra.LOGGER.error("Could not inject pages into current screen: {}", e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Check if the given screen is a VulkanMod options screen
     */
    private static boolean isVulkanModOptionScreen(Object screen) {
        if (screen == null) return false;

        String className = screen.getClass().getName();
        VulkanModExtra.LOGGER.debug("Checking screen class: {}", className);

        // More flexible detection - just check for vulkanmod in the class name
        boolean isVulkanModScreen = className.toLowerCase().contains("vulkanmod");
        boolean isOptionScreen = className.toLowerCase().contains("option") || className.toLowerCase().contains("screen");

        VulkanModExtra.LOGGER.debug("Is VulkanMod screen: {}, Is option screen: {}", isVulkanModScreen, isOptionScreen);

        return isVulkanModScreen && isOptionScreen;
    }

    /**
     * Try to add extra pages to an active screen
     */
    private static void addPagesToScreen(Object screen, List<Object> extraPages) {
        try {
            // Look for common field names that might hold option pages
            String[] possibleFieldNames = {"optionPages", "pages", "tabs", "categories"};

            for (String fieldName : possibleFieldNames) {
                try {
                    java.lang.reflect.Field field = screen.getClass().getDeclaredField(fieldName);
                    field.setAccessible(true);
                    Object fieldValue = field.get(screen);

                    if (fieldValue instanceof List) {
                        @SuppressWarnings("unchecked")
                        List<Object> pages = (List<Object>) fieldValue;
                        pages.addAll(extraPages);
                        VulkanModExtra.LOGGER.info("Added {} extra pages to VulkanMod screen via field {}", extraPages.size(), fieldName);
                        return;
                    }
                } catch (NoSuchFieldException e) {
                    // Field not found, continue trying other fields
                }
            }

            // If we couldn't find a pages field, try to call addPage methods
            tryAddPagesViaMethods(screen, extraPages);

        } catch (Exception e) {
            VulkanModExtra.LOGGER.error("Could not add pages to screen: {}", e.getMessage());
        }
    }

    /**
     * Try to add pages using method calls
     */
    private static void tryAddPagesViaMethods(Object screen, List<Object> extraPages) {
        try {
            // Look for methods like addPage, addTab, etc.
            String[] possibleMethods = {"addPage", "addTab", "addCategory", "addOptionPage"};

            for (String methodName : possibleMethods) {
                try {
                    java.lang.reflect.Method method = screen.getClass().getDeclaredMethod(methodName, Object.class);
                    method.setAccessible(true);

                    for (Object page : extraPages) {
                        method.invoke(screen, page);
                    }
                    VulkanModExtra.LOGGER.info("Added {} pages via method {} to VulkanMod screen", extraPages.size(), methodName);
                    return;
                } catch (NoSuchMethodException e) {
                    // Method not found, continue trying other methods
                }
            }
        } catch (Exception e) {
            VulkanModExtra.LOGGER.error("Could not add pages via methods: {}", e.getMessage());
        }
    }

    // Cache VulkanMod classes to avoid repeated reflection lookups
    private static Class<?> cachedOptionPageClass;
    private static Class<?> cachedOptionBlockClass;
    private static Class<?> cachedSwitchOptionClass;
    private static Class<?> cachedCyclingOptionClass;
    private static Class<?> cachedOptionClass;
    private static boolean classesLoaded = false;
    
    private static boolean loadVulkanModClasses() {
        if (classesLoaded) return true;
        
        try {
            cachedOptionPageClass = Class.forName("net.vulkanmod.config.option.OptionPage");
            cachedOptionBlockClass = Class.forName("net.vulkanmod.config.gui.OptionBlock");
            cachedSwitchOptionClass = Class.forName("net.vulkanmod.config.option.SwitchOption");
            cachedCyclingOptionClass = Class.forName("net.vulkanmod.config.option.CyclingOption");
            cachedOptionClass = Class.forName("net.vulkanmod.config.option.Option");
            classesLoaded = true;
            return true;
        } catch (ClassNotFoundException e) {
            VulkanModExtra.LOGGER.error("Failed to load VulkanMod classes for GUI integration", e);
            return false;
        }
    }

    /**
     * Create VulkanMod-compatible option pages using cached reflection classes
     */
    public static List<Object> createVulkanModExtraPages() {
        if (!loadVulkanModClasses()) {
            VulkanModExtra.LOGGER.warn("Cannot create VulkanMod Extra pages - VulkanMod classes not available");
            return new ArrayList<>();
        }
        
        VulkanModExtra.LOGGER.debug("Creating VulkanMod Extra pages...");
        List<Object> pages = new ArrayList<>();

        try {

            VulkanModExtra.LOGGER.debug("VulkanMod classes loaded successfully");

            // Pre-calculate array classes for performance
            Class<?> optionArrayClass = java.lang.reflect.Array.newInstance(cachedOptionClass, 0).getClass();
            Class<?> optionBlockArrayClass = java.lang.reflect.Array.newInstance(cachedOptionBlockClass, 0).getClass();

            // Create Animation page with comprehensive options
            VulkanModExtra.LOGGER.debug("Creating animation page...");
            List<Object> animationOptions = createComprehensiveAnimationOptions(cachedSwitchOptionClass);
            Object[] animationArray = animationOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(cachedOptionClass, animationOptions.size()));
            Object animationBlock = cachedOptionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("Animation Settings", animationArray);

            // Create OptionBlock array for OptionPage constructor
            Object[] animationBlocks = (Object[]) java.lang.reflect.Array.newInstance(cachedOptionBlockClass, 1);
            animationBlocks[0] = animationBlock;

            Object animationPage = cachedOptionPageClass.getConstructor(String.class, optionBlockArrayClass).newInstance("Animations", animationBlocks);
            pages.add(animationPage);

            // Create Particle page with comprehensive options
            VulkanModExtra.LOGGER.debug("Creating particle page...");
            List<Object> particleOptions = createComprehensiveParticleOptions(cachedSwitchOptionClass);
            Object[] particleArray = particleOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(cachedOptionClass, particleOptions.size()));
            Object particleBlock = cachedOptionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("Particle Settings", particleArray);
            Object[] particleBlocks = (Object[]) java.lang.reflect.Array.newInstance(cachedOptionBlockClass, 1);
            particleBlocks[0] = particleBlock;
            Object particlePage = cachedOptionPageClass.getConstructor(String.class, optionBlockArrayClass).newInstance("Particles", particleBlocks);
            pages.add(particlePage);

            // Create Details page with comprehensive options
            VulkanModExtra.LOGGER.debug("Creating details page...");
            List<Object> detailOptions = createComprehensiveDetailOptions(cachedSwitchOptionClass);
            Object[] detailArray = detailOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(cachedOptionClass, detailOptions.size()));
            Object detailBlock = cachedOptionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("Detail Settings", detailArray);
            Object[] detailBlocks = (Object[]) java.lang.reflect.Array.newInstance(cachedOptionBlockClass, 1);
            detailBlocks[0] = detailBlock;
            Object detailPage = cachedOptionPageClass.getConstructor(String.class, optionBlockArrayClass).newInstance("Details", detailBlocks);
            pages.add(detailPage);

            // Create Render page with comprehensive options including fog
            VulkanModExtra.LOGGER.debug("Creating render page...");
            List<Object> renderOptions = createComprehensiveRenderOptions(cachedSwitchOptionClass);
            Object[] renderArray = renderOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(cachedOptionClass, renderOptions.size()));
            Object renderBlock = cachedOptionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("Render Settings", renderArray);
            Object[] renderBlocks = (Object[]) java.lang.reflect.Array.newInstance(cachedOptionBlockClass, 1);
            renderBlocks[0] = renderBlock;
            Object renderPage = cachedOptionPageClass.getConstructor(String.class, optionBlockArrayClass).newInstance("Render", renderBlocks);
            pages.add(renderPage);

            // Create HUD page with comprehensive options
            VulkanModExtra.LOGGER.debug("Creating HUD page...");
            List<Object> hudOptions = createComprehensiveHUDOptions(cachedSwitchOptionClass, cachedCyclingOptionClass);
            Object[] hudArray = hudOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(cachedOptionClass, hudOptions.size()));
            Object hudBlock = cachedOptionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("HUD Settings", hudArray);
            Object[] hudBlocks = (Object[]) java.lang.reflect.Array.newInstance(cachedOptionBlockClass, 1);
            hudBlocks[0] = hudBlock;
            Object hudPage = cachedOptionPageClass.getConstructor(String.class, optionBlockArrayClass).newInstance("HUD", hudBlocks);
            pages.add(hudPage);

            VulkanModExtra.LOGGER.info("Successfully created {} VulkanMod Extra pages", pages.size());

        } catch (Exception e) {
            VulkanModExtra.LOGGER.error("Failed to create VulkanMod Extra pages", e);
        }

        return pages;
    }

    /**
     * Create VulkanMod Extra pages with proper typing for direct mixin injection
     */
    public static List<net.vulkanmod.config.option.OptionPage> createVulkanModExtraOptionPages() {

        List<net.vulkanmod.config.option.OptionPage> pages = new ArrayList<>();

        try {
            // Load VulkanMod classes
            Class<?> optionBlockClass = Class.forName("net.vulkanmod.config.gui.OptionBlock");
            Class<?> switchOptionClass = Class.forName("net.vulkanmod.config.option.SwitchOption");
            Class<?> cyclingOptionClass = Class.forName("net.vulkanmod.config.option.CyclingOption");
            Class<?> optionClass = Class.forName("net.vulkanmod.config.option.Option");
            Class<?> optionPageClass = Class.forName("net.vulkanmod.config.option.OptionPage");

            Class<?> optionArrayClass = java.lang.reflect.Array.newInstance(optionClass, 0).getClass();
            Class<?> optionBlockArrayClass = java.lang.reflect.Array.newInstance(optionBlockClass, 0).getClass();

            // Create Animation page with comprehensive options
            List<Object> animationOptions = createComprehensiveAnimationOptions(switchOptionClass);
            Object[] animationArray = animationOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(optionClass, animationOptions.size()));
            Object animationBlock = optionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("Animation Settings", animationArray);
            Object[] animationBlocks = (Object[]) java.lang.reflect.Array.newInstance(optionBlockClass, 1);
            animationBlocks[0] = animationBlock;
            net.vulkanmod.config.option.OptionPage animationPage = (net.vulkanmod.config.option.OptionPage) optionPageClass
                    .getConstructor(String.class, optionBlockArrayClass).newInstance("Animations", animationBlocks);
            pages.add(animationPage);

            // Create Particle page with comprehensive options
            List<Object> particleOptions = createComprehensiveParticleOptions(switchOptionClass);
            Object[] particleArray = particleOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(optionClass, particleOptions.size()));
            Object particleBlock = optionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("Particle Settings", particleArray);
            Object[] particleBlocks = (Object[]) java.lang.reflect.Array.newInstance(optionBlockClass, 1);
            particleBlocks[0] = particleBlock;
            net.vulkanmod.config.option.OptionPage particlePage = (net.vulkanmod.config.option.OptionPage) optionPageClass
                    .getConstructor(String.class, optionBlockArrayClass).newInstance("Particles", particleBlocks);
            pages.add(particlePage);

            // Create Details page with comprehensive options
            List<Object> detailOptions = createComprehensiveDetailOptions(switchOptionClass);
            Object[] detailArray = detailOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(optionClass, detailOptions.size()));
            Object detailBlock = optionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("Detail Settings", detailArray);
            Object[] detailBlocks = (Object[]) java.lang.reflect.Array.newInstance(optionBlockClass, 1);
            detailBlocks[0] = detailBlock;
            net.vulkanmod.config.option.OptionPage detailPage = (net.vulkanmod.config.option.OptionPage) optionPageClass
                    .getConstructor(String.class, optionBlockArrayClass).newInstance("Details", detailBlocks);
            pages.add(detailPage);

            // Create Render page with comprehensive options including fog
            List<Object> renderOptions = createComprehensiveRenderOptions(switchOptionClass);
            Object[] renderArray = renderOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(optionClass, renderOptions.size()));
            Object renderBlock = optionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("Render Settings", renderArray);
            Object[] renderBlocks = (Object[]) java.lang.reflect.Array.newInstance(optionBlockClass, 1);
            renderBlocks[0] = renderBlock;
            net.vulkanmod.config.option.OptionPage renderPage = (net.vulkanmod.config.option.OptionPage) optionPageClass
                    .getConstructor(String.class, optionBlockArrayClass).newInstance("Render", renderBlocks);
            pages.add(renderPage);

            // Create HUD page with comprehensive options
            List<Object> hudOptions = createComprehensiveHUDOptions(switchOptionClass, cyclingOptionClass);
            Object[] hudArray = hudOptions.toArray((Object[]) java.lang.reflect.Array.newInstance(optionClass, hudOptions.size()));
            Object hudBlock = optionBlockClass.getConstructor(String.class, optionArrayClass).newInstance("HUD Settings", hudArray);
            Object[] hudBlocks = (Object[]) java.lang.reflect.Array.newInstance(optionBlockClass, 1);
            hudBlocks[0] = hudBlock;
            net.vulkanmod.config.option.OptionPage hudPage = (net.vulkanmod.config.option.OptionPage) optionPageClass
                    .getConstructor(String.class, optionBlockArrayClass).newInstance("HUD", hudBlocks);
            pages.add(hudPage);

        } catch (Exception e) {
            VulkanModExtra.LOGGER.error("Failed to create VulkanMod Extra pages", e);
        }

        return pages;
    }

    private static List<Object> createAnimationOptions(Class<?> switchOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // Helper for consistent option creation
        java.util.function.BiFunction<String, java.util.function.Function<Boolean, Void>, Object> createOption = (key, setter) -> {
            try {
                return switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(class_2561.method_43471("vulkanmod-extra.option." + key),
                        (java.util.function.Consumer<Boolean>) setter::apply,
                        (java.util.function.Supplier<Boolean>) () -> {
                            try {
                                var field = VulkanModExtra.CONFIG.animationSettings.getClass().getDeclaredField(key);
                                field.setAccessible(true);
                                return field.getBoolean(VulkanModExtra.CONFIG.animationSettings);
                            } catch (Exception e) { return true; }
                        });
            } catch (Exception e) { return null; }
        };

        // Animation master toggle
        options.add(createOption.apply("animation", value -> {
            VulkanModExtra.CONFIG.animationSettings.animation = value;
            VulkanModExtra.CONFIG.writeChanges();
            return null;
        }));

        // Individual animation options
        String[] animationTypes = {"water", "lava", "fire", "portal", "blockAnimations", "sculkSensor"};
        for (String type : animationTypes) {
            Object option = createOption.apply(type, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.animationSettings.getClass().getDeclaredField(type);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.animationSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.error("Failed to set animation option: " + type, e);
                }
                return null;
            });
            if (option != null) options.add(option);
        }

        return options;
    }

    private static List<Object> createParticleOptions(Class<?> switchOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // Helper for creating particle options
        java.util.function.BiFunction<String, java.util.function.Function<Boolean, Void>, Object> createOption = (key, setter) -> {
            try {
                return switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(class_2561.method_43471("vulkanmod-extra.option." + key),
                        (java.util.function.Consumer<Boolean>) setter::apply,
                        (java.util.function.Supplier<Boolean>) () -> {
                            try {
                                var field = VulkanModExtra.CONFIG.particleSettings.getClass().getDeclaredField(key);
                                field.setAccessible(true);
                                return field.getBoolean(VulkanModExtra.CONFIG.particleSettings);
                            } catch (Exception e) { return true; }
                        });
            } catch (Exception e) { return null; }
        };

        // Particles master toggle
        options.add(createOption.apply("particles", value -> {
            VulkanModExtra.CONFIG.particleSettings.particles = value;
            VulkanModExtra.CONFIG.writeChanges();
            return null;
        }));

        // Core particle types (most commonly used)
        String[] coreParticles = {
            "rainSplash", "blockBreak", "blockBreaking", "flame", "smoke", 
            "bubble", "splash", "rain", "drippingWater", "explosion", "heart", 
            "crit", "enchant", "note", "portal", "lava", "firework", "happyVillager", 
            "angryVillager", "ash", "campfireCosySmoke", "effect", "dust", "poof"
        };

        for (String particle : coreParticles) {
            Object option = createOption.apply(particle, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.particleSettings.getClass().getDeclaredField(particle);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.particleSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.error("Failed to set particle option: " + particle, e);
                }
                return null;
            });
            if (option != null) options.add(option);
        }

        // Environmental particles
        String[] environmentalParticles = {
            "cherryLeaves", "crimsonSpore", "warpedSpore", "whiteAsh", "sporeBlossomAir", 
            "mycelium", "cloud", "composter", "drippingHoney", "fallingHoney", "landingHoney"
        };

        for (String particle : environmentalParticles) {
            Object option = createOption.apply(particle, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.particleSettings.getClass().getDeclaredField(particle);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.particleSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.debug("Particle field not found: " + particle);
                }
                return null;
            });
            if (option != null) options.add(option);
        }

        return options;
    }

    private static List<Object> createDetailOptions(Class<?> switchOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // Helper for consistent option creation
        java.util.function.BiFunction<String, java.util.function.Function<Boolean, Void>, Object> createOption = (key, setter) -> {
            try {
                return switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(class_2561.method_43471("vulkanmod-extra.option." + key),
                        (java.util.function.Consumer<Boolean>) setter::apply,
                        (java.util.function.Supplier<Boolean>) () -> {
                            try {
                                var field = VulkanModExtra.CONFIG.detailSettings.getClass().getDeclaredField(key);
                                field.setAccessible(true);
                                return field.getBoolean(VulkanModExtra.CONFIG.detailSettings);
                            } catch (Exception e) { return true; }
                        });
            } catch (Exception e) { return null; }
        };

        // Detail options from sodium-extra
        String[] detailTypes = {"sky", "sun", "moon", "stars", "rainSnow", "biomeColors", "skyColors"};
        for (String type : detailTypes) {
            Object option = createOption.apply(type, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.detailSettings.getClass().getDeclaredField(type);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.detailSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.error("Failed to set detail option: " + type, e);
                }
                return null;
            });
            if (option != null) options.add(option);
        }

        return options;
    }

    private static List<Object> createRenderOptions(Class<?> switchOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // Helper for consistent option creation
        java.util.function.BiFunction<String, java.util.function.Function<Boolean, Void>, Object> createOption = (key, setter) -> {
            try {
                return switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(class_2561.method_43471("vulkanmod-extra.option." + key),
                        (java.util.function.Consumer<Boolean>) setter::apply,
                        (java.util.function.Supplier<Boolean>) () -> {
                            try {
                                var field = VulkanModExtra.CONFIG.renderSettings.getClass().getDeclaredField(key);
                                field.setAccessible(true);
                                return field.getBoolean(VulkanModExtra.CONFIG.renderSettings);
                            } catch (Exception e) { return true; }
                        });
            } catch (Exception e) { return null; }
        };

        // Render options from sodium-extra
        String[] renderTypes = {"lightUpdates", "itemFrame", "armorStand", "painting", "piston", 
                               "beaconBeam", "limitBeaconBeamHeight", "enchantingTableBook", "itemFrameNameTag", "playerNameTag"};
        for (String type : renderTypes) {
            Object option = createOption.apply(type, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.renderSettings.getClass().getDeclaredField(type);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.renderSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.error("Failed to set render option: " + type, e);
                }
                return null;
            });
            if (option != null) options.add(option);
        }
        
        // Add prevent shaders from extra settings
        Object preventShadersOption = createOption.apply("preventShaders", value -> {
            VulkanModExtra.CONFIG.extraSettings.preventShaders = value;
            VulkanModExtra.CONFIG.writeChanges();
            return null;
        });
        if (preventShadersOption != null) options.add(preventShadersOption);

        return options;
    }

    private static List<Object> createHUDOptions(Class<?> switchOptionClass, Class<?> cyclingOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // FPS display
        class_2561 fpsComponent = class_2561.method_43471("vulkanmod-extra.option.show_fps");
        Object fpsOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(fpsComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.showFps = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.showFps);
        options.add(fpsOption);

        // FPS Display Mode using CyclingOption pattern
        try {
            class_2561 fpsModeComponent = class_2561.method_43471("vulkanmod-extra.option.fps_display_mode");
            
            // Create CyclingOption with FPSDisplayMode enum values
            var fpsDisplayModeValues = com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.values();
            Object fpsModeOption = cyclingOptionClass
                    .getConstructor(class_2561.class, Object[].class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(fpsModeComponent,
                        fpsDisplayModeValues, // All enum values as options
                        (java.util.function.Consumer<com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode>) value -> {
                            VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode = value;
                            VulkanModExtra.CONFIG.writeChanges();
                        },
                        (java.util.function.Supplier<com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode>) () -> 
                            VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode);
            
            // Set translator for display names
            java.lang.reflect.Method setTranslatorMethod = cyclingOptionClass.getMethod("setTranslator", java.util.function.Function.class);
            setTranslatorMethod.invoke(fpsModeOption, 
                (java.util.function.Function<com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode, class_2561>) value -> 
                    class_2561.method_43471(com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.getComponentName(value)));
            
            options.add(fpsModeOption);
        } catch (Exception e) {
            // Fallback to switch option if CyclingOption is not available
            class_2561 fpsModeComponent = class_2561.method_43471("vulkanmod-extra.option.fps_display_mode");
            Object fpsModeOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(fpsModeComponent,
                        (java.util.function.Consumer<Boolean>) value -> {
                            // Cycle through FPS modes: BASIC -> EXTENDED -> DETAILED -> BASIC
                            var currentMode = VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode;
                            var nextMode = switch (currentMode) {
                                case BASIC -> com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.EXTENDED;
                                case EXTENDED -> com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.DETAILED;
                                case DETAILED -> com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.BASIC;
                            };
                            VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode = nextMode;
                            VulkanModExtra.CONFIG.writeChanges();
                        },
                        (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode != com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.BASIC);
            options.add(fpsModeOption);
        }

        // Overlay Corner using CyclingOption
        try {
            class_2561 overlayCornerComponent = class_2561.method_43471("vulkanmod-extra.option.overlay_corner");
            var overlayCornerValues = com.criticalrange.config.VulkanModExtraConfig.OverlayCorner.values();
            Object overlayCornerOption = cyclingOptionClass
                    .getConstructor(class_2561.class, Object[].class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(overlayCornerComponent,
                        overlayCornerValues,
                        (java.util.function.Consumer<com.criticalrange.config.VulkanModExtraConfig.OverlayCorner>) value -> {
                            VulkanModExtra.CONFIG.extraSettings.overlayCorner = value;
                            VulkanModExtra.CONFIG.writeChanges();
                        },
                        (java.util.function.Supplier<com.criticalrange.config.VulkanModExtraConfig.OverlayCorner>) () -> 
                            VulkanModExtra.CONFIG.extraSettings.overlayCorner);
                            
            // Set translator for display names
            java.lang.reflect.Method setTranslatorMethod = cyclingOptionClass.getMethod("setTranslator", java.util.function.Function.class);
            setTranslatorMethod.invoke(overlayCornerOption, 
                (java.util.function.Function<com.criticalrange.config.VulkanModExtraConfig.OverlayCorner, class_2561>) value -> 
                    class_2561.method_43471("vulkanmod-extra.option.overlay_corner." + value.toString().toLowerCase()));
                    
            options.add(overlayCornerOption);
        } catch (Exception e) {
            VulkanModExtra.LOGGER.warn("Failed to create Overlay Corner cycling option", e);
        }

        // Text Contrast using CyclingOption
        try {
            class_2561 textContrastComponent = class_2561.method_43471("vulkanmod-extra.option.text_contrast");
            var textContrastValues = com.criticalrange.config.VulkanModExtraConfig.TextContrast.values();
            Object textContrastOption = cyclingOptionClass
                    .getConstructor(class_2561.class, Object[].class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(textContrastComponent,
                        textContrastValues,
                        (java.util.function.Consumer<com.criticalrange.config.VulkanModExtraConfig.TextContrast>) value -> {
                            VulkanModExtra.CONFIG.extraSettings.textContrast = value;
                            VulkanModExtra.CONFIG.writeChanges();
                        },
                        (java.util.function.Supplier<com.criticalrange.config.VulkanModExtraConfig.TextContrast>) () -> 
                            VulkanModExtra.CONFIG.extraSettings.textContrast);
                            
            // Set translator for display names
            java.lang.reflect.Method setTranslatorMethod2 = cyclingOptionClass.getMethod("setTranslator", java.util.function.Function.class);
            setTranslatorMethod2.invoke(textContrastOption, 
                (java.util.function.Function<com.criticalrange.config.VulkanModExtraConfig.TextContrast, class_2561>) value -> 
                    class_2561.method_43471("vulkanmod-extra.option.text_contrast." + value.toString().toLowerCase()));
                    
            options.add(textContrastOption);
        } catch (Exception e) {
            VulkanModExtra.LOGGER.warn("Failed to create Text Contrast cycling option", e);
        }

        // Coordinates display
        class_2561 coordsComponent = class_2561.method_43471("vulkanmod-extra.option.show_coords");
        Object coordsOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(coordsComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.showCoords = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.showCoords);
        options.add(coordsOption);

        // Toasts
        class_2561 toastsComponent = class_2561.method_43471("vulkanmod-extra.option.toasts");
        Object toastsOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(toastsComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.toasts = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.toasts);
        options.add(toastsOption);

        // Advancement toasts
        class_2561 advancementToastComponent = class_2561.method_43471("vulkanmod-extra.option.advancement_toast");
        Object advancementToastOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(advancementToastComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.advancementToast = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.advancementToast);
        options.add(advancementToastOption);

        // Recipe toasts
        class_2561 recipeToastComponent = class_2561.method_43471("vulkanmod-extra.option.recipe_toast");
        Object recipeToastOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(recipeToastComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.recipeToast = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.recipeToast);
        options.add(recipeToastOption);

        // System toasts
        class_2561 systemToastComponent = class_2561.method_43471("vulkanmod-extra.option.system_toast");
        Object systemToastOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(systemToastComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.systemToast = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.systemToast);
        options.add(systemToastOption);

        // Tutorial toasts
        class_2561 tutorialToastComponent = class_2561.method_43471("vulkanmod-extra.option.tutorial_toast");
        Object tutorialToastOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(tutorialToastComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.tutorialToast = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.tutorialToast);
        options.add(tutorialToastOption);

        // Instant sneak
        class_2561 instantSneakComponent = class_2561.method_43471("vulkanmod-extra.option.instant_sneak");
        Object instantSneakOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(instantSneakComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.instantSneak = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.instantSneak);
        options.add(instantSneakOption);

        // Adaptive sync
        class_2561 adaptiveSyncComponent = class_2561.method_43471("vulkanmod-extra.option.use_adaptive_sync");
        Object adaptiveSyncOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(adaptiveSyncComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.useAdaptiveSync = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.useAdaptiveSync);
        options.add(adaptiveSyncOption);

        // Steady debug HUD
        class_2561 steadyDebugHudComponent = class_2561.method_43471("vulkanmod-extra.option.steady_debug_hud");
        Object steadyDebugHudOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(steadyDebugHudComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.steadyDebugHud = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.steadyDebugHud);
        options.add(steadyDebugHudOption);

        return options;
    }









    /**
     * Custom List implementation that dynamically handles VulkanMod Extra pages
     * This prevents IndexOutOfBoundsException by providing safe access to all pages
     */
    private static class CustomPageList extends ArrayList<Object> {
        private final List<Object> originalPages;
        private final List<Object> extraPages;

        public CustomPageList(List<Object> originalPages, List<Object> extraPages) {
            super();
            this.originalPages = originalPages;
            this.extraPages = extraPages;

            // Add all pages to this list
            addAll(originalPages);
            addAll(extraPages);
        }

        @Override
        public Object get(int index) {
            try {
                // First try to get from original pages
                if (index < originalPages.size()) {
                    return originalPages.get(index);
                }
                // Then try to get from extra pages
                else if (index < originalPages.size() + extraPages.size()) {
                    return extraPages.get(index - originalPages.size());
                }
                // If index is out of bounds, return null instead of crashing
                else {
                    return null;
                }
            } catch (Exception e) {
                return null;
            }
        }

        @Override
        public int size() {
            return originalPages.size() + extraPages.size();
        }

        @Override
        public boolean contains(Object o) {
            return originalPages.contains(o) || extraPages.contains(o);
        }

        @Override
        public int indexOf(Object o) {
            int index = originalPages.indexOf(o);
            if (index >= 0) return index;

            index = extraPages.indexOf(o);
            if (index >= 0) return originalPages.size() + index;

            return -1;
        }
    }

    /**
     * Alternative integration method using mixin injection
     * This would be called by the MixinVOptionScreen
     */
    public static void injectPagesIntoVulkanMod(Object vOptionScreenInstance) {
        try {
            // Check if this screen instance has already been injected
            if (injectedInstances.containsKey(vOptionScreenInstance)) {
                return;
            }

            // Get the VulkanMod Extra pages
            List<Object> extraPages = createVulkanModExtraPages();

            // Get the optionPages field from VOptionScreen
            Class<?> vOptionScreenClass = vOptionScreenInstance.getClass();

            java.lang.reflect.Field optionPagesField = vOptionScreenClass.getDeclaredField("optionPages");
            optionPagesField.setAccessible(true);

            // Cast to the correct type
            @SuppressWarnings("unchecked")
            List<Object> originalOptionPages = (List<Object>) optionPagesField.get(vOptionScreenInstance);

            // Create a custom page list that wraps the original and adds our pages
            CustomPageList customPageList = new CustomPageList(originalOptionPages, extraPages);

            // Replace the original page list with our custom one
            optionPagesField.set(vOptionScreenInstance, customPageList);

                   // Initialize the VOptionList for each new page
                   try {
                       // Use VulkanMod's standard dimensions
                       int top = 40;
                       int bottom = 60;
                       int itemHeight = 20;
                       int leftMargin = 100;
                       int listWidth = 277;
                       int listHeight = 160;

                       // Initialize each new page
                       for (Object page : extraPages) {
                           try {
                               java.lang.reflect.Method createListMethod = page.getClass().getMethod("createList", int.class, int.class, int.class, int.class, int.class);
                               createListMethod.invoke(page, leftMargin, top, listWidth, listHeight, itemHeight);
                           } catch (Exception e) {
                               // Page initialization failed, continue with other pages
                           }
                       }
                   } catch (Exception e) {
                       // Page list initialization failed
                   }

                   // Refresh the UI to include our new pages
                   try {
                       java.lang.reflect.Method buildPageMethod = vOptionScreenInstance.getClass().getDeclaredMethod("buildPage");
                       buildPageMethod.setAccessible(true);
                       buildPageMethod.invoke(vOptionScreenInstance);
                   } catch (Exception e) {
                       // UI refresh failed
                   }

                   VulkanModExtra.LOGGER.info("Successfully injected {} VulkanMod Extra pages into GUI", extraPages.size());

                   // Mark this screen instance as injected to prevent multiple injections for this instance
                   injectedInstances.put(vOptionScreenInstance, true);

        } catch (Exception e) {
            VulkanModExtra.LOGGER.error("Failed to inject pages into VulkanMod GUI", e);
        }
    }



    private static Class<?> findClass(String className) {
        try {
            return Class.forName(className, false, VulkanModExtraIntegration.class.getClassLoader());
        } catch (ClassNotFoundException e) {
            // Try with different class loaders
            try {
                return Class.forName(className, false, Thread.currentThread().getContextClassLoader());
            } catch (ClassNotFoundException ex) {
                return null;
            }
        }
    }

    private static Object getFieldValue(Object instance, String fieldName) {
        try {
            java.lang.reflect.Field field = instance.getClass().getDeclaredField(fieldName);
            field.setAccessible(true);
            return field.get(instance);
        } catch (Exception e) {
            return null;
        }
    }

    private static Object getStaticFieldValue(Class<?> clazz, String fieldName) {
        try {
            java.lang.reflect.Field field = clazz.getDeclaredField(fieldName);
            field.setAccessible(true);
            return field.get(null);
        } catch (Exception e) {
            return null;
        }
    }

    public static boolean isIntegrationSuccessful() {
        return integrationSuccessful;
    }

    // Comprehensive option creation methods

    private static List<Object> createComprehensiveAnimationOptions(Class<?> switchOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // Helper for consistent option creation
        java.util.function.BiFunction<String, java.util.function.Function<Boolean, Void>, Object> createOption = (key, setter) -> {
            try {
                return switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(class_2561.method_43471("vulkanmod-extra.option." + key),
                        (java.util.function.Consumer<Boolean>) setter::apply,
                        (java.util.function.Supplier<Boolean>) () -> {
                            try {
                                var field = VulkanModExtra.CONFIG.animationSettings.getClass().getDeclaredField(key);
                                field.setAccessible(true);
                                return field.getBoolean(VulkanModExtra.CONFIG.animationSettings);
                            } catch (Exception e) { return true; }
                        });
            } catch (Exception e) { return null; }
        };

        // Animation master toggle
        options.add(createOption.apply("animation", value -> {
            VulkanModExtra.CONFIG.animationSettings.animation = value;
            VulkanModExtra.CONFIG.writeChanges();
            return null;
        }));

        // Individual animation options
        String[] animationTypes = {"water", "lava", "fire", "portal", "blockAnimations", "sculkSensor"};
        for (String type : animationTypes) {
            Object option = createOption.apply(type, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.animationSettings.getClass().getDeclaredField(type);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.animationSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.error("Failed to set animation option: " + type, e);
                }
                return null;
            });
            if (option != null) options.add(option);
        }

        return options;
    }

    private static List<Object> createComprehensiveParticleOptions(Class<?> switchOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // Helper for creating particle options
        java.util.function.BiFunction<String, java.util.function.Function<Boolean, Void>, Object> createOption = (key, setter) -> {
            try {
                return switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(class_2561.method_43471("vulkanmod-extra.option." + key),
                        (java.util.function.Consumer<Boolean>) setter::apply,
                        (java.util.function.Supplier<Boolean>) () -> {
                            try {
                                var field = VulkanModExtra.CONFIG.particleSettings.getClass().getDeclaredField(key);
                                field.setAccessible(true);
                                return field.getBoolean(VulkanModExtra.CONFIG.particleSettings);
                            } catch (Exception e) { return true; }
                        });
            } catch (Exception e) { return null; }
        };

        // Particles master toggle
        options.add(createOption.apply("particles", value -> {
            VulkanModExtra.CONFIG.particleSettings.particles = value;
            VulkanModExtra.CONFIG.writeChanges();
            return null;
        }));

        // Core particle types (comprehensive list)
        String[] coreParticles = {
            "rainSplash", "blockBreak", "blockBreaking", "flame", "smoke",
            "bubble", "splash", "rain", "drippingWater", "explosion", "heart",
            "crit", "enchant", "note", "portal", "lava", "firework", "happyVillager",
            "angryVillager", "ash", "campfireCosySmoke", "effect", "dust", "poof",
            "largeSmoke", "smallFlame", "smallGust", "sneeze", "snowflake", "sonicBoom",
            "soul", "soulFireFlame", "spit", "splash", "sporeBlossomAir", "squidInk",
            "sweepAttack", "totemOfUndying", "trialOmen", "trialSpawnerDetection",
            "trialSpawnerDetectionOminous", "underwater", "vaultConnection", "vibration",
            "warpedSpore", "waxOff", "waxOn", "whiteAsh", "whiteSmoke", "witch",
            "ambientEntityEffect", "barrier", "block", "blockCrumble", "blockMarker",
            "bubbleColumnUp", "bubblePop", "campfireSignalSmoke", "cherryLeaves",
            "cloud", "composter", "crimsonSpore", "currentDown", "damageIndicator",
            "dolphin", "dragonBreath", "drippingDripstoneLava", "drippingDripstoneWater",
            "drippingHoney", "drippingLava", "drippingObsidianTear", "dustColorTransition",
            "dustPillar", "dustPlume", "eggCrack", "elderGuardian", "electricSpark",
            "enchantedHit", "endRod", "entityEffect", "explosionEmitter", "fallingDripstoneLava",
            "fallingDripstoneWater", "fallingDust", "fallingHoney", "fallingLava",
            "fallingNectar", "fallingObsidianTear", "fallingSporeBlossom", "fallingWater",
            "fishing", "flash", "glow", "glowSquidInk", "gust", "gustEmitterLarge",
            "gustEmitterSmall", "happyVillager", "infested", "instantEffect", "item",
            "itemCobweb", "itemSlime", "itemSnowball", "landingHoney", "landingLava",
            "landingObsidianTear", "mycelium", "nautilus", "raidOmen", "reversePortal",
            "scrape", "sculkCharge", "sculkChargePop", "sculkSoul", "shriek", "trail"
        };

        for (String particle : coreParticles) {
            Object option = createOption.apply(particle, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.particleSettings.getClass().getDeclaredField(particle);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.particleSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.debug("Particle field not found: " + particle);
                }
                return null;
            });
            if (option != null) options.add(option);
        }

        return options;
    }

    private static List<Object> createComprehensiveDetailOptions(Class<?> switchOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // Helper for consistent option creation
        java.util.function.BiFunction<String, java.util.function.Function<Boolean, Void>, Object> createOption = (key, setter) -> {
            try {
                return switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(class_2561.method_43471("vulkanmod-extra.option." + key),
                        (java.util.function.Consumer<Boolean>) setter::apply,
                        (java.util.function.Supplier<Boolean>) () -> {
                            try {
                                var field = VulkanModExtra.CONFIG.detailSettings.getClass().getDeclaredField(key);
                                field.setAccessible(true);
                                return field.getBoolean(VulkanModExtra.CONFIG.detailSettings);
                            } catch (Exception e) { return true; }
                        });
            } catch (Exception e) { return null; }
        };

        // Detail options from sodium-extra
        String[] detailTypes = {"sky", "sun", "moon", "stars", "rainSnow", "biomeColors", "skyColors"};
        for (String type : detailTypes) {
            Object option = createOption.apply(type, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.detailSettings.getClass().getDeclaredField(type);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.detailSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.error("Failed to set detail option: " + type, e);
                }
                return null;
            });
            if (option != null) options.add(option);
        }

        return options;
    }

    private static List<Object> createComprehensiveRenderOptions(Class<?> switchOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // Helper for consistent option creation
        java.util.function.BiFunction<String, java.util.function.Function<Boolean, Void>, Object> createOption = (key, setter) -> {
            try {
                return switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(class_2561.method_43471("vulkanmod-extra.option." + key),
                        (java.util.function.Consumer<Boolean>) setter::apply,
                        (java.util.function.Supplier<Boolean>) () -> {
                            try {
                                var field = VulkanModExtra.CONFIG.renderSettings.getClass().getDeclaredField(key);
                                field.setAccessible(true);
                                return field.getBoolean(VulkanModExtra.CONFIG.renderSettings);
                            } catch (Exception e) { return true; }
                        });
            } catch (Exception e) { return null; }
        };

        // Basic render options
        String[] renderTypes = {"lightUpdates", "itemFrame", "armorStand", "painting", "piston",
                               "beaconBeam", "limitBeaconBeamHeight", "enchantingTableBook", "itemFrameNameTag", "playerNameTag"};
        for (String type : renderTypes) {
            Object option = createOption.apply(type, value -> {
                try {
                    var field = VulkanModExtra.CONFIG.renderSettings.getClass().getDeclaredField(type);
                    field.setAccessible(true);
                    field.setBoolean(VulkanModExtra.CONFIG.renderSettings, value);
                    VulkanModExtra.CONFIG.writeChanges();
                } catch (Exception e) {
                    VulkanModExtra.LOGGER.error("Failed to set render option: " + type, e);
                }
                return null;
            });
            if (option != null) options.add(option);
        }

        // Add fog options
        Object fogOption = createOption.apply("global_fog", value -> {
            VulkanModExtra.CONFIG.renderSettings.globalFog = value;
            VulkanModExtra.CONFIG.writeChanges();
            return null;
        });
        if (fogOption != null) options.add(fogOption);

        Object multiDimFogOption = createOption.apply("multi_dimension_fog", value -> {
            VulkanModExtra.CONFIG.renderSettings.multiDimensionFog = value;
            VulkanModExtra.CONFIG.writeChanges();
            return null;
        });
        if (multiDimFogOption != null) options.add(multiDimFogOption);

        // Add prevent shaders from extra settings
        Object preventShadersOption = createOption.apply("preventShaders", value -> {
            VulkanModExtra.CONFIG.extraSettings.preventShaders = value;
            VulkanModExtra.CONFIG.writeChanges();
            return null;
        });
        if (preventShadersOption != null) options.add(preventShadersOption);

        return options;
    }

    private static List<Object> createComprehensiveHUDOptions(Class<?> switchOptionClass, Class<?> cyclingOptionClass) throws Exception {
        List<Object> options = new ArrayList<>();

        // FPS display
        class_2561 fpsComponent = class_2561.method_43471("vulkanmod-extra.option.show_fps");
        Object fpsOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(fpsComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.showFps = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.showFps);
        options.add(fpsOption);

        // FPS Display Mode using CyclingOption pattern
        try {
            class_2561 fpsModeComponent = class_2561.method_43471("vulkanmod-extra.option.fps_display_mode");

            // Create CyclingOption with FPSDisplayMode enum values
            var fpsDisplayModeValues = com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.values();
            Object fpsModeOption = cyclingOptionClass
                    .getConstructor(class_2561.class, Object[].class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(fpsModeComponent,
                        fpsDisplayModeValues, // All enum values as options
                        (java.util.function.Consumer<com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode>) value -> {
                            VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode = value;
                            VulkanModExtra.CONFIG.writeChanges();
                        },
                        (java.util.function.Supplier<com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode>) () ->
                            VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode);

            // Set translator for display names
            java.lang.reflect.Method setTranslatorMethod = cyclingOptionClass.getMethod("setTranslator", java.util.function.Function.class);
            setTranslatorMethod.invoke(fpsModeOption,
                (java.util.function.Function<com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode, class_2561>) value ->
                    class_2561.method_43471(com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.getComponentName(value)));

            options.add(fpsModeOption);
        } catch (Exception e) {
            // Fallback to switch option if CyclingOption is not available
            class_2561 fpsModeComponent = class_2561.method_43471("vulkanmod-extra.option.fps_display_mode");
            Object fpsModeOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(fpsModeComponent,
                        (java.util.function.Consumer<Boolean>) value -> {
                            // Cycle through FPS modes: BASIC -> EXTENDED -> DETAILED -> BASIC
                            var currentMode = VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode;
                            var nextMode = switch (currentMode) {
                                case BASIC -> com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.EXTENDED;
                                case EXTENDED -> com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.DETAILED;
                                case DETAILED -> com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.BASIC;
                            };
                            VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode = nextMode;
                            VulkanModExtra.CONFIG.writeChanges();
                        },
                        (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.fpsDisplayMode != com.criticalrange.config.VulkanModExtraConfig.FPSDisplayMode.BASIC);
            options.add(fpsModeOption);
        }

        // Overlay Corner using CyclingOption
        try {
            class_2561 overlayCornerComponent = class_2561.method_43471("vulkanmod-extra.option.overlay_corner");
            var overlayCornerValues = com.criticalrange.config.VulkanModExtraConfig.OverlayCorner.values();
            Object overlayCornerOption = cyclingOptionClass
                    .getConstructor(class_2561.class, Object[].class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(overlayCornerComponent,
                        overlayCornerValues,
                        (java.util.function.Consumer<com.criticalrange.config.VulkanModExtraConfig.OverlayCorner>) value -> {
                            VulkanModExtra.CONFIG.extraSettings.overlayCorner = value;
                            VulkanModExtra.CONFIG.writeChanges();
                        },
                        (java.util.function.Supplier<com.criticalrange.config.VulkanModExtraConfig.OverlayCorner>) () ->
                            VulkanModExtra.CONFIG.extraSettings.overlayCorner);

            // Set translator for display names
            java.lang.reflect.Method setTranslatorMethod = cyclingOptionClass.getMethod("setTranslator", java.util.function.Function.class);
            setTranslatorMethod.invoke(overlayCornerOption,
                (java.util.function.Function<com.criticalrange.config.VulkanModExtraConfig.OverlayCorner, class_2561>) value ->
                    class_2561.method_43471("vulkanmod-extra.option.overlay_corner." + value.toString().toLowerCase()));

            options.add(overlayCornerOption);
        } catch (Exception e) {
            VulkanModExtra.LOGGER.warn("Failed to create Overlay Corner cycling option", e);
        }

        // Text Contrast using CyclingOption
        try {
            class_2561 textContrastComponent = class_2561.method_43471("vulkanmod-extra.option.text_contrast");
            var textContrastValues = com.criticalrange.config.VulkanModExtraConfig.TextContrast.values();
            Object textContrastOption = cyclingOptionClass
                    .getConstructor(class_2561.class, Object[].class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                    .newInstance(textContrastComponent,
                        textContrastValues,
                        (java.util.function.Consumer<com.criticalrange.config.VulkanModExtraConfig.TextContrast>) value -> {
                            VulkanModExtra.CONFIG.extraSettings.textContrast = value;
                            VulkanModExtra.CONFIG.writeChanges();
                        },
                        (java.util.function.Supplier<com.criticalrange.config.VulkanModExtraConfig.TextContrast>) () ->
                            VulkanModExtra.CONFIG.extraSettings.textContrast);

            // Set translator for display names
            java.lang.reflect.Method setTranslatorMethod2 = cyclingOptionClass.getMethod("setTranslator", java.util.function.Function.class);
            setTranslatorMethod2.invoke(textContrastOption,
                (java.util.function.Function<com.criticalrange.config.VulkanModExtraConfig.TextContrast, class_2561>) value ->
                    class_2561.method_43471("vulkanmod-extra.option.text_contrast." + value.toString().toLowerCase()));

            options.add(textContrastOption);
        } catch (Exception e) {
            VulkanModExtra.LOGGER.warn("Failed to create Text Contrast cycling option", e);
        }

        // Coordinates display
        class_2561 coordsComponent = class_2561.method_43471("vulkanmod-extra.option.show_coords");
        Object coordsOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(coordsComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.showCoords = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.showCoords);
        options.add(coordsOption);

        // Toasts
        class_2561 toastsComponent = class_2561.method_43471("vulkanmod-extra.option.toasts");
        Object toastsOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(toastsComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.toasts = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.toasts);
        options.add(toastsOption);

        // Advancement toasts
        class_2561 advancementToastComponent = class_2561.method_43471("vulkanmod-extra.option.advancement_toast");
        Object advancementToastOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(advancementToastComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.advancementToast = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.advancementToast);
        options.add(advancementToastOption);

        // Recipe toasts
        class_2561 recipeToastComponent = class_2561.method_43471("vulkanmod-extra.option.recipe_toast");
        Object recipeToastOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(recipeToastComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.recipeToast = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.recipeToast);
        options.add(recipeToastOption);

        // System toasts
        class_2561 systemToastComponent = class_2561.method_43471("vulkanmod-extra.option.system_toast");
        Object systemToastOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(systemToastComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.systemToast = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.systemToast);
        options.add(systemToastOption);

        // Tutorial toasts
        class_2561 tutorialToastComponent = class_2561.method_43471("vulkanmod-extra.option.tutorial_toast");
        Object tutorialToastOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(tutorialToastComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.tutorialToast = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.tutorialToast);
        options.add(tutorialToastOption);

        // Instant sneak
        class_2561 instantSneakComponent = class_2561.method_43471("vulkanmod-extra.option.instant_sneak");
        Object instantSneakOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(instantSneakComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.instantSneak = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.instantSneak);
        options.add(instantSneakOption);

        // Adaptive sync
        class_2561 adaptiveSyncComponent = class_2561.method_43471("vulkanmod-extra.option.use_adaptive_sync");
        Object adaptiveSyncOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(adaptiveSyncComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.useAdaptiveSync = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.useAdaptiveSync);
        options.add(adaptiveSyncOption);

        // Steady debug HUD
        class_2561 steadyDebugHudComponent = class_2561.method_43471("vulkanmod-extra.option.steady_debug_hud");
        Object steadyDebugHudOption = switchOptionClass.getConstructor(class_2561.class, java.util.function.Consumer.class, java.util.function.Supplier.class)
                .newInstance(steadyDebugHudComponent,
                    (java.util.function.Consumer<Boolean>) value -> {
                        VulkanModExtra.CONFIG.extraSettings.steadyDebugHud = value;
                        VulkanModExtra.CONFIG.writeChanges();
                    },
                    (java.util.function.Supplier<Boolean>) () -> VulkanModExtra.CONFIG.extraSettings.steadyDebugHud);
        options.add(steadyDebugHudOption);

        return options;
    }
}
