/*
 * Copyright (c) NeoForged and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package net.neoforged.neoforge.client.gui;

import com.electronwill.nightconfig.core.ConfigSpec;
import com.electronwill.nightconfig.core.UnmodifiableConfig;
import com.electronwill.nightconfig.core.UnmodifiableConfig.Entry;
import com.google.common.collect.ImmutableList;
import com.mojang.serialization.Codec;
import it.unimi.dsi.fastutil.booleans.BooleanConsumer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_315;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_342;
import net.minecraft.class_3532;
import net.minecraft.class_364;
import net.minecraft.class_410;
import net.minecraft.class_4185;
import net.minecraft.class_424;
import net.minecraft.class_4325;
import net.minecraft.class_437;
import net.minecraft.class_442;
import net.minecraft.class_4667;
import net.minecraft.class_4926.class_6291;
import net.minecraft.class_500;
import net.minecraft.class_5244;
import net.minecraft.class_5250;
import net.minecraft.class_5676;
import net.minecraft.class_6382;
import net.minecraft.class_642;
import net.minecraft.class_7172;
import net.minecraft.class_7842;
import net.minecraft.class_7919;
import net.minecraft.class_8667;
import net.minecraft.class_9017;
import net.neoforged.fml.ModContainer;
import net.neoforged.fml.config.ModConfig;
import net.neoforged.fml.config.ModConfig.Type;
import net.neoforged.fml.config.ModConfigs;
import net.neoforged.neoforge.common.ModConfigSpec;
import net.neoforged.neoforge.common.ModConfigSpec.ConfigValue;
import net.neoforged.neoforge.common.ModConfigSpec.ListValueSpec;
import net.neoforged.neoforge.common.ModConfigSpec.Range;
import net.neoforged.neoforge.common.ModConfigSpec.RestartType;
import net.neoforged.neoforge.common.ModConfigSpec.ValueSpec;
import net.neoforged.neoforge.common.TranslatableEnum;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Strings;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

@SuppressWarnings("SpellCheckingInspection")
public final class ConfigScreen extends class_4667 {
    private static final class TooltipConfirmScreen extends class_410 {
        boolean seenYes = false;

        private TooltipConfirmScreen(BooleanConsumer callback, class_2561 title, class_2561 message, class_2561 yesButton, class_2561 noButton) {
            super(callback, title, message, yesButton, noButton);
        }

        @Override
        protected void method_25426() {
            seenYes = false;
            super.method_25426();
        }

        @Override
        protected void method_37052(class_4185 button) {
            if (seenYes) {
                button.method_47400(class_7919.method_47407(RESTART_NO_TOOLTIP));
            } else {
                seenYes = true;
            }
            super.method_37052(button);
        }
    }


    public static class TranslationChecker {
        private static final Logger LOGGER = LogManager.getLogger();
        private final Set<String> untranslatables = new HashSet<>();
        private final Set<String> untranslatablesWithFallback = new HashSet<>();

        public String check(final String translationKey) {
            if (!class_1074.method_4663(translationKey)) {
                untranslatables.add(translationKey);
            }
            return translationKey;
        }

        public String check(final String translationKey, final String fallback) {
            if (!class_1074.method_4663(translationKey)) {
                untranslatablesWithFallback.add(translationKey);
                return check(fallback);
            }
            return translationKey;
        }

        public boolean existsWithFallback(final String translationKey) {
            if (!class_1074.method_4663(translationKey)) {
                untranslatablesWithFallback.add(translationKey);
                return false;
            }
            return true;
        }

        /**
         * If the given translation key exists, returns it formatted with the given style(s) and as prefixed with the given component.
         * Otherwise returns an empty Component.
         */
        public class_2561 optional(final class_2561 prefix, final String translationKey, final class_124... style) {
            if (class_1074.method_4663(translationKey)) {
                return class_2561.method_43473().method_10852(prefix).method_10852(class_2561.method_43471(translationKey).method_27695(style));
            }
            return class_2561.method_43473();
        }

        public void finish() {
            if (/*CLIENT.logUntranslatedConfigurationWarnings.get() &&*/ FabricLoader.getInstance().isDevelopmentEnvironment() && (!untranslatables.isEmpty() || !untranslatablesWithFallback.isEmpty())) {
                StringBuilder stringBuilder = new StringBuilder();
                stringBuilder.append("""
                        \n	Dev warning - Untranslated configuration keys encountered. Please translate your configuration keys so users can properly configure your mod.
                        """);
                if (!untranslatables.isEmpty()) {
                    stringBuilder.append("\nUntranslated keys:");
                    for (String key : untranslatables) {
                        stringBuilder.append("\n  \"").append(key).append("\": \"\",");
                    }
                }
                if (!untranslatablesWithFallback.isEmpty()) {
                    stringBuilder.append("\nThe following keys have fallbacks. Please check if those are suitable, and translate them if they're not.");
                    for (String key : untranslatablesWithFallback) {
                        stringBuilder.append("\n  \"").append(key).append("\": \"\",");
                    }
                }

                LOGGER.warn(stringBuilder);
            }
            untranslatables.clear();
        }
    }

    /**
     * Prefix for static keys the configuration screens use internally.
     */
    private static final String LANG_PREFIX = "neoforge.configuration.uitext.";
    /**
     * A wrapper for the labels of buttons that open a new screen. Default: "%s..."
     */
    private static final String SECTION = LANG_PREFIX + "section";
    /**
     * A default for the labels of buttons that open a new screen. Default: "Edit"
     */
    private static final String SECTION_TEXT = LANG_PREFIX + "sectiontext";
    /**
     * The breadcrumb separator. Default: "%s > %s"
     */
    public static final class_2561 CRUMB_SEPARATOR = class_2561.method_43471(LANG_PREFIX + "breadcrumb.separator").method_27695(class_124.field_1065, class_124.field_1067);
    private static final String CRUMB = LANG_PREFIX + "breadcrumb.order";
    /**
     * The label of list elements. Will be supplied the index into the list. Default: "%s:"
     */
    private static final String LIST_ELEMENT = LANG_PREFIX + "listelement";
    /**
     * How the range will be added to the tooltip when using translated tooltips. Mimics what the comment does in ModConfigSpec.
     */
    private static final String RANGE_TOOLTIP = LANG_PREFIX + "rangetooltip";
    private static final class_124 RANGE_TOOLTIP_STYLE = class_124.field_1080;
    /**
     * How the filename will be added to the tooltip.
     */
    private static final String FILENAME_TOOLTIP = LANG_PREFIX + "filenametooltip";
    private static final class_124 FILENAME_TOOLTIP_STYLE = class_124.field_1080;
    /**
     * A literal to create an empty line in a tooltip.
     */
    private static final class_5250 EMPTY_LINE = class_2561.method_43470("\n\n");

    public static final class_2561 TOOLTIP_CANNOT_EDIT_THIS_WHILE_ONLINE = class_2561.method_43471(LANG_PREFIX + "notonline").method_27692(class_124.field_1061);
    public static final class_2561 TOOLTIP_CANNOT_EDIT_THIS_WHILE_OPEN_TO_LAN = class_2561.method_43471(LANG_PREFIX + "notlan").method_27692(class_124.field_1061);
    public static final class_2561 TOOLTIP_CANNOT_EDIT_NOT_LOADED = class_2561.method_43471(LANG_PREFIX + "notloaded").method_27692(class_124.field_1061);
    public static final class_2561 NEW_LIST_ELEMENT = class_2561.method_43471(LANG_PREFIX + "newlistelement");
    public static final class_2561 MOVE_LIST_ELEMENT_UP = class_2561.method_43471(LANG_PREFIX + "listelementup");
    public static final class_2561 MOVE_LIST_ELEMENT_DOWN = class_2561.method_43471(LANG_PREFIX + "listelementdown");
    public static final class_2561 REMOVE_LIST_ELEMENT = class_2561.method_43471(LANG_PREFIX + "listelementremove");
    public static final class_2561 UNSUPPORTED_ELEMENT = class_2561.method_43471(LANG_PREFIX + "unsupportedelement").method_27692(class_124.field_1061);
    public static final class_2561 LONG_STRING = class_2561.method_43471(LANG_PREFIX + "longstring").method_27692(class_124.field_1061);
    public static final class_2561 GAME_RESTART_TITLE = class_2561.method_43471(LANG_PREFIX + "restart.game.title");
    public static final class_2561 GAME_RESTART_MESSAGE = class_2561.method_43471(LANG_PREFIX + "restart.game.text");
    public static final class_2561 GAME_RESTART_YES = class_2561.method_43471("menu.quit"); // TitleScreen.init() et.al.
    public static final class_2561 SERVER_RESTART_TITLE = class_2561.method_43471(LANG_PREFIX + "restart.server.title");
    public static final class_2561 SERVER_RESTART_MESSAGE = class_2561.method_43471(LANG_PREFIX + "restart.server.text");
    public static final class_2561 RETURN_TO_MENU = class_2561.method_43471("menu.returnToMenu"); // PauseScreen.RETURN_TO_MENU
    public static final class_2561 SAVING_LEVEL = class_2561.method_43471("menu.savingLevel"); // PauseScreen.SAVING_LEVEL
    public static final class_2561 RESTART_NO = class_2561.method_43471(LANG_PREFIX + "restart.return");
    public static final class_2561 RESTART_NO_TOOLTIP = class_2561.method_43471(LANG_PREFIX + "restart.return.tooltip").method_27695(class_124.field_1061, class_124.field_1067);
    public static final class_2561 UNDO = class_2561.method_43471(LANG_PREFIX + "undo");
    public static final class_2561 UNDO_TOOLTIP = class_2561.method_43471(LANG_PREFIX + "undo.tooltip");
    public static final class_2561 RESET = class_2561.method_43471(LANG_PREFIX + "reset");
    public static final class_2561 RESET_TOOLTIP = class_2561.method_43471(LANG_PREFIX + "reset.tooltip");
    public static final int BIG_BUTTON_WIDTH = 310;

    // Ideally this should not be static, but we need it in the construtor's super() call
    private static final TranslationChecker translationChecker = new TranslationChecker();

    private final ModContainer mod;
    private final class_6291<ConfigScreen, ModConfig.Type, ModConfig, class_2561, class_437> sectionScreen;

    public RestartType needsRestart = RestartType.NONE;
    // If there is only one config type (and it can be edited, we show that instantly on the way "down" and want to close on the way "up".
    // But when returning from the restart/reload confirmation screens, we need to stay open.
    private boolean autoClose = false;

    public ConfigScreen(final ModContainer mod, final class_437 parent) {
        this(mod, parent, ConfigurationSectionScreen::new);
    }

    public ConfigScreen(final ModContainer mod, final class_437 parent, ConfigurationSectionScreen.Filter filter) {
        this(mod, parent, (a, b, c, d) -> new ConfigurationSectionScreen(a, b, c, d, filter));
    }

    @SuppressWarnings("resource")
    public ConfigScreen(final ModContainer mod, final class_437 parent, class_6291<ConfigScreen, ModConfig.Type, ModConfig, class_2561, class_437> sectionScreen) {
        super(parent, class_310.method_1551().field_1690, class_2561.method_43469(translationChecker.check(mod.getModId() + ".configuration.title", LANG_PREFIX + "title"), mod.getModInfo().getDisplayName()));
        this.mod = mod;
        this.sectionScreen = sectionScreen;
    }

    @Override
    protected void method_60325() {
        class_4185 btn = null;
        int count = 0;
        for (final Type type : ModConfig.Type.values()) {
            boolean headerAdded = false;
            for (final ModConfig modConfig : ModConfigs.getConfigSet(type)) {
                if (modConfig.getModId().equals(mod.getModId())) {
                    if (!headerAdded) {
                        field_51824.method_20407(new class_7842(BIG_BUTTON_WIDTH, class_4185.field_39501,
                                class_2561.method_43471(LANG_PREFIX + type.name().toLowerCase(Locale.ENGLISH)).method_27692(class_124.field_1073), field_22793).method_48596(), null);
                        headerAdded = true;
                    }
                    btn = class_4185.method_46430(class_2561.method_43469(SECTION, translatableConfig(modConfig, "", LANG_PREFIX + "type." + modConfig.getType().name().toLowerCase(Locale.ROOT))),
                            button -> field_22787.method_1507(sectionScreen.method_35906(this, type, modConfig, translatableConfig(modConfig, ".title", LANG_PREFIX + "title." + type.name().toLowerCase(Locale.ROOT))))).method_46432(BIG_BUTTON_WIDTH).method_46431();
                    class_5250 tooltip = class_2561.method_43473();
                    if (!((ModConfigSpec) modConfig.getSpec()).isLoaded()) {
                        tooltip.method_10852(TOOLTIP_CANNOT_EDIT_NOT_LOADED).method_10852(EMPTY_LINE);
                        btn.field_22763 = false;
                        count = 99; // prevent autoClose
                    } else if (type == Type.SERVER && field_22787.method_1558() != null && !field_22787.method_47392()) {
                        tooltip.method_10852(TOOLTIP_CANNOT_EDIT_THIS_WHILE_ONLINE).method_10852(EMPTY_LINE);
                        btn.field_22763 = false;
                        count = 99; // prevent autoClose
                    } else if (type == Type.SERVER && field_22787.method_1496() && field_22787.method_1576().method_3860()) {
                        tooltip.method_10852(TOOLTIP_CANNOT_EDIT_THIS_WHILE_OPEN_TO_LAN).method_10852(EMPTY_LINE);
                        btn.field_22763 = false;
                        count = 99; // prevent autoClose
                    }
                    tooltip.method_10852(class_2561.method_43469(FILENAME_TOOLTIP, modConfig.getFileName()).method_27692(FILENAME_TOOLTIP_STYLE));
                    btn.method_47400(class_7919.method_47407(tooltip));
                    field_51824.method_20407(btn, null);
                    count++;
                }
            }
        }
        if (count == 1) {
            autoClose = true;
            btn.method_25306();
        }
    }

    public class_2561 translatableConfig(ModConfig modConfig, String suffix, String fallback) {
        return class_2561.method_43469(translationChecker.check(mod.getModId() + ".configuration.section." + modConfig.getFileName().replaceAll("[^a-zA-Z0-9]+", ".").replaceFirst("^\\.", "").replaceFirst("\\.$", "").toLowerCase(Locale.ENGLISH) + suffix, fallback), mod.getModInfo().getDisplayName());
    }

    @Override
    public void method_49589() {
        super.method_49589();
        if (autoClose) {
            autoClose = false;
            method_25419();
        }
    }

    @SuppressWarnings("incomplete-switch")
    @Override
    public void method_25419() {
        translationChecker.finish();
        switch (needsRestart) {
            case GAME -> {
                field_22787.method_1507(new TooltipConfirmScreen(b -> {
                    if (b) {
                        field_22787.method_1592();
                    } else {
                        super.method_25419();
                    }
                }, GAME_RESTART_TITLE, GAME_RESTART_MESSAGE, GAME_RESTART_YES, RESTART_NO));
                return;
            }
            case WORLD -> {
                if (field_22787.field_1687 != null) {
                    field_22787.method_1507(new TooltipConfirmScreen(b -> {
                        if (b) {
                            // when changing server configs from the client is added, this is where we tell the server to restart and activate the new config.
                            // also needs a different text in MP ("server will restart/exit, yada yada") than in SP
                            onDisconnect();
                        } else {
                            super.method_25419();
                        }
                    }, SERVER_RESTART_TITLE, SERVER_RESTART_MESSAGE, field_22787.method_1542() ? RETURN_TO_MENU : class_5244.field_45692, RESTART_NO));
                    return;
                }
            }
        }
        super.method_25419();
    }

    // direct copy from PauseScreen (which has the best implementation), sadly it's not really accessible
    private void onDisconnect() {
        boolean flag = this.field_22787.method_1542();
        class_642 serverdata = this.field_22787.method_1558();
        this.field_22787.field_1687.method_8525();
        if (flag) {
            this.field_22787.method_56134(new class_424(SAVING_LEVEL));
        } else {
            this.field_22787.method_18099();
        }

        class_442 titlescreen = new class_442();
        if (flag) {
            this.field_22787.method_1507(titlescreen);
        } else if (serverdata != null && serverdata.method_52811()) {
            this.field_22787.method_1507(new class_4325(titlescreen));
        } else {
            this.field_22787.method_1507(new class_500(titlescreen));
        }
    }

    public static class ConfigurationSectionScreen extends class_4667 {
        protected static final long MAX_SLIDER_SIZE = 256L;

        public record Context(String modId, class_437 parent, ModConfig modConfig, ModConfigSpec modSpec,
                Set<? extends Entry> entries, Map<String, Object> valueSpecs, List<String> keylist, Filter filter) {
            @ApiStatus.Internal
            public Context {}

            public static Context top(final String modId, final class_437 parent, final ModConfig modConfig, Filter filter) {
                return new Context(modId, parent, modConfig, (ModConfigSpec) modConfig.getSpec(), ((ModConfigSpec) modConfig.getSpec()).getValues().entrySet(),
                        ((ModConfigSpec) modConfig.getSpec()).getSpec().valueMap(), List.of(), filter);
            }

            public static Context section(final Context parentContext, final class_437 parent, final Set<? extends Entry> entries, final Map<String, Object> valueSpecs,
                    final String key) {
                return new Context(parentContext.modId, parent, parentContext.modConfig, parentContext.modSpec, entries, valueSpecs,
                        parentContext.makeKeyList(key), parentContext.filter);
            }

            public static Context list(final Context parentContext, final class_437 parent) {
                return new Context(parentContext.modId, parent, parentContext.modConfig, parentContext.modSpec,
                        parentContext.entries, parentContext.valueSpecs, parentContext.keylist, null);
            }

            private ArrayList<String> makeKeyList(final String key) {
                final ArrayList<String> result = new ArrayList<>(keylist);
                result.add(key);
                return result;
            }
        }

        public record Element(@Nullable class_2561 name, @Nullable class_2561 tooltip, @Nullable class_339 widget, @Nullable class_7172<?> option, boolean undoable) {
            @ApiStatus.Internal
            public Element {}

            public Element(@Nullable final class_2561 name, @Nullable final class_2561 tooltip, final class_339 widget) {
                this(name, tooltip, widget, null, true);
            }

            public Element(@Nullable final class_2561 name, @Nullable final class_2561 tooltip, final class_339 widget, boolean undoable) {
                this(name, tooltip, widget, null, undoable);
            }

            public Element(final class_2561 name, final class_2561 tooltip, final class_7172<?> option) {
                this(name, tooltip, null, option, true);
            }

            public Element(final class_2561 name, final class_2561 tooltip, final class_7172<?> option, boolean undoable) {
                this(name, tooltip, null, option, undoable);
            }

            public class_339 getWidget(final class_315 options) {
                return widget != null ? widget : option.method_57701(options);
            }

            @Nullable
            public Object any() {
                return widget != null ? widget : option;
            }
        }

        /**
         * A filter callback to suppress certain elements from being shown in the configuration UI.
         * <p>
         * Return null to suppress the element or return a modified Element.
         */
        public interface Filter {
            @Nullable
            Element filterEntry(Context context, String key, Element original);
        }

        protected final Context context;
        protected boolean changed = false;
        protected RestartType needsRestart = RestartType.NONE;
        protected final Map<String, ConfigurationSectionScreen> sectionCache = new HashMap<>();
        @Nullable
        protected class_4185 undoButton, resetButton; // must not be changed after creation unless the reference inside the layout also is replaced
        protected final class_4185 doneButton = class_4185.method_46430(class_5244.field_24334, button -> method_25419()).method_46432(class_4185.field_39499).method_46431();
        protected final UndoManager undoManager = new UndoManager();

        /**
         * Constructs a new section screen for the top-most section in a {@link ModConfig}.
         * 
         * @param parent    The screen to return to when the user presses escape or the "Done" button.
         *                  If this is a {@link ConfigScreen}, additional information is passed before closing.
         * @param type      The {@link Type} this configuration is for. Only used to generate the title of the screen.
         * @param modConfig The actual config to show and edit.
         */
        public ConfigurationSectionScreen(final class_437 parent, final ModConfig.Type type, final ModConfig modConfig, class_2561 title) {
            this(parent, type, modConfig, title, (c, k, e) -> e);
        }

        /**
         * Constructs a new section screen for the top-most section in a {@link ModConfig}.
         * 
         * @param parent    The screen to return to when the user presses escape or the "Done" button.
         *                  If this is a {@link ConfigScreen}, additional information is passed before closing.
         * @param type      The {@link Type} this configuration is for. Only used to generate the title of the screen.
         * @param filter    The {@link Filter} to use.
         * @param modConfig The actual config to show and edit.
         */
        public ConfigurationSectionScreen(final class_437 parent, final ModConfig.Type type, final ModConfig modConfig, class_2561 title, Filter filter) {
            this(Context.top(modConfig.getModId(), parent, modConfig, filter), title);
            needsRestart = type == Type.STARTUP ? RestartType.GAME : RestartType.NONE;
        }

        /**
         * Constructs a new section screen for a sub-section of a config.
         * 
         * @param parentContext The {@link Context} object of the parent.
         * @param parent        The screen to return to when the user presses escape or the "Done" button.
         *                      If this is a {@link ConfigurationSectionScreen}, additional information is passed before closing.
         * @param valueSpecs    The source for the {@link ValueSpec} objects for this section.
         * @param key           The key of the section.
         * @param entrySet      The source for the {@link ConfigValue} objects for this section.
         */
        public ConfigurationSectionScreen(final Context parentContext, final class_437 parent, final Map<String, Object> valueSpecs, final String key,
                final Set<? extends Entry> entrySet, class_2561 title) {
            this(Context.section(parentContext, parent, entrySet, valueSpecs, key), class_2561.method_43469(CRUMB, parent.method_25440(), CRUMB_SEPARATOR, title));
        }

        @SuppressWarnings("resource")
        protected ConfigurationSectionScreen(final Context context, final class_2561 title) {
            super(context.parent, class_310.method_1551().field_1690, title);
            this.context = context;
        }

        @Nullable
        protected ValueSpec getValueSpec(final String key) {
            final Object object = context.valueSpecs.get(key);
            if (object instanceof final ValueSpec vs) {
                return vs;
            } else {
                return null;
            }
        }

        protected String getTranslationKey(final String key) {
            final ValueSpec valueSpec = getValueSpec(key);
            final String result = valueSpec != null ? valueSpec.getTranslationKey() : context.modSpec.getLevelTranslationKey(context.makeKeyList(key));
            return result != null ? result : context.modId + ".configuration." + key;
        }

        protected class_5250 getTranslationComponent(final String key) {
            return class_2561.method_43471(translationChecker.check(getTranslationKey(key)));
        }

        protected String getComment(final String key) {
            final ValueSpec valueSpec = getValueSpec(key);
            return valueSpec != null ? getValueSpec(key).getComment() : context.modSpec.getLevelComment(context.makeKeyList(key));
        }

        protected <T> class_7172.class_7277<T> getTooltip(final String key, @Nullable Range<?> range) {
            return class_7172.method_42717(getTooltipComponent(key, range));
        }

        protected class_2561 getTooltipComponent(final String key, @Nullable Range<?> range) {
            final String tooltipKey = getTranslationKey(key) + ".tooltip";
            final String comment = getComment(key);
            final boolean hasTranslatedTooltip = translationChecker.existsWithFallback(tooltipKey);
            class_5250 component = class_2561.method_43473().method_10852(getTranslationComponent(key).method_27692(class_124.field_1067));
            if (hasTranslatedTooltip || !Strings.isBlank(comment)) {
                component = component.method_10852(EMPTY_LINE).method_10852(class_2561.method_48321(tooltipKey, comment));
            }
            // The "tooltip.warning" key is to be considered an internal API. It will be removed once Neo has a
            // generic styling mechanism for translation texts. Use at your own risk.
            component = component.method_10852(translationChecker.optional(EMPTY_LINE, tooltipKey + ".warning", class_124.field_1061, class_124.field_1067));
            if (hasTranslatedTooltip && range != null) {
                component = component.method_10852(EMPTY_LINE).method_10852(class_2561.method_43469(RANGE_TOOLTIP, range.toString()).method_27692(RANGE_TOOLTIP_STYLE));
            }
            return component;
        }

        /**
         * This is called whenever a value is changed and the change is submitted to the appropriate {@link ConfigSpec}.
         * 
         * @param key The key of the changed configuration. To get an absolute key, use {@link Context#makeKeyList(String)}.
         */
        protected void onChanged(final String key) {
            changed = true;
            final ValueSpec valueSpec = getValueSpec(key);
            if (valueSpec != null) {
                needsRestart = needsRestart.with(valueSpec.restartType());
            }
        }

        @Override
        protected void method_60325() {
            rebuild();
        }

        @SuppressWarnings({ "unchecked", "rawtypes" })
        protected ConfigurationSectionScreen rebuild() {
            if (list != null) { // this may be called early, skip and wait for init() then
                list.children().clear();
                boolean hasUndoableElements = false;

                final List<@Nullable Element> elements = new ArrayList<>();
                for (final Entry entry : context.entries) {
                    final String key = entry.getKey();
                    final Object rawValue = entry.getRawValue();
                    switch (entry.getRawValue()) {
                        case ConfigValue cv -> {
                            var valueSpec = getValueSpec(key);
                            var element = switch (valueSpec) {
                                case ListValueSpec listValueSpec -> createList(key, listValueSpec, cv);
                                case ValueSpec spec when cv.getClass() == ConfigValue.class && spec.getDefault() instanceof String -> createStringValue(key, valueSpec::test, () -> (String) cv.getRaw(), cv::set);
                                case ValueSpec spec when cv.getClass() == ConfigValue.class && spec.getDefault() instanceof Integer -> createIntegerValue(key, valueSpec, () -> (Integer) cv.getRaw(), cv::set);
                                case ValueSpec spec when cv.getClass() == ConfigValue.class && spec.getDefault() instanceof Long -> createLongValue(key, valueSpec, () -> (Long) cv.getRaw(), cv::set);
                                case ValueSpec spec when cv.getClass() == ConfigValue.class && spec.getDefault() instanceof Double -> createDoubleValue(key, valueSpec, () -> (Double) cv.getRaw(), cv::set);
                                case ValueSpec spec when cv.getClass() == ConfigValue.class && spec.getDefault() instanceof Enum<?> -> createEnumValue(key, valueSpec, (Supplier) cv::getRaw, (Consumer) cv::set);
                                case null -> null;

                                default -> switch (cv) {
                                    case ModConfigSpec.BooleanValue value -> createBooleanValue(key, valueSpec, value::getRaw, value::set);
                                    case ModConfigSpec.IntValue value -> createIntegerValue(key, valueSpec, value::getRaw, value::set);
                                    case ModConfigSpec.LongValue value -> createLongValue(key, valueSpec, value::getRaw, value::set);
                                    case ModConfigSpec.DoubleValue value -> createDoubleValue(key, valueSpec, value::getRaw, value::set);
                                    case ModConfigSpec.EnumValue value -> createEnumValue(key, valueSpec, (Supplier) value::getRaw, (Consumer) value::set);
                                    default -> createOtherValue(key, cv);
                                };
                            };
                            elements.add(context.filter.filterEntry(context, key, element));
                        }
                        case UnmodifiableConfig subsection when context.valueSpecs.get(key) instanceof UnmodifiableConfig subconfig -> elements.add(createSection(key, subconfig, subsection));
                        default -> elements.add(context.filter.filterEntry(context, key, createOtherSection(key, rawValue)));
                    }
                }
                elements.addAll(createSyntheticValues());

                for (final Element element : elements) {
                    if (element != null) {
                        if (element.name() == null) {
                            list.addSmall(new StringWidget(Button.DEFAULT_WIDTH, Button.DEFAULT_HEIGHT, Component.empty(), font), element.getWidget(options));
                        } else {
                            final StringWidget label = new StringWidget(Button.DEFAULT_WIDTH, Button.DEFAULT_HEIGHT, element.name, font).alignLeft();
                            label.setTooltip(Tooltip.create(element.tooltip));
                            list.addSmall(label, element.getWidget(options));
                        }
                        hasUndoableElements |= element.undoable;
                    }
                }

                if (hasUndoableElements && undoButton == null) {
                    createUndoButton();
                    createResetButton();
                }
            }
            return this;
        }

        /**
         * Override this to add additional configuration elements to the list.
         * 
         * @return A collection of {@link Element}.
         */
        protected Collection<? extends Element> createSyntheticValues() {
            return Collections.emptyList();
        }

        protected boolean isNonDefault(ModConfigSpec.ConfigValue<?> cv) {
            return !Objects.equals(cv.getRaw(), cv.getDefault());
        }

        protected boolean isAnyNondefault() {
            for (final Entry entry : context.entries) {
                if (entry.getRawValue() instanceof final ModConfigSpec.ConfigValue<?> cv) {
                    if (!(getValueSpec(entry.getKey()) instanceof ListValueSpec) && isNonDefault(cv)) {
                        return true;
                    }
                }
            }
            return false;
        }

        @Nullable
        protected Element createStringValue(final String key, final Predicate<String> tester, final Supplier<String> source, final Consumer<String> target) {
            if (source.get().length() > 192) {
                // That's just too much for the UI
                final class_7842 label = new class_7842(class_4185.field_39500, class_4185.field_39501, class_2561.method_43470(source.get().substring(0, 128)), field_22793).method_48596();
                label.method_47400(class_7919.method_47407(LONG_STRING));
                return new Element(getTranslationComponent(key), getTooltipComponent(key, null), label, false);
            }
            final class_342 box = new class_342(field_22793, class_4185.field_39500, class_4185.field_39501, getTranslationComponent(key));
            box.method_1888(true);
            // no filter or the user wouldn't be able to type
            box.method_47400(class_7919.method_47407(getTooltipComponent(key, null)));
            box.method_1880(class_3532.method_15340(source.get().length() + 5, 128, 192));
            box.method_1852(source.get());
            box.method_1863(newValue -> {
                if (newValue != null && tester.test(newValue)) {
                    if (!newValue.equals(source.get())) {
                        undoManager.add(v -> {
                            target.accept(v);
                            onChanged(key);
                        }, newValue, v -> {
                            target.accept(v);
                            onChanged(key);
                        }, source.get());
                    }
                    box.method_1868(class_342.field_32196);
                    return;
                }
                box.method_1868(0xFFFF0000);
            });
            return new Element(getTranslationComponent(key), getTooltipComponent(key, null), box);
        }

        /**
         * Called when an entry is encountered that is neither a {@link ModConfigSpec.ConfigValue} nor a section.
         * Override this to produce whatever UI elements are appropriate for this object.<p>
         * 
         * Note that this case is unusual and shouldn't happen unless someone injected something into the config system.
         * 
         * @param key   The key of the entry.
         * @param value The entry itself.
         * @return null if no UI element should be added or an {@link Element} to be added to the UI.
         */
        @Nullable
        protected Element createOtherSection(final String key, final Object value) {
            return null;
        }

        /**
         * Called when a {@link ModConfigSpec.ConfigValue} is found that has an unknown data type.
         * Override this to produce whatever UI elements are appropriate for this object.<p>
         * 
         * @param key   The key of the entry.
         * @param value The entry itself.
         * @return null if no UI element should be added or an {@link Element} to be added to the UI.
         */
        @Nullable
        protected Element createOtherValue(final String key, final ConfigValue<?> value) {
            final class_7842 label = new class_7842(class_4185.field_39500, class_4185.field_39501, class_2561.method_43470(Objects.toString(value.getRaw())), field_22793).method_48596();
            label.method_47400(class_7919.method_47407(UNSUPPORTED_ELEMENT));
            return new Element(getTranslationComponent(key), getTooltipComponent(key, null), label, false);
        }

        /**
         * A custom variant of OptionsInstance.Enum that doesn't show the key on the button, just the value
         */
        public record Custom<T>(List<T> values) implements class_7172.class_7178<T> {

            @Override
            public Function<class_7172<T>, class_339> method_41756(class_7172.class_7277<T> tooltip, class_315 options, int x, int y, int width, Consumer<T> target) {
                return optionsInstance -> class_5676.method_32606(optionsInstance.field_37864)
                        .method_42729(class_5676.class_5680.method_32627(this.values))
                        .method_32618(tooltip)
                        .method_32616()
                        .method_32619(optionsInstance.method_41753())
                        .method_32617(x, y, width, 20, optionsInstance.field_38280, (source, newValue) -> {
                            optionsInstance.method_41748(newValue);
                            options.method_1640();
                            target.accept(newValue);
                        });
            }

            @Override
            public Optional<T> method_41758(T value) {
                return values.contains(value) ? Optional.of(value) : Optional.empty();
            }

            @Override
            public Codec<T> comp_675() {
                return null;
            }
            public static final Custom<Boolean> BOOLEAN_VALUES_NO_PREFIX = new Custom<>(ImmutableList.of(Boolean.TRUE, Boolean.FALSE));
        }

        @Nullable
        protected Element createBooleanValue(final String key, final ValueSpec spec, final Supplier<Boolean> source, final Consumer<Boolean> target) {
            return new Element(getTranslationComponent(key), getTooltipComponent(key, null),
                    new class_7172<>(getTranslationKey(key), getTooltip(key, null), class_7172.field_41333, Custom.BOOLEAN_VALUES_NO_PREFIX, source.get(), newValue -> {
                        // regarding change detection: new value always is different (cycle button)
                        undoManager.add(v -> {
                            target.accept(v);
                            onChanged(key);
                        }, newValue, v -> {
                            target.accept(v);
                            onChanged(key);
                        }, source.get());
                    }));
        }

        @Nullable
        protected <T extends Enum<T>> Element createEnumValue(final String key, final ValueSpec spec, final Supplier<T> source, final Consumer<T> target) {
            @SuppressWarnings("unchecked")
            final Class<T> clazz = (Class<T>) spec.getClazz();
            assert clazz != null;

            final List<T> list = Arrays.stream(clazz.getEnumConstants()).filter(spec::test).toList();

            return new Element(getTranslationComponent(key), getTooltipComponent(key, null),
                    new class_7172<>(getTranslationKey(key), getTooltip(key, null), (caption, displayvalue) -> displayvalue instanceof TranslatableEnum tenum ? tenum.getTranslatedName() : class_2561.method_43470(displayvalue.name()),
                            new Custom<>(list), source.get(), newValue -> {
                                // regarding change detection: new value always is different (cycle button)
                                undoManager.add(v -> {
                                    target.accept(v);
                                    onChanged(key);
                                }, newValue, v -> {
                                    target.accept(v);
                                    onChanged(key);
                                }, source.get());
                            }));
        }

        @Nullable
        protected Element createIntegerValue(final String key, final ValueSpec spec, final Supplier<Integer> source, final Consumer<Integer> target) {
            final Range<Integer> range = spec.getRange();
            final int min = range != null ? range.getMin() : 0;
            final int max = range != null ? range.getMax() : Integer.MAX_VALUE;

            if ((long) max - (long) min < MAX_SLIDER_SIZE) {
                return createSlider(key, source, target, range);
            } else {
                return createNumberBox(key, spec, source, target, null, Integer::decode, 0);
            }
        }

        @Nullable
        protected Element createSlider(final String key, final Supplier<Integer> source, final Consumer<Integer> target, final @Nullable Range<Integer> range) {
            return new Element(getTranslationComponent(key), getTooltipComponent(key, null),
                    new class_7172<>(getTranslationKey(key), getTooltip(key, range),
                            (caption, displayvalue) -> class_2561.method_43470("" + displayvalue), new class_7172.class_7174(range != null ? range.getMin() : 0, range != null ? range.getMax() : Integer.MAX_VALUE),
                            null, source.get(), newValue -> {
                                if (!newValue.equals(source.get())) {
                                    undoManager.add(v -> {
                                        target.accept(v);
                                        onChanged(key);
                                    }, newValue, v -> {
                                        target.accept(v);
                                        onChanged(key);
                                    }, source.get());
                                }
                            }));
        }

        @Nullable
        protected Element createLongValue(final String key, final ValueSpec spec, final Supplier<Long> source, final Consumer<Long> target) {
            return createNumberBox(key, spec, source, target, null, Long::decode, 0L);
        }

        // if someone knows how to get a proper zero inside...
        @Nullable
        protected <T extends Number & Comparable<? super T>> Element createNumberBox(final String key, final ValueSpec spec, final Supplier<T> source,
                final Consumer<T> target, @Nullable final Predicate<T> tester, final Function<String, T> parser, final T zero) {
            final Range<T> range = spec.getRange();

            final class_342 box = new class_342(field_22793, class_4185.field_39500, class_4185.field_39501, getTranslationComponent(key));
            box.method_1888(true);
            box.method_1890(newValueString -> {
                try {
                    parser.apply(newValueString);
                    return true;
                } catch (final NumberFormatException e) {
                    return isPartialNumber(newValueString, (range == null || range.getMin().compareTo(zero) < 0));
                }
            });
            box.method_47400(class_7919.method_47407(getTooltipComponent(key, range)));
            box.method_1852(source.get() + "");
            box.method_1863(newValueString -> {
                try {
                    final T newValue = parser.apply(newValueString);
                    if (tester != null ? tester.test(newValue) : (newValue != null && (range == null || range.test(newValue)) && spec.test(newValue))) {
                        if (!newValue.equals(source.get())) {
                            undoManager.add(v -> {
                                target.accept(v);
                                onChanged(key);
                            }, newValue, v -> {
                                target.accept(v);
                                onChanged(key);
                            }, source.get());
                        }
                        box.method_1868(class_342.field_32196);
                        return;
                    }
                } catch (final NumberFormatException e) {
                    // field probably is just empty/partial, ignore that
                }
                box.method_1868(0xFFFF0000);
            });
            return new Element(getTranslationComponent(key), getTooltipComponent(key, null), box);
        }

        protected boolean isPartialNumber(String value, boolean allowNegative) {
            return switch (value) {
                case "", "0", "0x", "0X" -> true;
                case "#" -> true; // not valid for doubles, but not worth making a special case
                case "-", "-0", "-0x", "-0X" -> allowNegative;
                // case "-#" -> allowNegative; // Java allows this, but no thanks, that's just cursed.
                // doubles can also do NaN, inf, and 0e0. Again, not worth making a special case for those, I say.
                default -> false;
            };
        }

        @Nullable
        protected Element createDoubleValue(final String key, final ValueSpec spec, final Supplier<Double> source, final Consumer<Double> target) {
            return createNumberBox(key, spec, source, target, null, Double::parseDouble, 0.0);
        }

        @Nullable
        protected Element createSection(final String key, final UnmodifiableConfig subconfig, final UnmodifiableConfig subsection) {
            if (subconfig.isEmpty()) return null;
            return new Element(class_2561.method_43469(SECTION, getTranslationComponent(key)), getTooltipComponent(key, null),
                    class_4185.method_46430(class_2561.method_43469(SECTION, class_2561.method_43471(translationChecker.check(getTranslationKey(key) + ".button", SECTION_TEXT))),
                            button -> field_22787.method_1507(sectionCache.computeIfAbsent(key,
                                    k -> new ConfigurationSectionScreen(context, this, subconfig.valueMap(), key, subsection.entrySet(), class_2561.method_43471(getTranslationKey(key))).rebuild())))
                            .method_46436(class_7919.method_47407(getTooltipComponent(key, null)))
                            .method_46432(class_4185.field_39500)
                            .method_46431(),
                    false);
        }

        @Nullable
        protected <T> Element createList(final String key, final ListValueSpec spec, final ModConfigSpec.ConfigValue<List<T>> list) {
            return new Element(class_2561.method_43469(SECTION, getTranslationComponent(key)), getTooltipComponent(key, null),
                    class_4185.method_46430(class_2561.method_43469(SECTION, class_2561.method_43471(translationChecker.check(getTranslationKey(key) + ".button", SECTION_TEXT))),
                            button -> field_22787.method_1507(sectionCache.computeIfAbsent(key,
                                    k -> new ConfigurationListScreen<>(Context.list(context, this), key, class_2561.method_43469(CRUMB, this.method_25440(), CRUMB_SEPARATOR, getTranslationComponent(key)), spec, list)).rebuild()))
                            .method_46436(class_7919.method_47407(getTooltipComponent(key, null))).method_46431(),
                    false);
        }

        @Override
        public void method_25394(class_332 graphics, int p_281550_, int p_282878_, float p_282465_) {
            setUndoButtonstate(undoManager.canUndo()); // in render()? Really? --- Yes! This is how vanilla does it.
            setResetButtonstate(isAnyNondefault());
            super.method_25394(graphics, p_281550_, p_282878_, p_282465_);
        }

        @Override
        protected void method_31387() {
            if (undoButton != null || resetButton != null) {
                class_8667 linearlayout = field_49503.method_48996(class_8667.method_52742().method_52735(8));
                if (undoButton != null) {
                    linearlayout.method_52736(undoButton);
                }
                if (resetButton != null) {
                    linearlayout.method_52736(resetButton);
                }
                linearlayout.method_52736(doneButton);
            } else {
                super.method_31387();
            }
        }

        protected void createUndoButton() {
            undoButton = class_4185.method_46430(UNDO, button -> {
                undoManager.undo();
                rebuild();
            }).method_46436(class_7919.method_47407(UNDO_TOOLTIP)).method_46432(class_4185.field_39499).method_46431();
            undoButton.field_22763 = false;
        }

        protected void setUndoButtonstate(boolean state) {
            if (undoButton != null) {
                undoButton.field_22763 = state;
            }
        }

        @SuppressWarnings({ "unchecked", "rawtypes" })
        protected void createResetButton() {
            resetButton = class_4185.method_46430(RESET, button -> {
                List<UndoManager.Step<?>> list = new ArrayList<>();
                for (final Entry entry : context.entries) {
                    if (entry.getRawValue() instanceof final ModConfigSpec.ConfigValue cv && !(getValueSpec(entry.getKey()) instanceof ListValueSpec) && isNonDefault(cv)) {
                        final String key = entry.getKey();
                        list.add(undoManager.step(v -> {
                            cv.set(v);
                            onChanged(key);
                        }, getValueSpec(key).correct(null), v -> {
                            cv.set(v);
                            onChanged(key);
                        }, cv.getRaw()));
                    }
                }
                undoManager.add(list);
                rebuild();
            }).method_46436(class_7919.method_47407(RESET_TOOLTIP)).method_46432(class_4185.field_39499).method_46431();
        }

        protected void setResetButtonstate(boolean state) {
            if (resetButton != null) {
                resetButton.field_22763 = state;
            }
        }

        @Override
        public void method_25419() {
            if (changed) {
                if (field_21335 instanceof final ConfigurationSectionScreen parent) {
                    // "bubble up" the marker so the top-most section can change the ModConfig
                    parent.changed = true;
                } else {
                    // we are a top-level per-type config screen, i.e. one specific config file. Save the config and tell the mod to reload.
                    context.modSpec.save();
                }
                // the restart flag only matters when there were actual changes
                if (field_21335 instanceof final ConfigurationSectionScreen parent) {
                    parent.needsRestart = parent.needsRestart.with(needsRestart);
                } else if (field_21335 instanceof final ConfigScreen parent) {
                    parent.needsRestart = parent.needsRestart.with(needsRestart);
                }
            }
            super.method_25419();
        }
    }


    public static class ConfigurationListScreen<T> extends ConfigurationSectionScreen {
        protected final String key;
        protected final ListValueSpec spec;

        // the original data
        protected final ModConfigSpec.ConfigValue<List<T>> valueList;
        // the copy of the data we are working on
        protected List<T> cfgList;

        public ConfigurationListScreen(final Context context, final String key, final class_2561 title, final ListValueSpec spec,
                final ModConfigSpec.ConfigValue<List<T>> valueList) {
            super(context, title);
            this.key = key;
            this.spec = spec;
            this.valueList = valueList; // === (ListValueSpec)getValueSpec(key)
            this.cfgList = new ArrayList<>(valueList.getRaw());
        }

        @Override
        protected ConfigurationSectionScreen rebuild() {
            if (list != null) { // this may be called early, skip and wait for init() then
                list.children().clear();

                for (int idx = 0; idx < cfgList.size(); idx++) {
                    var entry = cfgList.get(idx);
                    var element = switch (entry) {
                        case null -> null;
                        case final Boolean value -> createBooleanListValue(idx, value);
                        case final Integer value -> createIntegerListValue(idx, value);
                        case final Long value -> createLongListValue(idx, value);
                        case final Double value -> createDoubleListValue(idx, value);
                        case final String value -> createStringListValue(idx, value);
                        default -> createOtherValue(idx, entry);
                    };

                    if (element != null) {
                        final AbstractWidget widget = element.getWidget(options);
                        if (widget instanceof EditBox box) {
                            // Force our responder to check content and set text colour.
                            // This is only needed on lists, as section cannot have new UI elements added with bad data.
                            // Here, this can happen when a new element is added to the list.
                            // As the new value is the old value, no undo record will be created.
                            box.setValue(box.getValue());
                        }
                        list.addSmall(createListLabel(idx), widget);
                    }
                }

                createAddElementButton();
                if (undoButton == null) {
                    createUndoButton();
                    createResetButton();
                }
            }
            return this;
        }

        protected boolean isAnyNondefault() {
            return !cfgList.equals(valueList.getDefault());
        }

        /**
         * Creates a button to add a new element to the end of the list and adds it to the UI.<p>
         * 
         * Override this if you want a different button or want to add more elements.
         */
        @SuppressWarnings("unchecked")
        protected void createAddElementButton() {
            final Supplier<?> newElement = spec.getNewElementSupplier();
            final Range<Integer> sizeRange = spec.getSizeRange();

            if (newElement != null && sizeRange.test(cfgList.size() + 1)) {
                field_51824.method_20407(new class_7842(class_4185.field_39500, class_4185.field_39501, class_2561.method_43473(), field_22793), class_4185.method_46430(NEW_LIST_ELEMENT, button -> {
                    List<T> newValue = new ArrayList<>(cfgList);
                    newValue.add((T) newElement.get());
                    undoManager.add(v -> {
                        cfgList = v;
                        onChanged(key);
                    }, newValue, v -> {
                        cfgList = v;
                        onChanged(key);
                    }, cfgList);
                    rebuild();
                }).method_46431());
            }
        }

        /**
         * Creates a new widget to label a list value and provide manipulation buttons for it.<p>
         * 
         * Override this if you want different labels/buttons.
         * 
         * @param idx The index into the list.
         * @return An {@link class_339} to be rendered in the left column of the options screen
         */
        protected class_339 createListLabel(int idx) {
            return new ListLabelWidget(0, 0, class_4185.field_39500, class_4185.field_39501, class_2561.method_43469(LIST_ELEMENT, idx), idx);
        }

        /**
         * Called when a list element is found that has an unknown or unsupported data type. Override this to produce whatever
         * UI elements are appropriate for this object.<p>
         * 
         * Note that all types of elements that can be read from the config file as part of a list are already supported. You
         * only need this if you manipulate the contents of the list after it has been loaded.<p>
         * 
         * If this returns null, no row will be shown on the screen, but the up/down buttons will still see your element.
         * Which means that the user will see no change when moving another element over the hidden line. Consider returning
         * a {@link class_7842} as a placeholder instead.<p>
         * 
         * Do <em>not</em> capture {@link #cfgList} here or in another create*Value() method. The undo/reset system will
         * replace the list, so you need to always access the field. You can (and should) capture the index.
         * 
         * @param idx   The index into the list.
         * @param entry The entry itself.
         * @return null if this element should be skipped or an {@link Element} to be added to the UI.
         */
        @Nullable
        protected Element createOtherValue(final int idx, final T entry) {
            final class_7842 label = new class_7842(class_4185.field_39500, class_4185.field_39501, class_2561.method_43470(Objects.toString(entry)), field_22793).method_48596();
            label.method_47400(class_7919.method_47407(UNSUPPORTED_ELEMENT));
            return new Element(getTranslationComponent(key), getTooltipComponent(key, null), label, false);
        }

        @SuppressWarnings("unchecked")
        @Nullable
        protected Element createStringListValue(final int idx, final String value) {
            return createStringValue(key, v -> spec.test(List.of(v)), () -> value, newValue -> cfgList.set(idx, (T) newValue));
        }

        @SuppressWarnings("unchecked")
        @Nullable
        protected Element createDoubleListValue(final int idx, final Double value) {
            return createNumberBox(key, spec, () -> value, newValue -> cfgList.set(idx, (T) newValue), v -> spec.test(List.of(v)), Double::parseDouble, 0.0);
        }

        @SuppressWarnings("unchecked")
        @Nullable
        protected Element createLongListValue(final int idx, final Long value) {
            return createNumberBox(key, spec, () -> value, newValue -> cfgList.set(idx, (T) newValue), v -> spec.test(List.of(v)), Long::decode, 0L);
        }

        @SuppressWarnings("unchecked")
        @Nullable
        protected Element createIntegerListValue(final int idx, final Integer value) {
            return createNumberBox(key, spec, () -> value, newValue -> cfgList.set(idx, (T) newValue), v -> spec.test(List.of(v)), Integer::decode, 0);
        }

        @SuppressWarnings("unchecked")
        @Nullable
        protected Element createBooleanListValue(final int idx, final Boolean value) {
            return createBooleanValue(key, spec, () -> value, newValue -> cfgList.set(idx, (T) newValue));
        }

        /**
         * Swap the given element with the next one. Should be called by the list label widget to manipulate the list.
         */
        protected boolean swap(final int idx, final boolean simulate) {
            final List<T> values = new ArrayList<>(cfgList);
            values.add(idx, values.remove(idx + 1));
            return addUndoListener(simulate, values);
        }

        /**
         * Remove the given element. Should be called by the list label widget to manipulate the list.
         */
        protected boolean del(final int idx, final boolean simulate) {
            final List<T> values = new ArrayList<>(cfgList);
            values.remove(idx);
            return addUndoListener(simulate, values);
        }

        private boolean addUndoListener(boolean simulate, List<T> values) {
            final boolean valid = spec.test(values);
            if (!simulate && valid) {
                undoManager.add(v -> {
                    cfgList = v;
                    onChanged(key);
                }, values, v -> {
                    cfgList = v;
                    onChanged(key);
                }, cfgList);
                rebuild();
            }
            return valid;
        }

        @Override
        public void method_25419() {
            if (changed && spec.test(cfgList)) {
                valueList.set(cfgList);
                if (context.parent instanceof ConfigurationSectionScreen parent) {
                    parent.onChanged(key);
                }
            }
            super.method_25419();
        }

        @Override
        public void method_25394(class_332 graphics, int p_281550_, int p_282878_, float p_282465_) {
            doneButton.field_22763 = spec.test(cfgList);
            super.method_25394(graphics, p_281550_, p_282878_, p_282465_);
        }

        protected void onChanged(final String key) {
            changed = true;
            // parent's onChanged() will be fired when we actually assign the changed list. For now,
            // we've only changed our working copy.
        }

        @SuppressWarnings("unchecked")
        protected void createResetButton() {
            resetButton = class_4185.method_46430(RESET, button -> {
                undoManager.add(
                        v -> {
                            cfgList = v;
                            onChanged(key);
                        }, new ArrayList<>((List<T>) getValueSpec(key).correct(null)),
                        v -> {
                            cfgList = v;
                            onChanged(key);
                        }, cfgList);
                rebuild();
            }).method_46436(class_7919.method_47407(RESET_TOOLTIP)).method_46432(class_4185.field_39499).method_46431();
        }

        /**
         * A widget to be used as a label in a list of configuration values.<p>
         * 
         * It includes buttons for "move element up", "move element down", and "delete element" as well as a label.
         * 
         */
        public class ListLabelWidget extends class_9017 {
            protected final class_4185 upButton = class_4185.method_46430(MOVE_LIST_ELEMENT_UP, this::up).method_46431();
            protected final class_4185 downButton = class_4185.method_46430(MOVE_LIST_ELEMENT_DOWN, this::down).method_46431();
            protected final class_4185 delButton = class_4185.method_46430(REMOVE_LIST_ELEMENT, this::rem).method_46431();
            protected final class_7842 label = new class_7842(0, 0, 0, 0, class_2561.method_43473(), field_22793).method_48596();
            protected final int idx;
            protected final boolean isFirst;
            protected final boolean isLast;

            public ListLabelWidget(final int x, final int y, final int width, final int height, final class_2561 labelText, final int idx) {
                super(x, y, width, height, labelText);
                this.idx = idx;
                this.isFirst = idx == 0;
                this.isLast = idx + 1 == cfgList.size();
                label.method_25355(labelText);
                checkButtons();
                updateLayout();
            }

            @Override
            public void method_46421(final int pX) {
                super.method_46421(pX);
                updateLayout();
            }

            @Override
            public void method_46419(final int pY) {
                super.method_46419(pY);
                updateLayout();
            }

            @Override
            public void method_53533(final int pHeight) {
                super.method_53533(pHeight);
                updateLayout();
            }

            @Override
            public void method_25358(int pWidth) {
                super.method_25358(pWidth);
                updateLayout();
            }

            @Override
            public void method_55445(int pWidth, int pHeight) {
                super.method_55445(pWidth, pHeight);
                updateLayout();
            }

            protected void updateLayout() {
                upButton.method_46421(method_46426());
                downButton.method_46421(method_46426() + method_25364() + 2);
                delButton.method_46421(method_46426() + method_25368() - method_25364());
                label.method_46421(method_46426() + 2 * 22);

                upButton.method_46419(method_46427());
                downButton.method_46419(method_46427());
                delButton.method_46419(method_46427());
                label.method_46419(method_46427());

                upButton.method_53533(method_25364());
                downButton.method_53533(method_25364());
                delButton.method_53533(method_25364());
                label.method_53533(method_25364());

                upButton.method_25358(method_25364());
                downButton.method_25358(method_25364());
                delButton.method_25358(method_25364());
                label.method_25358(method_25368() - 3 * (method_25364() + 2));
            }

            void up(final class_4185 button) {
                swap(idx - 1, false);
            }

            void down(final class_4185 button) {
                swap(idx, false);
            }

            void rem(final class_4185 button) {
                del(idx, false);
            }

            @Override
            public List<? extends class_364> method_25396() {
                return List.of(upButton, label, downButton, delButton);
            }

            @Override
            protected void method_48579(final class_332 pGuiGraphics, final int pMouseX, final int pMouseY, final float pPartialTick) {
                checkButtons();
                label.method_25394(pGuiGraphics, pMouseX, pMouseY, pPartialTick);
                if (!isFirst) {
                    upButton.method_25394(pGuiGraphics, pMouseX, pMouseY, pPartialTick);
                }
                if (!isLast) {
                    downButton.method_25394(pGuiGraphics, pMouseX, pMouseY, pPartialTick);
                }
                delButton.method_25394(pGuiGraphics, pMouseX, pMouseY, pPartialTick);
            }

            protected void checkButtons() {
                upButton.field_22764 = !isFirst;
                upButton.field_22763 = !isFirst && swap(idx - 1, true);
                downButton.field_22764 = !isLast;
                downButton.field_22763 = !isLast && swap(idx, true);
                Range<Integer> sizeRange = spec.getSizeRange();
                delButton.field_22763 = !cfgList.isEmpty() && (sizeRange == null || sizeRange.test(cfgList.size() - 1)) && del(idx, true);
            }

            @Override
            protected void method_47399(final class_6382 pNarrationElementOutput) {
                // TODO I have no idea. Help?
            }
        }
    }

    /**
     * A class representing an undo/redo buffer.<p>
     * 
     * Every undo step is represented as 2 actions, one to initially execute when the step is added and
     * to redo after an undo, and one to execute to undo the step. Both get a captured parameter to make
     * defining them inline or reusing the code portion easier.
     */
    public static final class UndoManager {
        public record Step<T>(Consumer<T> run, T newValue, Consumer<T> undo, T oldValue) {
            private void runUndo() {
                undo.accept(oldValue);
            }

            private void runRedo() {
                run.accept(newValue);
            }
        };

        private final List<Step<?>> undos = new ArrayList<>();
        private final List<Step<?>> redos = new ArrayList<>();

        public void undo() {
            if (canUndo()) {
                Step<?> step = undos.removeLast();
                step.runUndo();
                redos.add(step);
            }
        }

        public void redo() {
            if (canRedo()) {
                Step<?> step = redos.removeLast();
                step.runRedo();
                undos.add(step);
            }
        }

        private void add(Step<?> step, boolean execute) {
            undos.add(step);
            redos.clear();
            if (execute) {
                step.runRedo();
            }
        }

        public <T> Step<T> step(Consumer<T> run, T newValue, Consumer<T> undo, T oldValue) {
            return new Step<>(run, newValue, undo, oldValue);
        }

        public <T> void add(Consumer<T> run, T newValue, Consumer<T> undo, T oldValue) {
            add(step(run, newValue, undo, oldValue), true);
        }

        public <T> void addNoExecute(Consumer<T> run, T newValue, Consumer<T> undo, T oldValue) {
            add(step(run, newValue, undo, oldValue), false);
        }

        public void add(Step<?>... steps) {
            add(ImmutableList.copyOf(steps));
        }

        public void add(final List<Step<?>> steps) {
            add(new Step<>(n -> steps.forEach(Step::runRedo), null, n -> steps.forEach(Step::runUndo), null), true);
        }

        public boolean canUndo() {
            return !undos.isEmpty();
        }

        public boolean canRedo() {
            return !redos.isEmpty();
        }
    }
}
