package dev.kikugie.techutils.feature.containerscan.scanners;

import com.google.common.collect.Iterables;
import dev.kikugie.techutils.feature.containerscan.LinkedStorageEntry;
import dev.kikugie.techutils.feature.containerscan.PlacementContainerAccess;
import dev.kikugie.techutils.feature.containerscan.handlers.InteractionHandler;
import dev.kikugie.techutils.feature.containerscan.screens.BlockingScreenHandler;
import dev.kikugie.techutils.render.outline.OutlineRenderer;
import dev.kikugie.techutils.util.ContainerUtils;
import dev.kikugie.techutils.util.LocalPlacementPos;
import dev.kikugie.techutils.util.ValidBox;
import fi.dy.masa.litematica.schematic.LitematicaSchematic;
import fi.dy.masa.litematica.schematic.container.LitematicaBlockStateContainer;
import fi.dy.masa.litematica.schematic.placement.SchematicPlacement;
import fi.dy.masa.malilib.util.Color4f;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import net.minecraft.class_1263;
import net.minecraft.class_1268;
import net.minecraft.class_1277;
import net.minecraft.class_1657;
import net.minecraft.class_2281;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_243;
import net.minecraft.class_2480;
import net.minecraft.class_2487;
import net.minecraft.class_2680;
import net.minecraft.class_2745;
import net.minecraft.class_310;
import net.minecraft.class_3936;
import net.minecraft.class_3965;
import net.minecraft.class_437;
import net.minecraft.class_636;
import net.minecraft.class_638;
import net.minecraft.class_746;

/**
 * Records contents of nearby containers by opening them manually via the player entity. This method is the most inefficient due to relying on ordered packets and player limitations, but doesn't have any additional requirements.
 */
public class InteractionScanner implements Scanner {
	private final Map<class_2338, LinkedStorageEntry> cache = new Hashtable<>();
	/* -Util- */
	private final class_310 client = class_310.method_1551();
	private final class_746 player = Objects.requireNonNull(this.client.field_1724);
	private final class_638 world = Objects.requireNonNull(this.client.field_1687);
	private final class_636 interactionManager = Objects.requireNonNull(this.client.field_1761);
	/* ------ */

	/**
	 * Used Litematica placement. If not null, will limit scanned containers only to ones matching in the placement and will record it's items in {@link LinkedStorageEntry}.
	 */
	@Nullable
	private final SchematicPlacement placement;
	/**
	 * Only used if {@link InteractionScanner#placement} isn't null. Represents all containers to be scanned.
	 */
	private final Set<class_2338> waiting = new HashSet<>();
	private boolean running = true;

	public InteractionScanner() {
		this.placement = null;
	}

	public InteractionScanner(@Nullable SchematicPlacement placement) {
		this.placement = placement;
		initPlacementBlocks();
	}

	/**
	 * Called upon modifying the active placement. Resets {@link InteractionScanner#cache}, {@link InteractionScanner#waiting} lists and writes new positions for placement containers.
	 * <br>
	 * Generally, don't move your placement while its being scanned.
	 */
	private void initPlacementBlocks() {
		if (this.placement == null)
			return;
		this.waiting.clear();
		this.cache.clear();
		LitematicaSchematic schematic = this.placement.getSchematic();
		for (String region : schematic.getAreas().keySet()) {
			LitematicaBlockStateContainer container = schematic.getSubRegionContainer(region);
			Map<class_2338, class_2487> blockEntities = schematic.getBlockEntityMapForRegion(region);
			if (blockEntities == null)
				continue;

			assert container != null;
			for (class_2338 pos : blockEntities.keySet()) {
				class_2338 worldPos = LocalPlacementPos.getWorldPos(pos, region, this.placement);

				class_2680 worldState = this.world.method_8320(worldPos);
				class_2680 schemState = container.get(pos.method_10263(), pos.method_10264(), pos.method_10260());
				if (worldState.equals(schemState))
					this.waiting.add(worldPos);
			}
		}
	}

	/**
	 * @return an unordered set of available containers within player's reach.
	 */
	private Set<class_2338> getNearbyContainers() {
		class_243 camera = getEyesPos(this.player);
		double reach = this.player.method_55754();
		List<class_2338> positions = this.placement != null ? getPlacementContainers(camera, reach) : getWorldContainers(camera, reach);
		Set<class_2338> result = new HashSet<>();
		for (class_2338 pos : positions)
			validatePos(pos, camera).ifPresent(result::add);
		return result;
	}

	/**
	 * @return All containers in player's reach
	 */
	private List<class_2338> getWorldContainers(class_243 camera, double reach) {
		class_2338 corner1 = class_2338.method_49638(camera.method_1031(reach, reach, reach));
		class_2338 corner2 = class_2338.method_49638(camera.method_1023(reach, reach, reach));
		return getAvailable(camera, reach, class_2338.method_10097(corner1, corner2));
	}

	/**
	 * @return Containers matching entries in {@link InteractionScanner#waiting} list within player's reach
	 */
	private List<class_2338> getPlacementContainers(class_243 camera, double reach) {
		class_2338 corner1 = class_2338.method_49638(camera.method_1031(reach, reach, reach));
		class_2338 corner2 = class_2338.method_49638(camera.method_1023(reach, reach, reach));
		ValidBox box = new ValidBox(corner1, corner2);
		return getAvailable(camera, reach, Iterables.filter(this.waiting, box::contains));
	}

	private List<class_2338> getAvailable(class_243 camera, double reach, Iterable<class_2338> positions) {
		List<class_2338> result = new ArrayList<>();
		for (class_2338 pos : positions) {
			if (isInReach(pos, camera, reach))
				result.add(new class_2338(pos));
		}
		return result;
	}

	/**
	 * Checks if player is able to access a container. Conditions can be summarised as:<br>
	 * <pre>
	 *     - position has a container;
	 *     - {@link InteractionScanner#cache} doesn't contain the position;
	 *     - player must be able to open the container.</pre>
	 *
	 * @return {@link class_2338} if input is valid, {@link Optional#empty()} otherwise
	 */
	private Optional<class_2338> validatePos(class_2338 pos, class_243 camera) {
		class_2680 state = this.world.method_8320(pos);
		if (state.method_26215() || this.cache.containsKey(pos) || ContainerUtils.validateContainer(pos, state).isEmpty())
			return Optional.empty();

		if (state.method_26204() instanceof class_2281) {
			if (!this.player.method_7325() && !ContainerUtils.isChestAccessible(this.world, pos, state))
				return Optional.empty();

			class_2745 type = state.method_11654(class_2281.field_10770);
			if (type == class_2745.field_12569)
				return Optional.of(pos);

			class_2338 adjacent = pos.method_10081(class_2281.method_9758(state).method_10163());
			return Optional.of(pos.method_19770(camera) < adjacent.method_19770(camera)
				? pos : adjacent);
		}
		if (state.method_26204() instanceof class_2480) {
			return this.player.method_7325() || ContainerUtils.isShulkerBoxAccessible(this.world, pos, state)
				? Optional.of(pos) : Optional.empty();
		}
		return Optional.of(pos);
	}

	private boolean isInReach(class_2338 pos, class_243 camera, double reach) {
		double dx = camera.method_10216() - ((double) pos.method_10263() + 0.5D);
		double dy = camera.method_10214() - ((double) pos.method_10264() + 0.5D);
		double dz = camera.method_10215() - ((double) pos.method_10260() + 0.5D);
		return dx * dx + dy * dy + dz * dz <= reach * reach;
	}

	private class_243 getEyesPos(class_1657 player) {
		return new class_243(player.method_23317(), player.method_23318() + player.method_18381(player.method_18376()), player.method_23321());
	}

	/**
	 * Registers a position in {@link InteractionHandler} if it's not already in the queue. Registered handler is configured to close the screen as soon as contents packet arrives. After registering the handler it interacts with the block via the player entity.
	 *
	 * @param pos  position to register
	 * @param tick time of register
	 * @see BlockingScreenHandler
	 */
	private void register(class_2338 pos, long tick) {
		if (InteractionHandler.contains(pos))
			return;

		class_2680 state = this.world.method_8320(pos);
		Optional<class_1263> inventory = ContainerUtils.validateContainer(pos, state);
		if (inventory.isEmpty())
			return;

		class_1277 worldInv = new class_1277(inventory.get().method_5439());
		LinkedStorageEntry entry = this.placement == null ? new LinkedStorageEntry(pos, worldInv, null) : PlacementContainerAccess.getEntry(pos, state, worldInv);
		this.cache.put(pos, entry);
		Optional<Color4f> color = entry.validate();
		Color4f red = new Color4f(1, 0, 0, 1);
		color.ifPresent(it -> OutlineRenderer.add(this.world, red.intValue, camera -> pos.method_19770(camera) <= 32 * 32, pos));
		this.waiting.remove(pos);
		InteractionHandler.add(new InteractionHandler(pos, tick) {
			@Override
			public boolean accept(class_437 screen) {
				class_746 player = Objects.requireNonNull(class_310.method_1551().field_1724);
				player.field_7512 = new BlockingScreenHandler(((class_3936<?>) screen).method_17577(), worldInv);
				return false;
			}
		});
		interact(pos, getEyesPos(this.player));
	}

	private void interact(class_2338 pos, class_243 player) {
		class_243 click = class_243.method_24954(pos).method_1031(0.5D, 0.5D, 0.5D);
		class_3965 hit = new class_3965(click, class_2350.method_10142(click.field_1352 - player.field_1352, click.field_1351 - player.field_1351, click.field_1350 - player.field_1350), pos, false);
		this.interactionManager.method_2896(this.client.field_1724, class_1268.field_5808, hit);
	}

	@Override
	public void tick() {
		if (!this.running || (this.placement != null && this.waiting.isEmpty()))
			return;

		long tick = this.world.method_8510();
		Set<class_2338> nearby = getNearbyContainers();
		for (class_2338 pos : nearby)
			register(pos, tick);
	}

	@Override
	public void start() {
		this.running = true;
	}

	@Override
	public void stop() {
		this.running = false;
	}

	@Override
	public void update() {
		initPlacementBlocks();
	}
}
