package dev.kikugie.techutils.feature.preview.model;

import com.google.common.collect.HashMultimap;
import com.mojang.blaze3d.systems.RenderSystem;
import dev.kikugie.techutils.Reference;
import dev.kikugie.techutils.mixin.preview.BlockEntityAccessor;
import dev.kikugie.techutils.util.ValidBox;
import fi.dy.masa.litematica.schematic.LitematicaSchematic;
import fi.dy.masa.litematica.util.EntityUtils;
import net.fabricmc.fabric.api.renderer.v1.RendererAccess;
import net.fabricmc.fabric.impl.client.indigo.renderer.IndigoRenderer;
import net.fabricmc.fabric.impl.client.indigo.renderer.render.WorldMesherRenderContext;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1297;
import net.minecraft.class_156;
import net.minecraft.class_1657;
import net.minecraft.class_1920;
import net.minecraft.class_1921;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2343;
import net.minecraft.class_238;
import net.minecraft.class_243;
import net.minecraft.class_2464;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_287;
import net.minecraft.class_290;
import net.minecraft.class_291;
import net.minecraft.class_293;
import net.minecraft.class_310;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4608;
import net.minecraft.class_4696;
import net.minecraft.class_5819;
import net.minecraft.class_750;
import net.minecraft.class_776;
import net.minecraft.class_8251;
import net.minecraft.client.render.*;
import org.apache.commons.lang3.function.TriFunction;
import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4f;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;

public class LitematicMesh {
	private static final Logger LOGGER = LoggerFactory.getLogger(LitematicMesh.class);

	// Render setup data
	private final LitematicaSchematic schematic;
	private final class_310 client;
	private final DummyWorld dummyWorld;
	private final class_2338 from, to;
	private final class_238 dimensions;

	private final boolean cull;

	private final TriFunction<class_1657, class_2338, class_2338, List<class_1297>> entitySupplier;
	private DynamicRenderInfo renderInfo = DynamicRenderInfo.EMPTY;

	// Build process data
	private MeshState state = MeshState.NEW;

	private float buildProgress = 0;
	private @Nullable CompletableFuture<Void> buildFuture = null;

	// Vertex storage
	private final Map<class_1921, class_291> bufferStorage = new HashMap<>();

	public LitematicMesh(LitematicaSchematic schematic) {
		this.schematic = schematic;
		this.client = class_310.method_1551();
		this.dummyWorld = DummyWorld.fromWorld(this.client.field_1687);
		final var fullBox = fullBox(schematic.getAreas().values());

		this.from = fullBox.getMin();
		this.to = fullBox.getMax();

		this.cull = true;
		this.dimensions = class_238.method_54784(this.from, this.to);
		this.entitySupplier = (playerEntity, blockPos, blockPos2) -> readEntities();

//		this.renderStartAction = renderStartAction;
//		this.renderEndAction = renderEndAction;

		this.scheduleRebuild();
	}

	private List<class_1297> readEntities() {
		ArrayList<class_1297> entities = new ArrayList<>();
		this.schematic.getAreas().keySet().forEach(region -> {
			List<LitematicaSchematic.EntityInfo> schematicEntities = this.schematic.getEntityListForRegion(region);
			var offset = this.schematic.getSubRegionPosition(region);
			assert schematicEntities != null;

			schematicEntities.forEach(entityInfo -> {
				var entity = EntityUtils.createEntityAndPassengersFromNBT(entityInfo.nbt, this.dummyWorld);
				entity.method_33574(
					entity.method_19538()
						.method_1031(Math.max(offset.method_10263(), 0), Math.max(offset.method_10264(), 0), Math.max(offset.method_10260(), 0)));
				entities.add(entity);
			});
		});
		return entities;
	}

	private ValidBox fullBox(Collection<fi.dy.masa.litematica.selection.Box> boxes) {
		int[] corners = {0, 0, 0, 0, 0, 0};
		for (fi.dy.masa.litematica.selection.Box box : boxes) {
			ValidBox validBox = ValidBox.of(box);
			class_2338 min = validBox.getMin();
			class_2338 max = validBox.getMax();

			corners[0] = Math.min(corners[0], min.method_10263());
			corners[1] = Math.min(corners[1], min.method_10264());
			corners[2] = Math.min(corners[2], min.method_10260());
			corners[3] = Math.max(corners[3], max.method_10263());
			corners[4] = Math.max(corners[4], max.method_10264());
			corners[5] = Math.max(corners[5], max.method_10260());
		}
		return new ValidBox(corners);
	}

	/**
	 * Renders this world mesh into the current framebuffer, translated using the given matrix
	 *
	 * @param matrices The translation matrices. This is applied to the entire mesh
	 */
	public void render(class_4587 matrices) {
		if (!this.canRender()) {
			throw new IllegalStateException("World mesh not prepared!");
		}

		var matrix = matrices.method_23760().method_23761();
		var translucent = class_1921.method_23583();

		this.bufferStorage.forEach((renderLayer, vertexBuffer) -> {
			if (renderLayer == translucent) return;
			this.drawBuffer(vertexBuffer, renderLayer, matrix);
		});

		if (this.bufferStorage.containsKey(translucent)) {
			this.drawBuffer(bufferStorage.get(translucent), translucent, matrix);
		}

		class_291.method_1354();
	}

	private void drawBuffer(class_291 vertexBuffer, class_1921 renderLayer, Matrix4f matrix) {
		renderLayer.method_23516();
//		renderStartAction.run();

		vertexBuffer.method_1353();
		vertexBuffer.method_34427(matrix, RenderSystem.getProjectionMatrix(), RenderSystem.getShader());

//		renderEndAction.run();
		renderLayer.method_23518();
	}

	/**
	 * Checks whether this mesh is ready for rendering
	 */
	public boolean canRender() {
		return this.state.canRender;
	}

	/**
	 * Returns the current state of this mesh, used to indicate building progress and rendering availability
	 *
	 * @return The current {@code MeshState} constant
	 */
	public MeshState state() {
		return this.state;
	}

	/**
	 * Renamed to {@link #state()}
	 */
	@Deprecated(forRemoval = true)
	public MeshState getState() {
		return this.state();
	}

	/**
	 * How much of this mesh is built
	 *
	 * @return The build progress of this mesh
	 */
	public float buildProgress() {
		return this.buildProgress;
	}

	/**
	 * Renamed to {@link #buildProgress()}
	 */
	@Deprecated(forRemoval = true)
	public float getBuildProgress() {
		return this.buildProgress();
	}

	/**
	 * @return An object describing the entities and block
	 * entities in the area this mesh is covering, with positions
	 * relative to the mesh
	 */
	public DynamicRenderInfo renderInfo() {
		return this.renderInfo;
	}

	/**
	 * Renamed to {@link #renderInfo()}
	 */
	@Deprecated(forRemoval = true)
	public DynamicRenderInfo getRenderInfo() {
		return this.renderInfo();
	}

	/**
	 * @return The origin position of this mesh's area
	 */
	public class_2338 startPos() {
		return this.from;
	}

	/**
	 * @return The end position of this mesh's area
	 */
	public class_2338 endPos() {
		return this.to;
	}

	/**
	 * @return The dimensions of this mesh's entire area
	 */
	public class_238 dimensions() {
		return dimensions;
	}

	/**
	 * Reset this mesh to {@link MeshState#NEW}, releasing
	 * all vertex buffers in the process
	 */
	public void reset() {
		this.bufferStorage.forEach((renderLayer, vertexBuffer) -> vertexBuffer.close());
		this.bufferStorage.clear();

		this.state = MeshState.NEW;
	}

	/**
	 * Renamed to {@link #reset()}
	 */
	@Deprecated(forRemoval = true)
	public void clear() {
		this.reset();
	}

	/**
	 * Schedule a rebuild of this mesh on
	 * the main worker executor
	 */
	public synchronized void scheduleRebuild() {
		this.scheduleRebuild(class_156.method_18349());
	}

	/**
	 * Schedule a rebuild of this mesh,
	 * on the supplied executor
	 *
	 * @return A future completing when the build process is finished,
	 * or {@code null} if this mesh is already building
	 */
	public synchronized CompletableFuture<Void> scheduleRebuild(Executor executor) {
		if (this.buildFuture != null) return this.buildFuture;

		this.buildProgress = 0;
		this.state = this.state != MeshState.NEW
			? MeshState.REBUILDING
			: MeshState.BUILDING;

		this.buildFuture = CompletableFuture.runAsync(this::build, executor).whenComplete((unused, throwable) -> {
			this.buildFuture = null;

			if (throwable == null) {
				state = MeshState.READY;
			} else {
				LOGGER.warn("World mesh building failed", throwable);
				state = MeshState.CORRUPT;
			}
		});

		return this.buildFuture;
	}

	private void build() {
		var allocatorStorage = new class_750();

		var blockRenderManager = this.client.method_1541();

		var matrices = new class_4587();
		var builderStorage = new HashMap<class_1921, class_287>();
		var random = class_5819.method_43053();

		WorldMesherRenderContext renderContext = null;
		try {
			//noinspection UnstableApiUsage
			renderContext = RendererAccess.INSTANCE.getRenderer() instanceof IndigoRenderer
				? new WorldMesherRenderContext(this.dummyWorld, layer -> this.getOrCreateBuilder(allocatorStorage, builderStorage, layer))
				: null;
		} catch (Throwable throwable) {
			var fabricApiVersion = FabricLoader.getInstance().getModContainer(Reference.MOD_ID).get().getMetadata().getCustomValue("fabric_api_build_version").getAsString();
			LOGGER.error(
				"Could not create a context for rendering Fabric API models. This is most likely due to an incompatible Fabric API version - this build of {} was compiled against '{}', try that instead",
				Reference.MOD_NAME,
				fabricApiVersion,
				throwable
			);
		}

		var entitiesFuture = new CompletableFuture<List<DynamicRenderInfo.EntityEntry>>();
		this.client.execute(() ->
			entitiesFuture.complete(this.entitySupplier.apply(this.client.field_1724, this.from, this.to.method_10069(1, 1, 1))
				.stream()
				.map(entity -> {
					entity.method_5773();
					return new DynamicRenderInfo.EntityEntry(
						entity,
						this.client.method_1561().method_23839(entity, 0)
					);
				}).toList()
			)
		);

		var blockEntities = new HashMap<class_2338, class_2586>();

		WorldMesherRenderContext finalRenderContext = renderContext;
		this.schematic.getAreas().keySet().forEach(region -> buildRegion(region, blockEntities, matrices, blockRenderManager, allocatorStorage, builderStorage, finalRenderContext, random));

		var future = new CompletableFuture<Void>();
		RenderSystem.recordRenderCall(() -> {
			this.bufferStorage.forEach((renderLayer, vertexBuffer) -> vertexBuffer.close());
			this.bufferStorage.clear();

			builderStorage.forEach((renderLayer, bufferBuilder) -> {
				var newBuffer = new class_291(class_291.class_8555.field_44793);

				var built = bufferBuilder.method_60794();
				if (built == null)
					return;

				if (renderLayer == class_1921.method_23583()) {
					built.method_60819(allocatorStorage.method_3154(renderLayer), class_8251.method_49906(0, 0, 1000));
				}

				newBuffer.method_1353();
				newBuffer.method_1352(built);

				var discardedBuffer = this.bufferStorage.put(renderLayer, newBuffer);
				if (discardedBuffer != null) {
					discardedBuffer.close();
				}
			});

			future.complete(null);
		});
		future.join();

		var entities = HashMultimap.<class_243, DynamicRenderInfo.EntityEntry>create();
		for (var entityEntry : entitiesFuture.join()) {
			entities.put(
				entityEntry.entity().method_19538()/*.subtract(this.from.getX(), this.from.getY(), this.from.getZ())*/,
				entityEntry
			);
		}

		allocatorStorage.close();
		this.renderInfo = new DynamicRenderInfo(
			blockEntities, entities
		);
	}

	private void buildRegion(String region, HashMap<class_2338, class_2586> blockEntities, class_4587 matrices, class_776 blockRenderManager, class_750 allocatorStorage, HashMap<class_1921, class_287> builderStorage, WorldMesherRenderContext renderContext, class_5819 random) {
		fi.dy.masa.litematica.selection.Box box = this.schematic.getAreas().get(region);
		RegionBlockView view = new RegionBlockView(
			Objects.requireNonNull(this.schematic.getSubRegionContainer(region)),
			box);
		Map<class_2338, class_2487> schematicBlockEntities = this.schematic.getBlockEntityMapForRegion(region);

		int currentBlockIndex = 0;
		int blocksToBuild = (this.to.method_10263() - this.from.method_10263() + 1)
			* (this.to.method_10264() - this.from.method_10264() + 1)
			* (this.to.method_10260() - this.from.method_10260() + 1);

		for (var pos : class_2338.method_10097(this.from, this.to)) {
			currentBlockIndex++;
			this.buildProgress = currentBlockIndex / (float) blocksToBuild;

			var state = view.method_8320(pos);
			if (state.method_26215()) continue;

			var renderPos = pos.method_10059(from);
			if (state.method_26204() instanceof class_2343 provider) {
				class_2586 blockEntity = provider.method_10123(this.client.method_1560().method_24515(), state);

				if (blockEntity != null) {
					((BlockEntityAccessor) blockEntity).setCachedState(state);
					blockEntity.method_58690(schematicBlockEntities.getOrDefault(pos, new class_2487()), this.dummyWorld.method_30349());
					blockEntity.method_31662(this.dummyWorld);
					blockEntities.put(renderPos, blockEntity);
				}
			}

			if (!state.method_26227().method_15769()) {
				var fluidState = state.method_26227();
				var fluidLayer = class_4696.method_23680(fluidState);

				matrices.method_22903();
				matrices.method_46416(-(pos.method_10263() & 15), -(pos.method_10264() & 15), -(pos.method_10260() & 15));
				matrices.method_46416(renderPos.method_10263(), renderPos.method_10264(), renderPos.method_10260());

				blockRenderManager.method_3352(pos, view, new FluidVertexConsumer(this.getOrCreateBuilder(allocatorStorage, builderStorage, fluidLayer), matrices.method_23760().method_23761()), state, fluidState);

				matrices.method_22909();
			}

			matrices.method_22903();
			matrices.method_46416(renderPos.method_10263(), renderPos.method_10264(), renderPos.method_10260());

			var blockLayer = class_4696.method_23679(state);

			final var model = blockRenderManager.method_3349(state);
			if (renderContext != null && !model.isVanillaAdapter()) {
				renderContext.tessellateBlock(view, state, pos, model, matrices);
			} else if (state.method_26217() == class_2464.field_11458) {
				blockRenderManager.method_3350().method_3374(view, model, state, pos, matrices, this.getOrCreateBuilder(allocatorStorage, builderStorage, blockLayer), cull, random, state.method_26190(pos), class_4608.field_21444);
			}

			matrices.method_22909();
		}
	}

	private class_4588 getOrCreateBuilder(class_750 allocatorStorage, Map<class_1921, class_287> builderStorage, class_1921 layer) {
		return builderStorage.computeIfAbsent(layer, renderLayer ->
			new class_287(allocatorStorage.method_3154(layer),  class_293.class_5596.field_27382, class_290.field_1590)
		);
	}

	public static class Builder {

		private final class_1920 world;
		private final TriFunction<class_1657, class_2338, class_2338, List<class_1297>> entitySupplier;

		private final class_2338 origin;
		private final class_2338 end;
		private boolean cull = true;
		private boolean useGlobalNeighbors = false;
		private boolean freezeEntities = false;

		private Runnable startAction = () -> {
		};
		private Runnable endAction = () -> {
		};

		@Deprecated(forRemoval = true)
		public Builder(class_1920 world, class_2338 origin, class_2338 end, Function<class_1657, List<class_1297>> entitySupplier) {
			this.world = world;
			this.origin = origin;
			this.end = end;
			this.entitySupplier = (player, $, $$) -> entitySupplier.apply(player);
		}

		public Builder(class_1920 world, class_2338 origin, class_2338 end, TriFunction<class_1657, class_2338, class_2338, List<class_1297>> entitySupplier) {
			this.world = world;
			this.origin = origin;
			this.end = end;
			this.entitySupplier = entitySupplier;
		}

		public Builder(class_1937 world, class_2338 origin, class_2338 end) {
			this(world, origin, end, (except, min, max) -> world.method_8333(except, class_238.method_54784(min, max), entity -> !(entity instanceof class_1657)));
		}

		public Builder(class_1920 world, class_2338 origin, class_2338 end) {
			this(world, origin, end, (except) -> List.of());
		}

		public Builder disableCulling() {
			this.cull = false;
			return this;
		}

		public Builder useGlobalNeighbors() {
			this.useGlobalNeighbors = true;
			return this;
		}

		public Builder freezeEntities() {
			this.freezeEntities = true;
			return this;
		}

		public Builder renderActions(Runnable startAction, Runnable endAction) {
			this.startAction = startAction;
			this.endAction = endAction;
			return this;
		}
	}

	public enum MeshState {
		NEW(false, false),
		BUILDING(true, false),
		REBUILDING(true, true),
		READY(false, true),
		CORRUPT(false, false);

		public final boolean isBuildStage;
		public final boolean canRender;

		MeshState(boolean buildStage, boolean canRender) {
			this.isBuildStage = buildStage;
			this.canRender = canRender;
		}
	}

}

