/*
 * Copyright (c) 2024 Fzzyhmstrs
 *
 * This file is part of Fzzy Config, a mod made for minecraft; as such it falls under the license of Fzzy Config.
 *
 * Fzzy Config is free software provided under the terms of the Timefall Development License - Modified (TDL-M).
 * You should have received a copy of the TDL-M with this software.
 * If you did not, see <https://github.com/fzzyhmstrs/Timefall-Development-Licence-Modified>.
 */

package me.fzzyhmstrs.fzzy_config.util

import me.fzzyhmstrs.fzzy_config.cast
import net.minecraft.client.MinecraftClient
import net.minecraft.client.gui.screen.Screen
import net.minecraft.entity.player.PlayerEntity
import net.minecraft.item.Item
import net.minecraft.item.ItemConvertible
import net.minecraft.recipe.Ingredient
import net.minecraft.registry.Registries
import net.minecraft.registry.Registry
import net.minecraft.registry.RegistryKey
import net.minecraft.registry.RegistryWrapper
import net.minecraft.registry.entry.RegistryEntryList
import net.minecraft.registry.tag.TagKey
import net.minecraft.server.network.ServerPlayerEntity
import net.minecraft.text.Text
import net.minecraft.util.Identifier
import net.minecraft.util.math.ColorHelper
import java.awt.Color
import java.util.Optional
import java.util.function.Predicate

object PortingUtils {

    fun getWhite(alpha: Float): Int {
        return Color(1f, 1f, 1f, alpha).rgb
    }

    fun fullAlpha(color: Int): Int {
        return color or (0xFF shl 24)
    }

    fun PlayerEntity.sendChat(message: Text) {
        this.sendMessage(message, false)
    }

    fun getDynamicManager(player: ServerPlayerEntity): RegistryWrapper.WrapperLookup {
        return player.world.registryManager.cast()
    }

    fun <T> Registry<T>.optional(id: Identifier): Optional<T> {
        return this.getOrEmpty(id)
    }

    fun RegistryWrapper.WrapperLookup.anyOptional(key: RegistryKey<out Registry<*>>):  Optional<out RegistryWrapper.Impl<*>> {
        return this.getOptionalWrapper(key)
    }

    fun <T> RegistryWrapper.WrapperLookup.optional(key: RegistryKey<out Registry<T>>):  Optional<out RegistryWrapper.Impl<T>> {
        return this.getOptionalWrapper(key)
    }

    fun <T> Registry<T>.tagIdList(): List<Identifier> {
        return this.streamTags().map { it.id }.toList()
    }

    fun <T> Registry<T>.tagIdList(predicate: Predicate<Identifier>? = null): List<Identifier> {
        return if(predicate == null)
            this.streamTags().map { it.id }.toList()
        else
            this.streamTags().map { it.id }.filter(predicate).toList()
    }

    fun <T> RegistryWrapper.Impl<T>.tagIdList(predicate: Predicate<Identifier>? = null): List<Identifier> {
        return if(predicate == null)
            this.streamTags().map { it.tag.id }.toList()
        else
            this.streamTags().map { it.tag.id }.filter(predicate).toList()
    }

    fun <T> Registry<T>.namedEntryList(tagKey: TagKey<T>): Optional<RegistryEntryList.Named<T>> {
        return this.getEntryList(tagKey)
    }

    fun <T> TagKey<T>.regRef(): RegistryKey<out Registry<T>> {
        return this.registry
    }

    fun TagKey<*>.regRefId(): Identifier {
        return this.registry.value
    }

    fun emptyIngredient(id: String = ""): Ingredient {
        throw UnsupportedOperationException("Ingredients can't be empty; item ID [$id] not found in the Items Registry.")
    }

    fun itemIngredient(item: ItemConvertible): Ingredient {
        return Ingredient.ofItems(item)
    }

    fun listIngredient(stacks: List<ItemConvertible>): Ingredient {
        return Ingredient.ofStacks(stacks.map { it.asItem().defaultStack }.stream())
    }

    fun tagIngredient(tag: TagKey<Item>): Ingredient {
        return Ingredient.fromTag(tag)
    }

    fun isAltDown(): Boolean {
        return Screen.hasAltDown()
    }

    fun isShiftDown(): Boolean {
        return Screen.hasShiftDown()
    }

    fun isControlDown(): Boolean {
        return Screen.hasControlDown()
    }
}