#version 330 compatibility
#include "/lib/settings.glsl"

uniform sampler2D colortex0;
uniform sampler2D depthtex0;
uniform vec3 cameraPosition;
uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;
uniform float near;
uniform float far;
uniform float ambientLight;
uniform vec3 sunPosition;
uniform vec3 skyColor;
uniform vec3 fogColor;

in vec2 texCoord;

/* DRAWBUFFERS:0 */
layout(location = 0) out vec4 color;

const vec3 wavelengths = vec3(700, 530, 440)/800;
const vec3 scattering = pow(wavelengths, vec3(4));
const int sampleCount = 16;
const vec3 sphereOrigin = vec3(0, 100, 0);

vec3 projectAndDivide(mat4 projectionMatrix, vec3 position) {
    vec4 homogeneousPos = projectionMatrix * vec4(position, 1.);
    return homogeneousPos.xyz/homogeneousPos.w;
}

vec3 getRealFragCoord() {
    vec3 screenPos = vec3(texCoord, texture2D(depthtex0, texCoord));
    vec3 ndcPos = screenPos * 2. -1.;
    vec3 viewPos = projectAndDivide(gbufferProjectionInverse, ndcPos);
    return mat3(gbufferModelViewInverse) * viewPos;
}

// Args: (sphere y pos, sphere radius), ray origin, ray direction
// Returns: vec2(first intersection point, second intersection point)
vec2 getIntersections(float sphereRad, vec3 rayOrigin, vec3 rayDir) {
    float midDist = dot(sphereOrigin-rayOrigin, rayDir);
    vec3 midPoint = rayOrigin+rayDir*midDist;
    float y = length(sphereOrigin-midPoint);
    if (y>sphereRad) return vec2(0);

    float x = sqrt(sphereRad*sphereRad-y*y);
    return vec2(max(midDist-x, 0), max(midDist+x, 0));
}

/*float getPointDensity(vec2 sphereSpecs, vec3 samplePoint) {
    vec3 so = vec3(0, sphereSpecs.x, 0);
    float rad = sphereSpecs.y;
    float dist = length(samplePoint-so)/rad;
    if (dist > 1) return 0.;
    float falloff = pow(dist, 2);
    float density = (1-falloff)*(1-dist);
    return density;
    }*/

vec3 getColor(float sphereRad, vec3 rayDir, float dist) {
    vec3 viewer = vec3(0, cameraPosition.y, 0);

    vec3 sum = vec3(0);
    for (int i = 0; i < sampleCount; i++) {
        vec3 samplePos = viewer + rayDir * i;
        vec3 sunDir = normalize(mat3(gbufferModelViewInverse) * sunPosition - samplePos);

        // Trace from the samplePosition toward the sun.
        vec2 intersections = getIntersections(sphereRad, samplePos, sunDir);
        // Calculate the distance the view ray travels through the atmosphere
        float sunRayDist = abs(intersections.y-intersections.x);
        
        float sphereDiameter = sphereRad * 2;
        float scatter = 1 - (sphereDiameter - sunRayDist) / sphereDiameter; // Increases as light travels further.

        vec3 startColor = vec3(0.3, 0.6, 1);
        vec3 endColor = vec3(1)-startColor;
        
        vec3 color = startColor * (1 - scatter) + endColor * scatter;

        sum += color * (1 - scatter);
    }
    return normalize(sum);
}

void main() {
    // Set it to the original color to begin with...
    color = texture(colortex0, texCoord);

    vec3 fragDir = normalize(getRealFragCoord());
    float sphereRad = 1000;
    vec2 intersections = getIntersections(sphereRad, vec3(0, cameraPosition.y, 0), fragDir);

    if (intersections.y != 0) {
        // Use intersection with terrain if it's closer
        intersections.y = min(length(getRealFragCoord()), intersections.y);
        // Calculate the distance the view ray travels through the atmosphere
        float dist = abs(intersections.y-intersections.x);
        float nightOMeter = (200 - (sunPosition.y + 100)) / 200;
        vec3 atmoColor = mix(skyColor, fogColor, 0.5);;

        float mixFactor = pow(clamp(pow(dist / far, 2) + nightOMeter * (dist / far) * 2, 0, 1), 2);
        color.rgb = mix(color.rgb, atmoColor, mixFactor);
    }
}
