#version 330 compatibility

// Outputs
layout(location = 0) out vec4 color;

/* RENDERTARGETS: 0 */

// Uniforms added for natural sky
uniform float frameTimeCounter;
uniform float rainStrength;
uniform vec3 sunPosition;   // Provided by MC shaders
uniform vec3 moonPosition;  // Provided by MC shaders

// Simple hash for stars
float rand(vec2 c) {
    return fract(sin(dot(c, vec2(12.9898, 78.233))) * 43758.5453);
}

void main() {

    // -----------------------------
    //    SKY GRADIENT
    // -----------------------------

    vec2 uv = gl_FragCoord.xy / vec2(1920.0, 1080.0);  // screen ratio
    float h = uv.y;

    // Day sky colors
    vec3 topColor     = vec3(0.05, 0.20, 0.55);   // deep blue
    vec3 middleColor  = vec3(0.20, 0.40, 0.85);   // bright sky
    vec3 horizonColor = vec3(0.85, 0.55, 0.35);   // warm horizon

    // Blend horizon → middle → top smoothly
    vec3 daySky = mix(
        mix(horizonColor, middleColor, smoothstep(0.0, 0.5, h)),
        topColor,
        smoothstep(0.4, 1.0, h)
    );



    // -----------------------------
    //     LIGHT FROM THE SUN
    // -----------------------------
    
    float sunDot = clamp(dot(normalize(sunPosition), vec3(0,0,1)), 0.0, 1.0);
    vec3 sunGlow = vec3(1.0, 0.85, 0.65) * pow(sunDot, 10.0);

    // Slight warm tint around the sun
    daySky += sunGlow * 0.5;



    // -----------------------------
    //       MOON TINT (Night)
    // -----------------------------

    float moonDot = clamp(dot(normalize(moonPosition), vec3(0,0,1)), 0.0, 1.0);
    vec3 moonGlow = vec3(0.5, 0.6, 1.0) * pow(moonDot, 10.0);



    // -----------------------------
    //        DAY/NIGHT CYCLE
    // -----------------------------
    
    // If sun below horizon → night
    float dayFactor = clamp(sunDot * 5.0, 0.0, 1.0);

    // Night sky colors
    vec3 nightTop     = vec3(0.02, 0.05, 0.10);
    vec3 nightHorizon = vec3(0.10, 0.10, 0.15);
    vec3 nightSky = mix(nightHorizon, nightTop, smoothstep(0.0, 1.0, h));

    // Blend day & night together
    vec3 finalSky = mix(nightSky + moonGlow * 0.2, daySky, dayFactor);



    // -----------------------------
    //             STARS
    // -----------------------------
    if (dayFactor < 0.2) {
        // Star field
        vec2 starUV = uv * 200.0;
        float star = rand(floor(starUV));
        if (star > 0.995) {
            float sparkle = pow(abs(sin(frameTimeCounter * 2.0 + star * 10.0)), 2.0);
            finalSky += vec3(1.0) * sparkle;
        }
    }



    // -----------------------------
    //       WEATHER (RAIN)
    // -----------------------------
    finalSky = mix(finalSky, finalSky * vec3(0.4, 0.45, 0.5), rainStrength * 0.6);



    // -----------------------------
    // Output
    // -----------------------------
    color = vec4(finalSky, 1.0);
}
