#version 330 compatibility


// Inputs
in vec2 mc_Entity;
in vec4 at_midBlock;

// Outputs
out vec2 texCoord;
out vec2 lightCoord;
out vec4 vertColor;
out float camVertDist;
out float vertDist;
out vec3 viewNormal;

// Uniforms
uniform mat4 gbufferModelView;
uniform mat4 gbufferModelViewInverse;
uniform mat4 gbufferProjection;
uniform vec3 cameraPosition;
uniform vec3 eyePosition;
uniform sampler2D noisetex;
uniform float frameTimeCounter;
uniform bool is_on_ground;


// ============================
//  SMOOTH WAVE FUNCTION
// ============================
float getHeight(vec2 position, vec3 sio, bool flatBottom) {

    // Main smooth wave (sinus)
    float smoothWave =
        sin(position.x * 0.25 +
            position.y * 0.25 +
            frameTimeCounter * (0.6 * sio.x))   // slightly faster
        * (0.12 * sio.y);                        // smoother intensity

    // Soft noise detail (no jitter)
    float noiseWave = texture(noisetex, position / 1024.0).r * 0.05;

    float shift = smoothWave + noiseWave + sio.z;

    // Prevent clipping for bottom faces
    if (flatBottom && at_midBlock.y == 32)
        shift = min(0.0, shift);

    return shift;
}


// Normal reconstruction from height
vec3 normalFromHeight(vec2 pos, vec3 sio, bool flatBottom, bool flipNormal) {
    float stepSize = 0.5;
    vec2 e = vec2(stepSize, 0);

    vec3 px1 = vec3(pos.x - e.x, getHeight(pos - e.xy, sio, flatBottom), pos.y - e.y);
    vec3 px2 = vec3(pos.x + e.x, getHeight(pos + e.xy, sio, flatBottom), pos.y + e.y);
    vec3 py1 = vec3(pos.x - e.y, getHeight(pos - e.yx, sio, flatBottom), pos.y - e.x);
    vec3 py2 = vec3(pos.x + e.y, getHeight(pos + e.yx, sio, flatBottom), pos.y + e.x);

    if (flipNormal)
        return normalize(cross(py2 - py1, px2 - px1));

    return normalize(cross(px2 - px1, py2 - py1));
}



void main() {
    gl_Position = ftransform();
    texCoord = gl_MultiTexCoord0.xy;
    lightCoord = (gl_TextureMatrix[1] * gl_MultiTexCoord1).xy;
    vertColor = gl_Color;
    camVertDist = length((gl_ModelViewMatrix * gl_Vertex).xyz);
    vertDist = camVertDist - length(eyePosition - cameraPosition);
    viewNormal = gl_NormalMatrix * gl_Normal.xyz;

    // Vertex position in world space
    vec3 eyeCameraPosition = cameraPosition + gbufferModelViewInverse[3].xyz;
    vec3 worldVertex = mat3(gbufferModelViewInverse) * (gl_ModelViewMatrix * gl_Vertex).xyz + eyeCameraPosition;

    // ============================
    //     WAVING CODE
    // ============================
    if (mc_Entity.x >= 2) {

        vec3 sio = vec3(1., 1., 0.);
        bool flatBottom = false;

        // Leaves
        if (mc_Entity.x == 2) {sio = vec3(1., 1., -0.4); flatBottom = true;}

        // Flowers / Crops / Plants
        if (mc_Entity.x == 3) {sio = vec3(1.2, 0.6, -0.15); flatBottom = false;}

        float vertShift = getHeight(worldVertex.xz, sio, flatBottom);

        // Prevent downward clipping
        if (floor(worldVertex.y - 0.001) > floor(worldVertex.y - 0.001 + vertShift))
            worldVertex.y = floor(worldVertex.y);
        else
            worldVertex.y += vertShift;

        worldVertex.y = floor(worldVertex.y * 16) / 16 + 0.001;

        // Update normals for smooth waving lighting
        vec3 originalNormal = mat3(gbufferModelViewInverse) * viewNormal;
        if (originalNormal.y > 0.1 && originalNormal.y < -0.1) {
            bool flipNormal = originalNormal.y > 0.1;
            viewNormal = gl_NormalMatrix * normalFromHeight(worldVertex.xz, sio, flatBottom, flipNormal);
        }
    }

    // Grass/ Cross-planes take normal from up face
    if (mc_Entity.x == 1) viewNormal = gl_NormalMatrix * vec3(0, 1, 0);

    // Output final transformed vertex
    gl_Position = gbufferProjection * vec4(mat3(gbufferModelView) * (worldVertex - eyeCameraPosition), 1);
}
