/*
 * Decompiled with CFR 0.152.
 */
package net.xmx.velthoric.physics.terrain;

import com.github.stephengold.joltjni.BodyInterface;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import net.minecraft.core.BlockPos;
import net.minecraft.core.SectionPos;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.xmx.velthoric.init.VxMainClass;
import net.xmx.velthoric.physics.terrain.VxSectionPos;
import net.xmx.velthoric.physics.terrain.cache.VxTerrainShapeCache;
import net.xmx.velthoric.physics.terrain.generation.VxTerrainGenerator;
import net.xmx.velthoric.physics.terrain.job.VxTerrainJobSystem;
import net.xmx.velthoric.physics.terrain.management.VxTerrainManager;
import net.xmx.velthoric.physics.terrain.management.VxTerrainTracker;
import net.xmx.velthoric.physics.terrain.storage.VxChunkDataStore;
import net.xmx.velthoric.physics.terrain.util.VxUpdateContext;
import net.xmx.velthoric.physics.world.VxPhysicsWorld;

public final class VxTerrainSystem
implements Runnable {
    private final VxPhysicsWorld physicsWorld;
    private final ServerLevel level;
    private final MinecraftServer server;
    private final VxTerrainJobSystem jobSystem;
    private final Thread workerThread;
    private final AtomicBoolean isInitialized = new AtomicBoolean(false);
    private final VxChunkDataStore chunkDataStore;
    private final VxTerrainShapeCache shapeCache;
    private final VxTerrainGenerator terrainGenerator;
    private final VxTerrainManager terrainManager;
    private final VxTerrainTracker terrainTracker;
    private final Set<VxSectionPos> chunksToRebuild = ConcurrentHashMap.newKeySet();
    private static final ThreadLocal<VxUpdateContext> updateContext = ThreadLocal.withInitial(VxUpdateContext::new);

    public VxTerrainSystem(VxPhysicsWorld physicsWorld, ServerLevel level) {
        this.physicsWorld = physicsWorld;
        this.level = level;
        this.server = level.getServer();
        this.jobSystem = new VxTerrainJobSystem();
        this.shapeCache = new VxTerrainShapeCache(2048);
        this.chunkDataStore = new VxChunkDataStore();
        this.terrainGenerator = new VxTerrainGenerator(this.shapeCache);
        this.terrainManager = new VxTerrainManager(physicsWorld, level, this.terrainGenerator, this.chunkDataStore, this.jobSystem);
        this.terrainTracker = new VxTerrainTracker(physicsWorld, this.terrainManager, this.chunkDataStore, level);
        this.workerThread = new Thread((Runnable)this, "Velthoric Terrain System - " + level.dimension().location().getPath());
        this.workerThread.setDaemon(true);
    }

    public void initialize() {
        if (this.isInitialized.compareAndSet(false, true)) {
            this.workerThread.start();
        }
    }

    public void shutdown() {
        if (this.isInitialized.compareAndSet(true, false)) {
            this.jobSystem.shutdown();
            this.workerThread.interrupt();
            VxMainClass.LOGGER.debug("Shutting down Terrain System for '{}'. Waiting up to 30 seconds for worker thread to exit...", (Object)this.level.dimension().location());
            try {
                this.workerThread.join(30000L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                VxMainClass.LOGGER.warn("Interrupted while waiting for terrain system thread to stop.");
            }
            if (this.workerThread.isAlive()) {
                StringBuilder stackTraceBuilder = new StringBuilder();
                stackTraceBuilder.append("Stack trace of deadlocked thread '").append(this.workerThread.getName()).append("':\n");
                for (StackTraceElement ste : this.workerThread.getStackTrace()) {
                    stackTraceBuilder.append("\tat ").append(ste).append("\n");
                }
                VxMainClass.LOGGER.fatal("Terrain system thread for '{}' did not terminate in 30 seconds. Forcing shutdown.\n{}", (Object)this.level.dimension().location(), (Object)stackTraceBuilder.toString());
                VxMainClass.LOGGER.warn("Skipping resource cleanup for '{}' to prevent data corruption due to the deadlocked thread.", (Object)this.level.dimension().location());
            } else {
                VxMainClass.LOGGER.debug("Terrain system for '{}' shut down gracefully.", (Object)this.level.dimension().location());
                VxMainClass.LOGGER.debug("Cleaning up terrain physics bodies for '{}'...", (Object)this.level.dimension().location());
                this.terrainManager.cleanupAllBodies();
                this.shapeCache.clear();
                this.terrainGenerator.close();
                this.chunkDataStore.clear();
                this.chunksToRebuild.clear();
                this.terrainTracker.clear();
                VxMainClass.LOGGER.debug("Terrain system for '{}' has been fully shut down.", (Object)this.level.dimension().location());
            }
        }
    }

    @Override
    public void run() {
        while (this.isInitialized.get() && !Thread.currentThread().isInterrupted() && this.server.isRunning()) {
            try {
                if (this.physicsWorld.isRunning()) {
                    this.terrainTracker.update();
                    this.processRebuildQueue();
                }
                Thread.sleep(50L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
            catch (Exception e) {
                if (!this.server.isRunning() || !this.isInitialized.get()) continue;
                VxMainClass.LOGGER.error("Error in TerrainSystem worker thread", (Throwable)e);
            }
        }
    }

    public void onBlockUpdate(BlockPos worldPos) {
        if (!this.isInitialized.get() || !this.server.isRunning()) {
            return;
        }
        VxSectionPos pos = VxSectionPos.fromBlockPos(worldPos.immutable());
        if (this.terrainManager.isManaged(pos)) {
            this.chunksToRebuild.add(pos);
        }
        this.physicsWorld.execute(() -> {
            BodyInterface bi = this.physicsWorld.getPhysicsSystem().getBodyInterface();
            if (bi == null) {
                return;
            }
            VxUpdateContext ctx = updateContext.get();
            ctx.vec3_1.set((float)worldPos.getX() - 2.0f, (float)worldPos.getY() - 2.0f, (float)worldPos.getZ() - 2.0f);
            ctx.vec3_2.set((float)worldPos.getX() + 3.0f, (float)worldPos.getY() + 3.0f, (float)worldPos.getZ() + 3.0f);
            ctx.aabox_1.setMin(ctx.vec3_1);
            ctx.aabox_1.setMax(ctx.vec3_2);
            bi.activateBodiesInAaBox(ctx.aabox_1, ctx.bplFilter, ctx.olFilter);
        });
    }

    private void processRebuildQueue() {
        if (this.chunksToRebuild.isEmpty()) {
            return;
        }
        HashSet<VxSectionPos> batch = new HashSet<VxSectionPos>(this.chunksToRebuild);
        this.chunksToRebuild.removeAll(batch);
        for (VxSectionPos pos : batch) {
            this.terrainManager.rebuildChunk(pos);
        }
    }

    public boolean isReady(VxSectionPos pos) {
        return this.terrainManager.isReady(pos);
    }

    public boolean isPlaceholder(VxSectionPos pos) {
        return this.terrainManager.isPlaceholder(pos);
    }

    public boolean isSectionReady(SectionPos sectionPos) {
        if (sectionPos == null) {
            return false;
        }
        return this.isReady(new VxSectionPos(sectionPos.x(), sectionPos.y(), sectionPos.z()));
    }

    public boolean isTerrainBody(int bodyId) {
        if (bodyId <= 0) {
            return false;
        }
        for (int id : this.chunkDataStore.getBodyIds()) {
            if (id != bodyId) continue;
            return true;
        }
        return false;
    }

    public ServerLevel getLevel() {
        return this.level;
    }
}

