package be.immersivechess.logic;

import be.immersivechess.ImmersiveChess;
import be.immersivechess.block.BoardBlock;
import be.immersivechess.resource.BlockStateLuminanceMapper;
import ch.astorm.jchess.core.Color;
import ch.astorm.jchess.core.Coordinate;
import com.google.common.collect.Iterables;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.class_1657;
import net.minecraft.class_1920;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_2487;
import net.minecraft.class_2512;
import net.minecraft.class_2680;
import net.minecraft.class_2682;
import net.minecraft.class_3341;
import net.minecraft.class_3545;


public class MultiblockBoard {
    // constants
    public static final int BOARD_SIZE = 8;

    // nbt keys
    private static final String A1_KEY = "A1";
    private static final String WHITE_DIRECTION_KEY = "WhiteDirection";

    // properties
    // Position corresponding to A1 square (black) which is used to map all squares together with play direction.
    private final class_2338 a1;
    // Direction in the world towards which white is playing.
    private final class_2350 whitePlayDirection;
    private final class_3341 bounds;

    private MultiblockBoard(class_2338 a1, class_2350 whiteDirection) {
        this(a1, whiteDirection, computeBounds(a1, whiteDirection));
    }

    private MultiblockBoard(class_2338 a1, class_2350 whiteDirection, class_3341 bounds) {
        this.a1 = a1;
        this.whitePlayDirection = whiteDirection;
        this.bounds = bounds;
    }

    private static class_3341 computeBounds(class_2338 a1, class_2350 whiteDirection) {
        return class_3341.method_34390(a1, a1.method_10079(whiteDirection, BOARD_SIZE - 1).method_10079(whiteDirection.method_10170(), BOARD_SIZE - 1));
    }

    public class_3341 getBounds() {
        return bounds;
    }

    public class_2350 getWhitePlayDirection() {
        return whitePlayDirection;
    }

    @Nullable
    public Coordinate getSquare(class_2338 pos) {
        if (!bounds.method_14662(pos))
            return null;

        class_2338 localPos = pos.method_25503().method_10059(a1);

        class_2350.class_2351 forwardAxis = whitePlayDirection.method_10166();
        class_2350.class_2351 sidewaysAxis = forwardAxis == class_2350.class_2351.field_11048 ? class_2350.class_2351.field_11051 : class_2350.class_2351.field_11048;
        int forward = Math.abs(localPos.method_30558(forwardAxis));
        int sideways = Math.abs(localPos.method_30558(sidewaysAxis));

        return new Coordinate(forward, sideways);
    }

    public class_2338 getPos(Coordinate coordinate) {
        return a1.method_10079(whitePlayDirection, coordinate.getRow()).method_10079(whitePlayDirection.method_10170(), coordinate.getColumn());
    }

    /**
     * Returns the color of the square
     */
    @Nullable
    public Color getColorOfPos(class_2338 pos) {
        if (!bounds.method_14662(pos)) return null;
        class_2338 diff = pos.method_10059(a1);
        int index = Math.abs(diff.method_10263()) + Math.abs(diff.method_10260());
        return index % 2 == 0 ? Color.BLACK : Color.WHITE;
    }

    public Color getSide(class_2338 pos) {
        return getSide(getSquare(pos));
    }

    public Color getSide(Coordinate square) {
        if (square == null) return null;
        return square.getRow() <= 3 ? Color.WHITE : Color.BLACK;
    }

    /**
     * Returns the bounds of a valid board including position or null if not valid;
     */
    @Nullable
    public static MultiblockBoard getValidBoard(class_1937 world, class_1657 player, class_2338 origin) {
        class_2680 material1 = getAppearanceState(world, origin);

        if (!isBoardMaterial(material1))
            return null;

        double material1Luminance = BlockStateLuminanceMapper.INSTANCE.getLuminance(material1);

        // Create list of candidate materials in the four cardinal directions around origin.
        List<class_2680> candidateMaterials = Stream.of(
                        getAppearanceState(world, origin.method_10093(class_2350.field_11043)),
                        getAppearanceState(world, origin.method_10093(class_2350.field_11034)),
                        getAppearanceState(world, origin.method_10093(class_2350.field_11035)),
                        getAppearanceState(world, origin.method_10093(class_2350.field_11039))
                )
                // Count occurences
                .collect(Collectors.groupingBy(e -> e, Collectors.counting()))
                .entrySet().stream()
                // Filter materials that occur at least twice and are valid and different from first material
                .filter(entry -> entry.getValue() > 1 && isBoardMaterial(entry.getKey()) && entry.getKey() != material1)
                // Don't allow luminance to be the same because then we can't deterministically set black and white colors
                .filter(entry -> BlockStateLuminanceMapper.INSTANCE.getLuminance(entry.getKey()) != material1Luminance)
                .map(Map.Entry::getKey).toList();

        // Check for three directions. If valid, at least one of those will be correct.
        for (class_2680 material2 : candidateMaterials) {
            ImmersiveChess.LOGGER.debug("Trying to validate board with materials " + material1 + " and " + material2);
            MultiblockBoard board;
            if ((board = getValidBoard(world, player, origin, material1, material2)) != null) {
                return board;
            }
        }

        return null;
    }

    /**
     * Check if valid board can be formed at position with given materials.
     */
    @Nullable
    public static MultiblockBoard getValidBoard(class_1937 world, class_1657 player, class_2338 origin, class_2680 material1, class_2680 material2) {
        // Check materials
        if (!isBoardMaterial(material1) || !isBoardMaterial(material2))
            return null;

        class_2680[] materials = {material1, material2};

        class_3341 bounds = traceBoardBounds(world, origin, materials);

        if (bounds.method_35414() != BOARD_SIZE) {
            ImmersiveChess.LOGGER.debug("Invalid board size in X axis: " + bounds.method_35414());
            return null;
        }

        if (bounds.method_14663() != BOARD_SIZE) {
            ImmersiveChess.LOGGER.debug("Invalid board size in Z axis: " + bounds.method_14663());
            return null;
        }

        // Check material in board and edges
        return getValidBoard(world, player, origin, materials, bounds);
    }

    /**
     * Check if board with bounds can be formed with given materials.
     * <p>
     * Also checks if the border is made of a different material.
     */
    @Nullable
    public static MultiblockBoard getValidBoard(class_1937 world, class_1657 player, class_2338 origin, class_2680[] materials, class_3341 bounds) {
        assert materials.length == 2;
        assert bounds.method_14660() == 1;

        if (!checkBoardMaterials(world, origin, materials, bounds)) return null;
        if (!checkBoardBorder(world, origin, materials, bounds)) return null;
        if (!checkBoardAccessible(world, player, bounds)) return null;

        // determine white and black material based on luminance
        Color c = BlockStateLuminanceMapper.INSTANCE.getColorOfFirstBlock(materials[0], materials[1]);
        class_2680 blackBlock = c == Color.BLACK ? materials[0] : materials[1];
        class_2680 whiteBlock = c == Color.BLACK ? materials[1] : materials[0];

        // find a1 based on player that selected origin being black
        // first find two black corners
        class_2338 center = bounds.method_22874();
        List<class_2338> blackCorners = class_2338.method_23627(bounds)
                .map(class_2338::new)     // copy pos, because the same variable gets set in the iterator
                .filter(p -> getAppearanceState(world, p).equals(blackBlock))
                .filter(p -> p.method_10268(center.method_10263(), center.method_10264(), center.method_10260()) >= 2 * Math.pow(3.5, 2) - 0.01f)
                .toList();

        assert blackCorners.size() == 2;
        class_2338 c1 = blackCorners.get(0);
        class_2338 c2 = blackCorners.get(1);

        // find forward/backward direction to determine side
        // assume c1 as corner on white side for computation, gets swapped if incorrect
        class_2338 facing = c2.method_10059(c1);

        class_2350 d1 = class_2350.method_10147(facing.method_10263(), 0, 0);
        class_2350 d2 = class_2350.method_10147(0, 0, facing.method_10260());
        class_2350 forward = d1.method_10170().equals(d2) ? d1 : d2;

        // side closest to click position (origin) is black
        int dist1 = Math.abs(c1.method_10059(origin).method_30558(forward.method_10166()));
        int dist2 = Math.abs(c2.method_10059(origin).method_30558(forward.method_10166()));
        if (dist1 < dist2) {
            c1 = c2;
            forward = forward.method_10153();
        }

        return new MultiblockBoard(c1, forward, bounds);
    }

    public void endBoardBlocks(class_1937 world) {
        for (class_2338 blockPos : class_2338.method_10094(bounds.method_35415(), bounds.method_35416(), bounds.method_35417(), bounds.method_35418(), bounds.method_35419(), bounds.method_35420())) {
            BoardBlock.placeBack(world, blockPos);
        }
    }

    /**
     * Check if board of right materials.
     */
    private static boolean checkBoardMaterials(class_1937 world, class_2338 origin, class_2680[] materials, class_3341 bounds) {
        for (class_2338 pos : (Iterable<class_2338>) class_2338.method_23627(bounds)::iterator) {
            int distance = pos.method_19455(origin);
            if (!getAppearanceState(world, pos).equals(materials[distance % 2])) {
                ImmersiveChess.LOGGER.debug("Chess board material invalid at " + pos);
                ImmersiveChess.LOGGER.debug("Expected " + materials[distance % 2] + " but got " + getAppearanceState(world, pos));
                return false;
            }
        }
        return true;
    }

    /**
     * Check if borders are of different material than main board.
     */
    private static boolean checkBoardBorder(class_1937 world, class_2338 origin, class_2680[] materials, class_3341 bounds) {
        assert bounds.method_14660() == 1;

        Iterable<class_2338> borderBlocks = class_2338.method_10094(
                bounds.method_35415() - 1, bounds.method_35416(), bounds.method_35417() - 1,
                bounds.method_35418() + 1, bounds.method_35416(), bounds.method_35417() - 1);
        borderBlocks = Iterables.concat(borderBlocks, class_2338.method_10094(
                bounds.method_35415() - 1, bounds.method_35416(), bounds.method_35420() + 1,
                bounds.method_35418() + 1, bounds.method_35416(), bounds.method_35420() + 1));
        borderBlocks = Iterables.concat(borderBlocks, class_2338.method_10094(
                bounds.method_35418() + 1, bounds.method_35416(), bounds.method_35417(), // don't need +1 because already checked
                bounds.method_35418() + 1, bounds.method_35416(), bounds.method_35420()));
        borderBlocks = Iterables.concat(borderBlocks, class_2338.method_10094(
                bounds.method_35415() - 1, bounds.method_35416(), bounds.method_35417(),
                bounds.method_35415() - 1, bounds.method_35416(), bounds.method_35420()));

        for (class_2338 pos : borderBlocks) {
            class_2680 bs = getAppearanceState(world, pos);
            if (bs.equals(materials[0]) || bs.equals(materials[1])) {
                ImmersiveChess.LOGGER.debug("Invalid block " + bs + " at position " + pos);
                ImmersiveChess.LOGGER.debug("Borders of chess board are not allowed to be of same material");
                return false;
            }
            class_2680 appearanceState = bs.getAppearance(world, pos, class_2350.field_11036, null, null);
            if (appearanceState.equals(materials[0]) || appearanceState.equals(materials[1])) {
                ImmersiveChess.LOGGER.debug("Invalid block " + bs + " at position " + pos);
                ImmersiveChess.LOGGER.debug("Borders of chess board are not allowed to appear as the same material");
                return false;
            }
            if (bs.method_27852(be.immersivechess.block.Blocks.BOARD_BLOCK)) {
                ImmersiveChess.LOGGER.debug("Invalid block " + bs + " at position " + pos);
                ImmersiveChess.LOGGER.debug("Multiple chess boards are not allowed next to each other");
                return false;
            }
        }

        return true;
    }

    private static boolean checkBoardAccessible(class_1937 world, class_1657 player, class_3341 bounds) {
        for (class_2338 pos : (Iterable<class_2338>) class_2338.method_23627(bounds)::iterator) {
            if (!world.method_8505(player, pos)) return false;
        }
        return true;
    }

    /**
     * Traces the map from a given origin to see how far the board could go.
     * Checks up until sizes of 2 * BOARD_SIZE.
     */
    private static class_3341 traceBoardBounds(class_1937 world, class_2338 origin, class_2680[] materials) {
        // Determine bounds of board by tracing X and Y axis from origin piece
        class_3545<Integer, Integer> boundsX = getValidLength(world, origin, materials, class_2350.class_2351.field_11048);
        class_3545<Integer, Integer> boundsZ = getValidLength(world, origin, materials, class_2350.class_2351.field_11051);
        // Form bounding box
        return class_3341.method_34390(
                new class_2382(boundsX.method_15442(), origin.method_10264(), boundsZ.method_15442()),
                new class_2382(boundsX.method_15441(), origin.method_10264(), boundsZ.method_15441())
        );
    }

    /**
     * Determine bounds of board by tracing axis from origin piece
     */
    private static class_3545<Integer, Integer> getValidLength(class_1937 world, class_2338 origin, class_2680[] materials, class_2350.class_2351 axis) {
        assert materials.length == 2;

        class_3545<Integer, Integer> result = new class_3545<>(null, null);
        for (int sign = -1; sign < 2; sign += 2) {
            int limit = origin.method_30558(axis);
            for (int step = 1; step < BOARD_SIZE; step++) {
                class_2338 pos = origin.method_30513(axis, sign * step);
                int distance = pos.method_19455(origin);
                if (!getAppearanceState(world, pos).equals(materials[distance % 2])) break;
                limit = pos.method_30558(axis);
            }
            if (result.method_15442() == null) {
                result.method_34964(limit);
            } else {
                result.method_34965(limit);
            }

        }
        return result;
    }

    private static class_2680 getAppearanceState(class_1920 world, class_2338 pos){
        return world.method_8320(pos).getAppearance(world, pos, class_2350.field_11036, null, null);
    }

    public static boolean isBoardMaterial(class_2680 blockState) {
        if (blockState.method_26204() instanceof BoardBlock) return true;
        if (blockState.equals(class_2246.field_10124.method_9564())) return false;
        if (!blockState.method_26234(class_2682.field_12294, class_2338.field_10980)) return false;
        if (blockState.method_31709()) return false;
        return true;
    }

    public class_2487 writeNbt(class_2487 nbt) {
        nbt.method_10566(A1_KEY, class_2512.method_10692(a1));
        nbt.method_10566(WHITE_DIRECTION_KEY, class_2512.method_10692(new class_2338(whitePlayDirection.method_10163())));
        return nbt;
    }

    public static MultiblockBoard fromNbt(class_2487 nbt) {
        if (!nbt.method_10545(A1_KEY) || !nbt.method_10545(WHITE_DIRECTION_KEY)) return null;

        class_2338 a1 = class_2512.method_10691(nbt.method_10562(A1_KEY));
        class_2382 dir = class_2512.method_10691(nbt.method_10562(WHITE_DIRECTION_KEY));
        class_2350 whitePlayDirection = class_2350.method_50026(dir.method_10263(), dir.method_10264(), dir.method_10260());

        return new MultiblockBoard(a1, whitePlayDirection);
    }
}
