package be.immersivechess.world;

import be.immersivechess.ImmersiveChess;
import be.immersivechess.advancement.criterion.ChessGameCriterion;
import be.immersivechess.advancement.criterion.Criteria;
import be.immersivechess.block.BoardBlock;
import be.immersivechess.block.entity.BoardBlockEntity;
import be.immersivechess.block.entity.StructureRenderedBlockEntity;
import be.immersivechess.item.PieceItem;
import be.immersivechess.logic.MultiblockBoard;
import be.immersivechess.logic.Piece;
import be.immersivechess.structure.StructureMap;
import ch.astorm.jchess.JChessGame;
import ch.astorm.jchess.core.*;
import ch.astorm.jchess.core.entities.King;
import ch.astorm.jchess.core.rules.Displacement;
import ch.astorm.jchess.io.PGNReader;
import ch.astorm.jchess.io.PGNWriter;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import net.minecraft.class_1657;
import net.minecraft.class_18;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2498;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3341;
import net.minecraft.class_3417;
import net.minecraft.class_3419;

public class ChessGameState extends class_18 {
    // constants
    private static final String LOCATION = ImmersiveChess.MOD_ID + "/games/";

    // Nbt keys
    private static final String GAME_KEY = "Game";
    private static final String BOARD_KEY = "Board";
    private static final String MINED_KEY = "Mined";
    private static final String DRAW_OFFER_KEY = "DrawOfferedBy";
    private static final String WHITE_PLAYER_STRUCTURES_KEY = "StructuresOfWhite";
    private static final String BLACK_PLAYER_STRUCTURES_KEY = "StructuresOfBlack";
    private static final String BLACK_PIECES_RENDER_OPTION_KEY = "BlackRenderOption";
    private static final String WHITE_PIECES_RENDER_OPTION_KEY = "WhiteRenderOption";

    // pgn metadata keys
    private static final String GAME_ID_KEY = "GameId";
    private static final String WHITE_PLAYER_KEY = "White";
    private static final String BLACK_PLAYER_KEY = "Black";
    private static final String DATE_KEY = "Date";

    private final class_3218 world;

    // State that is persisted in nbt
    private final JChessGame game;
    private final MultiblockBoard board;
    private Coordinate minedSquare;

    // contains all structures (both colors) for each player. In total mapping contains two sets of white and two sets of black pieces.
    private final Map<Color, StructureMap> playerStructures = new HashMap<>();
    private final Map<Color, PieceRenderOption> renderOptions = PieceRenderOption.createDefaultRenderOptions();

    private Color drawOfferedBy;

    private ChessGameState(class_3218 world, JChessGame game, MultiblockBoard board) {
        this(world, game, board, null, null);
    }

    private ChessGameState(class_3218 world, JChessGame game, MultiblockBoard board, Coordinate minedSquare, Color DrawOfferedBy) {
        this.world = world;
        this.game = game;
        this.board = board;
        this.minedSquare = minedSquare;
        this.drawOfferedBy = DrawOfferedBy;
    }

    public static ChessGameState create(class_3218 world, MultiblockBoard board) {
        ChessGameState state = new ChessGameState(world, JChessGame.newGame(), board);
        state.game.getMetadata().put(GAME_ID_KEY, UUID.randomUUID().toString());
        state.game.getMetadata().put(DATE_KEY, LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy.MM.dd")));
        state.method_80();

        world.method_17983().method_123(state.getSavePath(), state);

        return state;
    }

    @Nullable
    public static ChessGameState get(class_3218 world, String gameId) {
        class_8645<ChessGameState> type = new class_8645<>(null, nbt -> fromNbt(world, nbt), null);
        return world.method_17983().method_20786(type, LOCATION + gameId);
    }

    public String getGameId() {
        return game.getMetadata().get(GAME_ID_KEY);
    }

    public class_2350 getWhitePlayDirection() {
        return board.getWhitePlayDirection();
    }

    private void setPlayerName(Color color, @Nullable String name) {
        String key = color == Color.WHITE ? WHITE_PLAYER_KEY : BLACK_PLAYER_KEY;
        if (name != null)
            game.getMetadata().put(key, name);
        else
            game.getMetadata().remove(key);
        method_80();
    }

    public void setPlayer(Color color, class_1657 mcPlayer) {
        setPlayerName(color, mcPlayer.method_5820());
        if (mcPlayer instanceof class_3222 serverPlayer)
            Criteria.GAME_CRITERION.trigger(serverPlayer, ChessGameCriterion.Type.START);
    }

    public void setPlayer(Color color, class_1657 mcPlayer, StructureMap structures) {
        setPlayer(color, mcPlayer);
        setStructures(color, structures);
    }

    public void removePlayer(Color color) {
        setPlayerName(color, null);
        playerStructures.remove(color);
        setRenderOption(color, PieceRenderOption.DEFAULT);


        // also removes when player is not present
        placePieces();
    }

    public boolean togglePlayer(Color color, class_1657 mcPlayer, StructureMap structures) {
        String playerName = getPlayerName(color);
        if (playerName == null) {
            setPlayer(color, mcPlayer, structures);
            return true;
        }
        if (playerName.equals(mcPlayer.method_5820())) {
            removePlayer(color);
            return true;
        }
        return false;
    }

    @Nullable
    public String getPlayerName(Color color) {
        String name = game.getMetadata().get(color == Color.WHITE ? WHITE_PLAYER_KEY : BLACK_PLAYER_KEY);
        if (name == null || name.isEmpty()) return null;
        return name;
    }

    public Color getColor(class_1657 player) {
        for (Color color : Color.values())
            if (player.method_5820().equals(getPlayerName(color))) return color;
        return null;
    }

    public Color getColorOnMove() {
        return game.getColorOnMove();
    }

    public int getCurrentMoveIndex() {
        return game.getPosition().getMoveHistory().size() + 1;
    }

    public MultiblockBoard getBoard() {
        return board;
    }

    public boolean hasPlayerMinedPiece() {
        return minedSquare != null;
    }

    @Nullable
    public String getActivePlayerName() {
        Color activeColor = game.getColorOnMove();
        return getPlayerName(activeColor);
    }

    @Nullable
    public Piece getPiece(Coordinate square) {
        Moveable moveable = game.getPosition().get(square);
        if (moveable == null) return null;
        return Piece.fromMoveable(moveable);
    }

    public StructureMap getStructuresOfPlayer(Color color) {
        return playerStructures.computeIfAbsent(color, c -> new StructureMap());
    }

    public void setStructures(Color color, StructureMap structureMap) {
        playerStructures.put(color, structureMap);
        if (structureMap.hasAnyOf(color))
            setRenderOption(color, PieceRenderOption.OWN);

        placePieces();
        method_80();
    }

    public PieceRenderOption getRenderOption(Color color) {
        return renderOptions.get(color);
    }

    public List<PieceRenderOption> getValidRenderOptions(Color color) {
        List<PieceRenderOption> list = new ArrayList<>();
        list.add(PieceRenderOption.DEFAULT);
        if (playerStructures.get(color) != null && playerStructures.get(color).hasAnyOf(color))
            list.add(PieceRenderOption.OWN);
        if (playerStructures.get(color.opposite()) != null && playerStructures.get(color.opposite()).hasAnyOf(color))
            list.add(PieceRenderOption.OPPONENT);
        return list;
    }

    public boolean setRenderOption(Color color, PieceRenderOption option) {
        if (!getValidRenderOptions(color).contains(option)) return false;
        renderOptions.put(color, option);
        method_80();
        updatePieceStructures();
        return true;
    }

    private void updatePieceStructures() {
        for (Coordinate square : game.getPosition().getMoveables().keySet()) {
            updatePieceStructure(square);
        }
    }

    private void updatePieceStructure(Coordinate square) {
        class_2338 destPos = board.getPos(square).method_10084();
        Piece piece = getPiece(square);
        if (piece == null) return;

        class_2586 blockEntity = world.method_8321(destPos);
        if (blockEntity instanceof StructureRenderedBlockEntity pieceBlockEntity) {
            pieceBlockEntity.setStructure(getStructure(piece));
        }
    }

    @Nullable
    public class_2487 getStructure(Piece piece) {
        StructureMap defaultStructures = StructureMap.getDefault(world);
        return switch (getRenderOption(piece.getColor())) {
            case DEFAULT -> defaultStructures.get(piece);
            case OWN -> playerStructures.get(piece.getColor()).getOrDefault(piece, defaultStructures.get(piece));
            case OPPONENT ->
                    playerStructures.get(piece.getColor().opposite()).getOrDefault(piece, defaultStructures.get(piece));
        };
    }

    public boolean isPlayerOnMove(class_1657 player) {
        return player.method_5820().equals(getActivePlayerName());
    }

    public JChessGame.Status getStatus() {
        return game.getStatus();
    }

    public boolean resign(class_1657 player) {
        if (game.getStatus().isFinished()) return false;

        // game not ready yet
        if (getPlayerName(Color.WHITE) == null || getPlayerName(Color.BLACK) == null) return false;

        Color color = getColor(player);
        if (color == null) return false;
        game.resign(color);
        method_80();
        onGameEnded();
        return true;
    }

    /**
     * Offers or accepts draw by player depending on state.
     */
    public boolean draw(class_1657 player) {
        if (game.getStatus().isFinished()) return false;

        // game not ready yet
        if (getPlayerName(Color.WHITE) == null || getPlayerName(Color.BLACK) == null) return false;

        Color playerColor = getColor(player);
        if (playerColor == null) return false;

        // offer draw
        if (drawOfferedBy == null) {
            drawOfferedBy = playerColor;
            method_80();

            class_3222 opponent = getPlayer(drawOfferedBy.opposite());
            if (opponent != null) {
                opponent.method_7353(class_2561.method_43469("immersivechess.draw_offer_message", getPlayerName(drawOfferedBy)), true);
                opponent.method_7353(class_2561.method_43469("immersivechess.draw_offer_chat", getPlayerName(drawOfferedBy)), false);
                opponent.method_17356(class_3417.field_14793.comp_349(), class_3419.field_15248, 1f, 1f);
            }
            return true;
        }

        // accept draw
        if (playerColor.equals(drawOfferedBy.opposite()) || getPlayerName(Color.WHITE).equals(getPlayerName(Color.BLACK))) {
            game.draw();
            method_80();
            onGameEnded();
            return true;
        }

        return false;
    }

    /**
     * Force draw if game not already finished.
     */
    public void forceDraw() {
        if (game.getStatus().isFinished()) return;
        game.draw();
        onGameEnded();
    }

    public String getDrawOfferedTo() {
        if (drawOfferedBy == null)
            return "";
        String opponent = getPlayerName(drawOfferedBy.opposite());
        return opponent != null ? opponent : "";
    }

    private void onGameEnded() {
        switch (game.getStatus()) {
            case NOT_FINISHED -> {
                ImmersiveChess.LOGGER.error("Invalid call to onGameEnded when status is NOT_FINISHED");
            }
            case WIN_WHITE -> {
                onGameEnded(Color.WHITE);
            }
            case WIN_BLACK -> {
                onGameEnded(Color.BLACK);
            }
            case DRAW, DRAW_STALEMATE, DRAW_REPETITION, DRAW_NOCAPTURE -> {
                class_3222 p1 = getPlayer(Color.WHITE);
                class_3222 p2 = getPlayer(Color.BLACK);

                if (p1 != null)
                    p1.method_7353(class_2561.method_43471("immersivechess.draw_message"), true);
                if (p2 != null)
                    p2.method_7353(class_2561.method_43471("immersivechess.draw_message"), true);

                String p1Name = getPlayerName(Color.WHITE);
                String p2Name = getPlayerName(Color.BLACK);
                if (p1Name != null && !p1Name.equals(p2Name))
                    world.method_8503().method_3760().method_43514(class_2561.method_43469("immersivechess.draw_broadcast", p1Name, p2Name), false);
                world.method_8396(null, board.getPos(new Coordinate("a1")), class_3417.field_39028.get(0).comp_349(), class_3419.field_15248, 1f, 1f);
            }
        }
    }

    private void onGameEnded(Color winColor) {
        class_3222 winner = getPlayer(winColor);
        class_3222 loser = getPlayer(winColor.opposite());

        // advancements
        Criteria.GAME_CRITERION.trigger(winner, ChessGameCriterion.Type.WIN);
        Criteria.GAME_CRITERION.trigger(loser, ChessGameCriterion.Type.LOSE);

        // announce in chat to all
        String winnerName = getPlayerName(winColor);
        String loserName = getPlayerName(winColor.opposite());
        if (winnerName != null && !winnerName.equals(loserName))
            world.method_8503().method_3760().method_43514(class_2561.method_43469("immersivechess.win_broadcast", winnerName, loserName), false);

        // hover text to players
        if (winner != null)
            winner.method_7353(class_2561.method_43471("immersivechess.win_message"), true);
        if (loser != null && !loser.equals(winner))
            loser.method_7353(class_2561.method_43471("immersivechess.lose_message"), true);

        // play victory sound for all except loser
        class_2338 pos = board.getPos(new Coordinate("a1"));
        world.method_8396(Objects.equals(loser, winner) ? null : loser, pos, class_3417.field_39028.get(0).comp_349(), class_3419.field_15248, 1f, 1.5f);
        if (loser != null && !loser.equals(winner))
            loser.method_17356(class_3417.field_39028.get(7).comp_349(), class_3419.field_15248, 1f, 0.8f);
    }

    /**
     * Checks if the active player still has at least a piece in their inventory and if not, places the mined block back.
     * Also updates world with logical board.
     */
    public void performIntegrityCheck() {
        class_1657 player = getActivePlayer();
        if (minedSquare != null && player != null && !player.method_31548().method_7382(PieceItem.PIECE_TAG))
            clearMinedSquare();

        placePieces();
    }

    public void placePieces() {
        for (Map.Entry<Coordinate, Moveable> entry : game.getPosition().getMoveables().entrySet()) {
            if (entry.getKey().equals(minedSquare))
                continue;

            if (getPlayerName(entry.getValue().getColor()) == null) {
                breakPiece(entry.getKey());
            } else {
                Piece piece = Piece.fromMoveable(entry.getValue());
                placePiece(entry.getKey(), piece);
            }
        }
    }

    @Nullable
    private class_3222 findPlayer(String name) {
        for (class_3222 player : world.method_18456()) {
            if (player.method_5820().equals(name))
                return player;
        }
        return null;
    }

    @Nullable
    private class_3222 getPlayer(Color color) {
        return findPlayer(getPlayerName(color));
    }

    @Nullable
    private class_3222 getActivePlayer() {
        return findPlayer(getActivePlayerName());

    }

    /**
     * Performs the cleanup after a game has ended
     */
    public void endBoardBlocks() {
        class_3341 bounds = board.getBounds();
        for (class_2338 blockPos : class_2338.method_10094(bounds.method_35415(), bounds.method_35416(), bounds.method_35417(), bounds.method_35418(), bounds.method_35419(), bounds.method_35420())) {
            BoardBlock.placeBack(world, blockPos);
        }
    }

    public boolean canMinePiece(Coordinate square, class_1657 player) {
        if (hasPlayerMinedPiece()) return false;

        // game not ready yet
        if (getPlayerName(Color.WHITE) == null || getPlayerName(Color.BLACK) == null) return false;

        if (game.getStatus().isFinished()) return false;

        if (game.getPosition().get(square) == null) return false;

        Color activeColor = game.getColorOnMove();
        if (game.getPosition().get(square).getColor() != activeColor) return false;

        String activePlayer = getActivePlayerName();
        if (activePlayer == null) return false;

        return activePlayer.equals(player.method_5820());
    }

    public void setMinedSquare(Coordinate square) {
        if (hasPlayerMinedPiece())
            ImmersiveChess.LOGGER.error("Trying to set piece mined when there already was one mined");
        minedSquare = square;
        method_80();

        // also set correct piece to BoardBlockEntity for rendering
        if (world.method_8321(board.getPos(square)) instanceof BoardBlockEntity boardBlockEntity) {
            boardBlockEntity.setPiece(getPiece(square));
        }
    }

    private void clearMinedSquare() {
        // also clear piece of BoardBlockEntity for rendering
        if (minedSquare != null && world.method_8321(board.getPos(minedSquare)) instanceof BoardBlockEntity boardBlockEntity) {
            boardBlockEntity.setPiece(null);
        }

        minedSquare = null;
        method_80();
    }

    public void setMinedSquare(class_2338 pos) {
        Coordinate square = board.getSquare(pos.method_10074());
        if (square == null) {
            ImmersiveChess.LOGGER.error("Failed to convert world pos to chess square " + pos);
            return;
        }
        setMinedSquare(square);
    }


    private Displacement createDisplacement(Coordinate source, Coordinate destination) {
        // if not a promotion -> do not provide piece
        Moveable sourcePiece = game.getPosition().get(source);
        return new Displacement(sourcePiece, source, destination);
    }

    /**
     * This is the main way in which legal moves are checked because it can handle the case of pawn promotion.
     * With pawn promotion some pieces are not allowed to be placed back at the source.
     */
    public List<Coordinate> getLegalDestinations(@Nullable Coordinate square) {
        if (square == null) return Collections.emptyList();
        List<Move> moves = game.getAvailableMoves(square.toString());
        if (moves == null) return Collections.emptyList();

        List<Coordinate> destinations = new ArrayList<>(moves.stream()
                .map(m -> m.getDisplacement().getNewLocation())
                .distinct()
                .toList());
        destinations.add(square);
        return destinations;
    }

    /**
     * Use stored getLegalDestinations of piece instead.
     */
    @Deprecated
    public boolean isMoveLegal(Coordinate source, Coordinate destination) {
        // placing back in original position is also allowed
        if (source.equals(destination)) return true;
        Displacement displacement = createDisplacement(source, destination);
        List<Move> validMoves = findMoves(displacement);
        return !validMoves.isEmpty();
    }

    private boolean equals(Displacement d1, Displacement d2) {
        if (!d1.getOldLocation().equals(d2.getOldLocation())) return false;
        if (!d1.getNewLocation().equals(d2.getNewLocation())) return false;
        if (Piece.fromMoveable(d1.getMoveable()) != Piece.fromMoveable(d2.getMoveable())) return false;

        return true;
    }

    private List<Move> findMoves(Displacement displacement) {
        List<Move> moves = game.getAvailableMoves(displacement.getMoveable());
        if (moves == null) return Collections.emptyList();
        return game.getAvailableMoves(displacement.getMoveable()).stream()
                .filter(m -> equals(m.getDisplacement(), displacement))
                .toList();
    }

    @Nullable
    private Move findMove(Displacement displacement, Piece promotion) {
        Move move = findMoves(displacement).stream().findFirst().orElse(null);
        if (move != null && move.isPromotionNeeded())
            move.setPromotion(promotion.createMoveable());
        return move;
    }

    /**
     * Piece required for promotion
     */
    public boolean doMove(Coordinate source, Coordinate destination, Piece promotion) {
        // placing back in original position is also allowed
        if (source.equals(destination)) {
            clearMinedSquare();
            return true;
        }

        // find intended move
        Displacement displacement = createDisplacement(source, destination);
        Move move = findMove(displacement, promotion);
        if (move == null) {
            ImmersiveChess.LOGGER.error("Could not find valid move for displacement " + displacement.getOldLocation() + " " + displacement.getNewLocation());
            return false;
        }


        // perform the actual move
        try {
            JChessGame.Status status = game.play(move);

            if (move.getLinkedDisplacements() != null)
                move.getLinkedDisplacements().forEach(this::executeDisplacement);

            // en passant happened
            if (move.getCapturedEntity() != game.getPosition().getPreviousPosition().get(move.getDisplacement().getNewLocation())) {
                // perform additional displacement of captured piece to null so it will be removed
                executeDisplacement(new Displacement(move.getCapturedEntity(), game.getPosition().getPreviousPosition().getLocation(move.getCapturedEntity()), null));
            }

            // check if king in check
            Coordinate kingSquare = findKingSquare(getColorOnMove());
            if (world.method_8321(board.getPos(kingSquare)) instanceof BoardBlockEntity boardBlockEntity) {
                boolean kingUnderAttack = game.getPosition().canBeReached(kingSquare, getColorOnMove().opposite());
                boardBlockEntity.setInCheck(kingUnderAttack);
            }

            // check for end of game
            if (status.isFinished()) {
                onGameEnded();
            } else {
                // clear king in check
                Coordinate ownKingSquare = findPreviousKingSquare(getColorOnMove().opposite());
                if (world.method_8321(board.getPos(ownKingSquare)) instanceof BoardBlockEntity boardBlockEntity) {
                    boardBlockEntity.setInCheck(false);
                }
            }

            // set structure again (could have changed while mined)
            updatePieceStructure(move.getDisplacement().getNewLocation());
            clearMinedSquare();

            // clear offer for draw when a move is played
            drawOfferedBy = null;
            method_80();
            return true;
        } catch (IllegalStateException e) {
            ImmersiveChess.LOGGER.error("Tried to perform move '" + move + "' but failed", e);
            return false;
        }
    }

    private Coordinate findKingSquare(Color color) {
        return findKingSquare(game.getPosition(), color);
    }

    private Coordinate findPreviousKingSquare(Color color) {
        return findKingSquare(game.getPosition().getPreviousPosition(), color);
    }

    private static Coordinate findKingSquare(Position position, Color color) {
        for (Map.Entry<Coordinate, Moveable> entry : position.getMoveables().entrySet()) {
            Moveable moveable = entry.getValue();
            if (moveable.getColor() == color && moveable instanceof King) {
                return entry.getKey();
            }
        }
        throw new IllegalStateException("No king found on board for color " + color);
    }

    /**
     * Executes move of piece based on displacement. Can be used for complex operations such as en passant and castling or also by AI
     * Note: Moveable piece not taken into account.
     */
    public boolean executeDisplacement(Displacement displacement) {
        class_2338 sourcePos = board.getPos(displacement.getOldLocation()).method_10084();

        // remove origin piece
        breakPiece(displacement.getOldLocation());

        // place destination piece (optional)
        if (displacement.getNewLocation() != null) {
            Piece piece = Piece.fromMoveable(displacement.getMoveable());
            placePiece(displacement.getNewLocation(), piece);
        }

        return true;
    }

    private void placePiece(Coordinate square, Piece piece) {
        class_2338 destPos = board.getPos(square).method_10084();
        class_2350 facing = piece.getColor() == Color.WHITE ? board.getWhitePlayDirection() : board.getWhitePlayDirection().method_10153();
        class_2680 state = piece.getBlockState(facing);

        if (world.method_8320(destPos).equals(state))
            return;

        world.method_8652(destPos, state, class_2248.field_31036);

        class_2498 blockSoundGroup = state.method_26231();
        world.method_8396(null, destPos, blockSoundGroup.method_10598(), class_3419.field_15245, (blockSoundGroup.method_10597() + 1.0f) / 2.0f, blockSoundGroup.method_10599() * 0.8f);

        class_2586 blockEntity = world.method_8321(destPos);
        if (blockEntity instanceof StructureRenderedBlockEntity pieceBlockEntity) {
            pieceBlockEntity.setStructure(getStructure(piece));
        }
    }

    private void breakPiece(Coordinate square) {
        class_2338 destPos = board.getPos(square).method_10084();
        world.method_22352(destPos, false);
    }

    public void undoMove() {
        game.back();
        method_80();
    }

    private static ChessGameState fromNbt(class_3218 world, class_2487 nbt) {
        if (!nbt.method_10545(GAME_KEY))
            return null;

        if (!nbt.method_10545(BOARD_KEY)) return null;
        MultiblockBoard board = MultiblockBoard.fromNbt(nbt.method_10562(BOARD_KEY));
        if (board == null) return null;

        String pgn = nbt.method_10558(GAME_KEY);
        try {
            Coordinate minedSquare = nbt.method_10545(MINED_KEY) ? new Coordinate(nbt.method_10558(MINED_KEY)) : null;
            Color drawOfferedBy = nbt.method_10545(DRAW_OFFER_KEY) ? Color.valueOf(nbt.method_10558(DRAW_OFFER_KEY)) : null;

            // Should only be one game per file
            JChessGame game = new PGNReader(new StringReader(pgn)).readGame();
            ChessGameState state = new ChessGameState(world, game, board, minedSquare, drawOfferedBy);

            if (nbt.method_10545(WHITE_PLAYER_STRUCTURES_KEY))
                state.playerStructures.put(Color.WHITE, StructureMap.fromNbt(nbt.method_10562(WHITE_PLAYER_STRUCTURES_KEY)));
            if (nbt.method_10545(BLACK_PLAYER_STRUCTURES_KEY))
                state.playerStructures.put(Color.BLACK, StructureMap.fromNbt(nbt.method_10562(BLACK_PLAYER_STRUCTURES_KEY)));

            if (nbt.method_10545(WHITE_PIECES_RENDER_OPTION_KEY))
                state.renderOptions.put(Color.WHITE, PieceRenderOption.valueOf(nbt.method_10558(WHITE_PIECES_RENDER_OPTION_KEY)));
            if (nbt.method_10545(BLACK_PIECES_RENDER_OPTION_KEY))
                state.renderOptions.put(Color.BLACK, PieceRenderOption.valueOf(nbt.method_10558(BLACK_PIECES_RENDER_OPTION_KEY)));

            return state;
        } catch (IOException e) {
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            ImmersiveChess.LOGGER.error("Failed to load chess game", e);
            return null;
        }
    }

    @Override
    public class_2487 method_75(class_2487 nbt) {
        if (drawOfferedBy != null)
            nbt.method_10582(DRAW_OFFER_KEY, drawOfferedBy.toString());

        nbt.method_10566(BOARD_KEY, board.writeNbt(new class_2487()));
        nbt.method_10566(WHITE_PLAYER_STRUCTURES_KEY, getStructuresOfPlayer(Color.WHITE).writeNbt(new class_2487()));
        nbt.method_10566(BLACK_PLAYER_STRUCTURES_KEY, getStructuresOfPlayer(Color.BLACK).writeNbt(new class_2487()));

        nbt.method_10582(WHITE_PIECES_RENDER_OPTION_KEY, getRenderOption(Color.WHITE).toString());
        nbt.method_10582(BLACK_PIECES_RENDER_OPTION_KEY, getRenderOption(Color.BLACK).toString());

        nbt.method_10582(GAME_KEY, getPgn());
        if (minedSquare != null)
            nbt.method_10582(MINED_KEY, minedSquare.toString());
        return nbt;
    }

    private String getSavePath() {
        return LOCATION + getGameSaveId();
    }

    public String getGameSaveId() {
        String date = game.getMetadata().getOrDefault(DATE_KEY, "unknown");
        return date + "/" + getGameId();
    }

    private String getPgn() {
        StringWriter strWriter = new StringWriter();
        new PGNWriter(strWriter).writeGame(game);
        return strWriter.toString();
    }

    @Override
    public void method_17919(File file) {
        file.getParentFile().mkdirs();

        // write pgn to file
        File pgnFile = new File(file.getParentFile(), file.getName().replace(".dat", ".pgn"));
        try (FileWriter outputStream = new FileWriter(pgnFile)) {
            outputStream.write(getPgn());
        } catch (IOException e) {
            ImmersiveChess.LOGGER.error("Could not save pgn file", e);
        }

        super.method_17919(file);

        // write nbt or remove data if no longer needed (disabled for now as gameState could still be attached to a board)
//        if (!getStatus().isFinished())
//            super.save(file);
//        else
//            file.delete();
    }
}
