package cz.yorick.command;

import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.builder.RequiredArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.suggestion.Suggestions;
import com.mojang.brigadier.suggestion.SuggestionsBuilder;
import cz.yorick.resources.Util;
import cz.yorick.resources.type.SimpleReloadableResource;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import net.minecraft.class_124;
import net.minecraft.class_2168;
import net.minecraft.class_2172;
import net.minecraft.class_2232;
import net.minecraft.class_2561;
import net.minecraft.class_2960;

public class ReloadConfigResourcesCommand<S extends class_2172> {
    private final BiConsumer<S, class_2561> feedbackSender;
    private ReloadConfigResourcesCommand(CommandDispatcher<S> dispatcher, String commandName, BiConsumer<S, class_2561> feedbackSender, Predicate<S> canExecute) {
        this.feedbackSender = feedbackSender;
        dispatcher.register(literal(commandName).requires(canExecute)
                .then(literal("all")
                        .executes(context -> reloadAllConfigs(context.getSource()))
                )
                .then(literal("only")
                        .then(argument("config", class_2232.method_9441()).suggests(this::suggestConfigResource)
                                .executes(context -> reloadConfig(context.getSource(), context.getArgument("config", class_2960.class)))
                        )
                )
        );
    }

    private LiteralArgumentBuilder<S> literal(String name) {
        return LiteralArgumentBuilder.literal(name);
    }

    private <T> RequiredArgumentBuilder<S, T> argument(String name, ArgumentType<T> type) {
        return RequiredArgumentBuilder.argument(name, type);
    }

    private CompletableFuture<Suggestions> suggestConfigResource(CommandContext<S> context, SuggestionsBuilder builder) {
        return CommandUtil.suggestMatching(Util.getReloadableResourceKeys().stream().map(class_2960::toString).toList(), builder);
    }

    private int reloadAllConfigs(S source) {
        Util.getReloadableResources().forEach(reloadableConfig -> reloadableConfig.reload(error -> handleError(source, error)));
        sendSuccess(source, "Reloaded config resources");
        return Command.SINGLE_SUCCESS;
    }

    private int reloadConfig(S source, class_2960 id) {
        SimpleReloadableResource<?> config = Util.getReloadableResource(id);
        if(config == null) {
            sendError(source, "Config resource " + id.toString() + " does not exist or is unreloadable");
            return 0;
        }

        config.reload(error -> handleError(source, error));
        sendSuccess(source, "Reloaded the config resource " + id.toString());
        return Command.SINGLE_SUCCESS;
    }

    private void handleError(S source, Exception error) {
        sendError(source, error.getMessage());
        Throwable throwable = error.getCause();
        while (throwable != null) {
            sendError(source, "Caused by " + throwable.getClass().getName() + ":");
            sendError(source, throwable.getMessage());
            throwable = throwable.getCause();
        }
    }

    private void sendSuccess(S source, String message) {
        this.feedbackSender.accept(source, class_2561.method_43470(message).method_27692(class_124.field_1060));
    }

    private void sendError(S source, String error) {
        this.feedbackSender.accept(source, class_2561.method_43470(error).method_27692(class_124.field_1061));
    }

    public static void registerServer(CommandDispatcher<class_2168> dispatcher) {
        new ReloadConfigResourcesCommand<>(dispatcher, "reloadServerConfigs", class_2168::method_45068, source -> source.method_9259(2));
    }
    public static<S extends class_2172> void registerClient(CommandDispatcher<S> dispatcher, BiConsumer<S, class_2561> feedbackConsumer) {
        new ReloadConfigResourcesCommand<>(dispatcher, "reloadConfigs", feedbackConsumer, source -> true);
    }
}
