/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.autolang.config;

import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.texboobcat.autolang.libs.gson.Gson;
import org.texboobcat.autolang.libs.gson.GsonBuilder;
import org.texboobcat.autolang.libs.gson.JsonArray;
import org.texboobcat.autolang.libs.gson.JsonElement;
import org.texboobcat.autolang.libs.gson.JsonObject;
import org.texboobcat.autolang.libs.gson.JsonParser;
import org.texboobcat.autolang.locale.Language;

public class ConfigManager {
    private final Path configFile;
    private final Logger logger;
    private final Gson gson = new GsonBuilder().setPrettyPrinting().create();
    private Map<String, Language> languagesByCode = new HashMap<String, Language>();
    private Language defaultLanguage;
    private boolean languagesAutoFromLibreTranslate;
    private Set<String> languagesInclude = new HashSet<String>();
    private Set<String> languagesExclude = new HashSet<String>();
    private int maxTranslationThreads = 4;
    private int translationTimeoutMs = 3000;
    private boolean cacheEnabled = true;
    private int cacheSize = 500;
    private int cacheTtlSeconds = 0;
    private boolean autodetectEnabled = true;
    private int minMessagesBeforeDetect = 3;
    private double autodetectConfidenceThreshold = 0.8;
    private boolean clientLocaleEnabled = true;
    private boolean clientLocaleAutoAccept = true;
    private String storageType = "sqlite";
    private String sqliteFile = "config/autolang/languages.db";
    private boolean logMessages = false;
    private String detectorType = "lingua";
    private boolean libreTranslateEnabled = false;
    private String libreTranslateBaseUrl = "http://localhost:5000";
    private boolean libreTranslateAllowRemote = false;
    private String libreTranslateApiKey = "";
    private boolean localModelEnabled = false;
    private String localModelCommand = "";

    public ConfigManager(Path configFile, Logger logger) {
        this.configFile = configFile;
        this.logger = logger;
    }

    public void load() {
        try {
            Files.createDirectories(this.configFile.getParent(), new FileAttribute[0]);
            if (!Files.exists(this.configFile, new LinkOption[0])) {
                this.saveDefaults();
            }
            String content = Files.readString(this.configFile, StandardCharsets.UTF_8);
            JsonObject config = JsonParser.parseString(content).getAsJsonObject();
            this.loadBasicSettings(config);
            this.loadLibreTranslateSettings(config);
            this.loadLanguages(config);
        }
        catch (Exception e) {
            this.logger.error("Failed to load config, using defaults", (Throwable)e);
            this.loadDefaults();
        }
    }

    private void loadBasicSettings(JsonObject config) {
        if (config.has("default-language")) {
            String defCode = config.get("default-language").getAsString();
            this.defaultLanguage = new Language(defCode, defCode);
        } else {
            this.defaultLanguage = new Language("en", "English");
        }
        if (config.has("max-translation-threads")) {
            this.maxTranslationThreads = config.get("max-translation-threads").getAsInt();
        }
        if (config.has("translation-timeout-ms")) {
            this.translationTimeoutMs = config.get("translation-timeout-ms").getAsInt();
        }
        if (config.has("cache")) {
            JsonObject cache2 = config.getAsJsonObject("cache");
            this.cacheEnabled = cache2.has("enabled") ? cache2.get("enabled").getAsBoolean() : true;
            this.cacheSize = cache2.has("size") ? cache2.get("size").getAsInt() : 500;
            int n = this.cacheTtlSeconds = cache2.has("ttl-seconds") ? cache2.get("ttl-seconds").getAsInt() : 0;
        }
        if (config.has("autodetect")) {
            JsonObject auto = config.getAsJsonObject("autodetect");
            this.autodetectEnabled = auto.has("enabled") ? auto.get("enabled").getAsBoolean() : true;
            this.minMessagesBeforeDetect = auto.has("min-messages") ? auto.get("min-messages").getAsInt() : 3;
            double d = this.autodetectConfidenceThreshold = auto.has("confidence-threshold") ? auto.get("confidence-threshold").getAsDouble() : 0.8;
        }
        if (config.has("client-locale")) {
            JsonObject cl = config.getAsJsonObject("client-locale");
            this.clientLocaleEnabled = cl.has("enabled") ? cl.get("enabled").getAsBoolean() : true;
            boolean bl = this.clientLocaleAutoAccept = cl.has("auto-accept") ? cl.get("auto-accept").getAsBoolean() : true;
        }
        if (config.has("storage")) {
            JsonObject storage2 = config.getAsJsonObject("storage");
            this.storageType = storage2.has("type") ? storage2.get("type").getAsString() : "sqlite";
            String string = this.sqliteFile = storage2.has("sqlite-file") ? storage2.get("sqlite-file").getAsString() : "config/autolang/languages.db";
        }
        if (config.has("privacy")) {
            JsonObject privacy = config.getAsJsonObject("privacy");
            boolean bl = this.logMessages = privacy.has("log-messages") ? privacy.get("log-messages").getAsBoolean() : false;
        }
        if (config.has("detector")) {
            JsonObject detector = config.getAsJsonObject("detector");
            String string = this.detectorType = detector.has("type") ? detector.get("type").getAsString() : "lingua";
        }
        if (config.has("local-model")) {
            JsonObject lm = config.getAsJsonObject("local-model");
            this.localModelEnabled = lm.has("enabled") ? lm.get("enabled").getAsBoolean() : false;
            this.localModelCommand = lm.has("command") ? lm.get("command").getAsString() : "";
        }
    }

    private void loadLanguages(JsonObject config) {
        this.languagesByCode.clear();
        if (config.has("languages")) {
            JsonElement langsElem = config.get("languages");
            if (langsElem.isJsonArray()) {
                JsonArray langs = langsElem.getAsJsonArray();
                for (JsonElement elem : langs) {
                    if (!elem.isJsonObject()) continue;
                    JsonObject langObj = elem.getAsJsonObject();
                    String code = langObj.get("code").getAsString().toLowerCase(Locale.ROOT);
                    String name = langObj.has("name") ? langObj.get("name").getAsString() : code;
                    this.languagesByCode.put(code, new Language(code, name));
                }
            } else if (langsElem.isJsonObject()) {
                JsonObject langsObj = langsElem.getAsJsonObject();
                boolean bl = this.languagesAutoFromLibreTranslate = langsObj.has("auto-from-libretranslate") && langsObj.get("auto-from-libretranslate").getAsBoolean();
                if (langsObj.has("include") && langsObj.get("include").isJsonArray()) {
                    langsObj.getAsJsonArray("include").forEach(e -> this.languagesInclude.add(e.getAsString().toLowerCase(Locale.ROOT)));
                }
                if (langsObj.has("exclude") && langsObj.get("exclude").isJsonArray()) {
                    langsObj.getAsJsonArray("exclude").forEach(e -> this.languagesExclude.add(e.getAsString().toLowerCase(Locale.ROOT)));
                }
                if (langsObj.has("static") && langsObj.get("static").isJsonArray()) {
                    JsonArray staticLangs = langsObj.getAsJsonArray("static");
                    for (JsonElement elem : staticLangs) {
                        if (!elem.isJsonObject()) continue;
                        JsonObject langObj = elem.getAsJsonObject();
                        String code = langObj.get("code").getAsString().toLowerCase(Locale.ROOT);
                        String name = langObj.has("name") ? langObj.get("name").getAsString() : code;
                        this.languagesByCode.put(code, new Language(code, name));
                    }
                }
                if (this.languagesAutoFromLibreTranslate && this.libreTranslateEnabled) {
                    this.fetchLanguagesFromLibreTranslate();
                }
            }
        }
        if (this.languagesByCode.isEmpty()) {
            this.languagesByCode.put("en", new Language("en", "English"));
            this.languagesByCode.put("es", new Language("es", "Spanish"));
            this.languagesByCode.put("fr", new Language("fr", "French"));
        }
    }

    private void loadLibreTranslateSettings(JsonObject config) {
        if (config.has("libretranslate")) {
            JsonObject lt = config.getAsJsonObject("libretranslate");
            this.libreTranslateEnabled = lt.has("enabled") ? lt.get("enabled").getAsBoolean() : false;
            this.libreTranslateBaseUrl = lt.has("base-url") ? lt.get("base-url").getAsString() : "http://localhost:5000";
            this.libreTranslateAllowRemote = lt.has("allow-remote") ? lt.get("allow-remote").getAsBoolean() : false;
            this.libreTranslateApiKey = lt.has("api-key") ? lt.get("api-key").getAsString() : "";
        }
    }

    private void fetchLanguagesFromLibreTranslate() {
        try {
            JsonElement parsed;
            boolean isLocal;
            String lower = this.libreTranslateBaseUrl.toLowerCase(Locale.ROOT);
            boolean bl = isLocal = lower.contains("127.0.0.1") || lower.contains("localhost") || lower.contains("0.0.0.0");
            if (!isLocal && !this.libreTranslateAllowRemote) {
                this.logger.warn("Skipping LibreTranslate languages fetch: base-url is remote and allow-remote is false");
                return;
            }
            String url = this.buildLibreTranslateUrl("/languages");
            this.logger.info("Fetching languages from LibreTranslate: {}", (Object)url);
            HttpClient client = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(5L)).build();
            HttpRequest.Builder requestBuilder = HttpRequest.newBuilder().uri(URI.create(url)).timeout(Duration.ofSeconds(5L)).header("Accept", "application/json").GET();
            HttpResponse<String> response = client.send(requestBuilder.build(), HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8));
            String body = response.body();
            if (response.statusCode() != 200) {
                this.logger.warn("/languages returned status {}, trying /v2/languages...", (Object)response.statusCode());
                String url2 = this.buildLibreTranslateUrl("/v2/languages");
                HttpRequest.Builder requestBuilder2 = HttpRequest.newBuilder().uri(URI.create(url2)).timeout(Duration.ofSeconds(5L)).header("Accept", "application/json").GET();
                response = client.send(requestBuilder2.build(), HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8));
                body = response.body();
                if (response.statusCode() != 200) {
                    int previewLen = body == null ? 0 : Math.min(256, body.length());
                    this.logger.error("LibreTranslate languages fetch failed: status={}, body-preview={}", (Object)response.statusCode(), (Object)(body == null ? "" : body.substring(0, previewLen)));
                    return;
                }
            }
            if (!(parsed = JsonParser.parseString(body)).isJsonArray()) {
                this.logger.warn("LibreTranslate languages response is not a JSON array");
                return;
            }
            JsonArray langs = parsed.getAsJsonArray();
            int count = 0;
            for (JsonElement elem : langs) {
                String name;
                JsonObject langObj;
                if (!elem.isJsonObject() || !(langObj = elem.getAsJsonObject()).has("code")) continue;
                String code = langObj.get("code").getAsString().toLowerCase(Locale.ROOT).trim();
                String string = name = langObj.has("name") ? langObj.get("name").getAsString() : code;
                if (code.isEmpty() || !this.languagesInclude.isEmpty() && !this.languagesInclude.contains(code) || this.languagesExclude.contains(code)) continue;
                this.languagesByCode.put(code, new Language(code, name));
                ++count;
            }
            this.logger.info("Fetched {} languages from LibreTranslate (total available: {})", (Object)count, (Object)langs.size());
        }
        catch (Exception e) {
            this.logger.warn("Failed to fetch languages from LibreTranslate: {}: {}", (Object)e.getClass().getSimpleName(), (Object)e.getMessage());
        }
    }

    private String buildLibreTranslateUrl(String path) {
        String base = this.libreTranslateBaseUrl.endsWith("/") ? this.libreTranslateBaseUrl.substring(0, this.libreTranslateBaseUrl.length() - 1) : this.libreTranslateBaseUrl;
        Object p = path.startsWith("/") ? path : "/" + path;
        String url = base + (String)p;
        if (this.libreTranslateApiKey != null && !this.libreTranslateApiKey.isEmpty()) {
            try {
                String sep = url.contains("?") ? "&" : "?";
                url = url + sep + "api_key=" + URLEncoder.encode(this.libreTranslateApiKey, StandardCharsets.UTF_8);
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return url;
    }

    private void saveDefaults() {
        JsonObject config = new JsonObject();
        config.addProperty("default-language", "en");
        config.addProperty("max-translation-threads", 4);
        config.addProperty("translation-timeout-ms", 3000);
        JsonObject cache2 = new JsonObject();
        cache2.addProperty("enabled", true);
        cache2.addProperty("size", 500);
        cache2.addProperty("ttl-seconds", 0);
        config.add("cache", cache2);
        JsonObject autodetect = new JsonObject();
        autodetect.addProperty("enabled", true);
        autodetect.addProperty("min-messages", 3);
        autodetect.addProperty("confidence-threshold", 0.8);
        config.add("autodetect", autodetect);
        JsonObject clientLocale = new JsonObject();
        clientLocale.addProperty("enabled", true);
        clientLocale.addProperty("auto-accept", true);
        config.add("client-locale", clientLocale);
        JsonObject storage2 = new JsonObject();
        storage2.addProperty("type", "sqlite");
        storage2.addProperty("sqlite-file", "config/autolang/languages.db");
        config.add("storage", storage2);
        JsonObject detector = new JsonObject();
        detector.addProperty("type", "lingua");
        config.add("detector", detector);
        JsonObject privacy = new JsonObject();
        privacy.addProperty("log-messages", false);
        config.add("privacy", privacy);
        JsonObject libretranslate = new JsonObject();
        libretranslate.addProperty("enabled", true);
        libretranslate.addProperty("base-url", "http://138.197.218.207:5000");
        libretranslate.addProperty("allow-remote", true);
        libretranslate.addProperty("api-key", "");
        config.add("libretranslate", libretranslate);
        JsonObject localModel = new JsonObject();
        localModel.addProperty("enabled", false);
        localModel.addProperty("command", "");
        config.add("local-model", localModel);
        JsonObject langsConfig = new JsonObject();
        langsConfig.addProperty("auto-from-libretranslate", true);
        langsConfig.add("include", new JsonArray());
        langsConfig.add("exclude", new JsonArray());
        JsonArray staticLangs = new JsonArray();
        JsonObject en = new JsonObject();
        en.addProperty("code", "en");
        en.addProperty("name", "English");
        staticLangs.add(en);
        JsonObject es = new JsonObject();
        es.addProperty("code", "es");
        es.addProperty("name", "Spanish");
        staticLangs.add(es);
        JsonObject fr = new JsonObject();
        fr.addProperty("code", "fr");
        fr.addProperty("name", "French");
        staticLangs.add(fr);
        langsConfig.add("static", staticLangs);
        config.add("languages", langsConfig);
        try {
            Files.writeString(this.configFile, (CharSequence)this.gson.toJson(config), StandardCharsets.UTF_8, new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.error("Failed to save default config", (Throwable)e);
        }
    }

    private void loadDefaults() {
        this.defaultLanguage = new Language("en", "English");
        this.languagesByCode.put("en", new Language("en", "English"));
        this.languagesByCode.put("es", new Language("es", "Spanish"));
        this.languagesByCode.put("fr", new Language("fr", "French"));
    }

    public Language getDefaultLanguage() {
        return this.defaultLanguage;
    }

    public List<Language> getLanguages() {
        return new ArrayList<Language>(this.languagesByCode.values());
    }

    public Language getLanguageByCode(String code) {
        return this.languagesByCode.get(code.toLowerCase(Locale.ROOT));
    }

    public boolean isLanguageSupported(String code) {
        return this.languagesByCode.containsKey(code.toLowerCase(Locale.ROOT));
    }

    public int getMaxTranslationThreads() {
        return this.maxTranslationThreads;
    }

    public int getTranslationTimeoutMs() {
        return this.translationTimeoutMs;
    }

    public boolean isCacheEnabled() {
        return this.cacheEnabled;
    }

    public int getCacheSize() {
        return this.cacheSize;
    }

    public int getCacheTtlSeconds() {
        return this.cacheTtlSeconds;
    }

    public boolean isAutodetectEnabled() {
        return this.autodetectEnabled;
    }

    public int getMinMessagesBeforeDetect() {
        return this.minMessagesBeforeDetect;
    }

    public double getAutodetectConfidenceThreshold() {
        return this.autodetectConfidenceThreshold;
    }

    public boolean isClientLocaleEnabled() {
        return this.clientLocaleEnabled;
    }

    public boolean isClientLocaleAutoAccept() {
        return this.clientLocaleAutoAccept;
    }

    public String getStorageType() {
        return this.storageType;
    }

    public String getSqliteFile() {
        return this.sqliteFile;
    }

    public boolean isLogMessages() {
        return this.logMessages;
    }

    public String getDetectorType() {
        return this.detectorType;
    }

    public boolean isLibreTranslateEnabled() {
        return this.libreTranslateEnabled;
    }

    public String getLibreTranslateBaseUrl() {
        return this.libreTranslateBaseUrl;
    }

    public boolean isLibreTranslateAllowRemote() {
        return this.libreTranslateAllowRemote;
    }

    public String getLibreTranslateApiKey() {
        return this.libreTranslateApiKey;
    }

    public boolean isLocalModelEnabled() {
        return this.localModelEnabled;
    }

    public String getLocalModelCommand() {
        return this.localModelCommand;
    }
}

