/*
 * Decompiled with CFR 0.152.
 */
package me.autolang.translate;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import me.autolang.AutoLang;
import me.autolang.config.ConfigManager;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitTask;

public class LibreTranslateProcessManager {
    private final AutoLang plugin;
    private final ConfigManager cfg;
    private final Logger log;
    private final HttpClient http;
    private volatile Process process;
    private volatile BukkitTask watchdogTask;
    private volatile int consecutiveFailures;
    private final File venvDir;

    public LibreTranslateProcessManager(AutoLang plugin, ConfigManager cfg) {
        this.plugin = plugin;
        this.cfg = cfg;
        this.log = plugin.getLogger();
        this.http = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(3L)).build();
        this.venvDir = new File(plugin.getDataFolder(), "lt-venv");
    }

    public synchronized boolean start() {
        if (!this.cfg.isLibreTranslateManagedEnabled()) {
            this.log.info("LibreTranslate managed mode is disabled; not starting process.");
            return false;
        }
        if (this.process != null && this.process.isAlive()) {
            this.log.info("LibreTranslate process already running at " + this.getBaseUrl());
            return true;
        }
        String configuredCommand = this.cfg.getLibreTranslateManagedCommand();
        if ((configuredCommand == null || configuredCommand.isBlank()) && this.cfg.isLibreTranslateManagedBootstrapEnabled()) {
            this.log.info("Bootstrapping LibreTranslate environment (this may take several minutes)...");
            this.plugin.getAsyncExecutor().supplyAsync(() -> this.bootstrapAndGetCommand()).whenComplete((cmdStr, ex) -> {
                if (ex != null || cmdStr == null || cmdStr.isBlank()) {
                    this.log.warning("LibreTranslate bootstrap failed: " + (ex != null ? ex.getMessage() : "unknown error"));
                    return;
                }
                this.startWithCommand((String)cmdStr);
            });
            return true;
        }
        int port = this.cfg.getLibreTranslateManagedPort();
        List<String> cmd = LibreTranslateProcessManager.parseCommand(configuredCommand);
        cmd.add("--host");
        cmd.add("127.0.0.1");
        cmd.add("--port");
        cmd.add(String.valueOf(port));
        String extra = this.cfg.getLibreTranslateManagedExtraArgs();
        if (extra != null && !extra.isBlank()) {
            cmd.addAll(LibreTranslateProcessManager.parseCommand(extra));
        }
        ProcessBuilder pb = new ProcessBuilder(cmd);
        pb.redirectErrorStream(true);
        if (this.cfg.getLibreTranslateManagedLoadOnly() != null && !this.cfg.getLibreTranslateManagedLoadOnly().isBlank()) {
            pb.environment().put("LT_LOAD_ONLY", this.cfg.getLibreTranslateManagedLoadOnly());
        }
        pb.environment().put("LT_DOWNLOAD_MODELS", String.valueOf(this.cfg.isLibreTranslateManagedDownloadModels()));
        return this.startProcess(pb);
    }

    public synchronized boolean stop() {
        if (this.process == null) {
            return false;
        }
        boolean wasAlive = this.process.isAlive();
        try {
            this.process.destroy();
            if (!this.process.waitFor(3L, TimeUnit.SECONDS)) {
                this.process.destroyForcibly();
            }
        }
        catch (InterruptedException ignored) {
            Thread.currentThread().interrupt();
        }
        finally {
            this.process = null;
            this.cfg.setLibreTranslateBaseUrlOverride(null);
            this.stopWatchdog();
        }
        if (wasAlive) {
            this.log.info("Stopped managed LibreTranslate.");
        }
        return wasAlive;
    }

    public boolean isRunning() {
        return this.process != null && this.process.isAlive();
    }

    public String getBaseUrl() {
        return "http://127.0.0.1:" + this.cfg.getLibreTranslateManagedPort();
    }

    public CompletableFuture<Boolean> healthCheck() {
        String base = this.cfg.getLibreTranslateBaseUrl();
        URI uri = URI.create(base.endsWith("/") ? base + "languages" : base + "/languages");
        HttpRequest req = HttpRequest.newBuilder(uri).timeout(Duration.ofSeconds(3L)).GET().build();
        return ((CompletableFuture)this.http.sendAsync(req, HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8)).thenApply(resp -> resp.statusCode() == 200)).exceptionally(ex -> false);
    }

    public CompletableFuture<Boolean> waitUntilReady(Duration timeout, Duration pollInterval) {
        long start = System.nanoTime();
        CompletableFuture<Boolean> cf = new CompletableFuture<Boolean>();
        this.pollUntilReady(cf, start, timeout, pollInterval);
        return cf;
    }

    private void pollUntilReady(CompletableFuture<Boolean> cf, long startNanos, Duration timeout, Duration pollInterval) {
        if (!this.isRunning()) {
            cf.complete(false);
            return;
        }
        this.healthCheck().whenComplete((ok, ex) -> {
            if (ok != null && ok.booleanValue()) {
                cf.complete(true);
                return;
            }
            if (System.nanoTime() - startNanos > timeout.toNanos()) {
                cf.complete(false);
            } else {
                this.plugin.getServer().getScheduler().runTaskLater((Plugin)this.plugin, () -> this.pollUntilReady(cf, startNanos, timeout, pollInterval), Math.max(1L, pollInterval.toMillis() / 50L));
            }
        });
    }

    private void streamProcessOutput(InputStream is) {
        this.plugin.getAsyncExecutor().execute(() -> {
            try (BufferedReader br = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8));){
                String line;
                while ((line = br.readLine()) != null) {
                    this.log.info("[LibreTranslate] " + line);
                }
            }
            catch (IOException e) {
                this.log.fine("LibreTranslate output closed: " + e.getMessage());
            }
        });
    }

    private synchronized void startWatchdog() {
        if (!this.cfg.isLibreTranslateManagedWatchdogEnabled()) {
            return;
        }
        this.stopWatchdog();
        long periodTicks = Math.max(1L, (long)this.cfg.getLibreTranslateManagedWatchdogIntervalSeconds() * 20L);
        this.consecutiveFailures = 0;
        this.watchdogTask = this.plugin.getServer().getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> {
            try {
                if (!this.cfg.isLibreTranslateManagedEnabled()) {
                    this.stopWatchdog();
                    return;
                }
                if (this.process == null || !this.process.isAlive()) {
                    this.log.warning("Managed LibreTranslate process not alive. Attempting restart...");
                    this.start();
                    return;
                }
                this.healthCheck().whenComplete((ok, ex) -> {
                    if (ok != null && ok.booleanValue()) {
                        this.consecutiveFailures = 0;
                    } else {
                        ++this.consecutiveFailures;
                        if (this.consecutiveFailures >= this.cfg.getLibreTranslateManagedWatchdogRestartThreshold()) {
                            this.log.warning("LibreTranslate health check failed " + this.consecutiveFailures + " times. Restarting...");
                            this.stop();
                            this.start();
                            this.consecutiveFailures = 0;
                        }
                    }
                });
            }
            catch (Throwable t) {
                this.log.fine("Watchdog tick error: " + t.getMessage());
            }
        }, periodTicks, periodTicks);
    }

    private synchronized void stopWatchdog() {
        if (this.watchdogTask != null) {
            this.watchdogTask.cancel();
            this.watchdogTask = null;
        }
    }

    private boolean startProcess(ProcessBuilder pb) {
        try {
            this.process = pb.start();
            this.cfg.setLibreTranslateBaseUrlOverride(this.getBaseUrl());
            this.log.info("Starting managed LibreTranslate on " + this.getBaseUrl() + " ...");
            this.streamProcessOutput(this.process.getInputStream());
            this.waitUntilReady(Duration.ofSeconds(30L), Duration.ofMillis(500L)).thenAccept(ready -> {
                if (ready.booleanValue()) {
                    this.log.info("LibreTranslate ready at " + this.getBaseUrl());
                } else {
                    this.log.warning("LibreTranslate did not become ready within timeout. It may still start in background.");
                }
            });
            this.startWatchdog();
            return true;
        }
        catch (IOException e) {
            this.log.log(Level.WARNING, "Failed to start LibreTranslate process: " + e.getMessage(), e);
            return false;
        }
    }

    private void startWithCommand(String commandStr) {
        try {
            List<String> cmd = LibreTranslateProcessManager.parseCommand(commandStr);
            cmd.add("--host");
            cmd.add("127.0.0.1");
            cmd.add("--port");
            cmd.add(String.valueOf(this.cfg.getLibreTranslateManagedPort()));
            String extra = this.cfg.getLibreTranslateManagedExtraArgs();
            if (extra != null && !extra.isBlank()) {
                cmd.addAll(LibreTranslateProcessManager.parseCommand(extra));
            }
            ProcessBuilder pb = new ProcessBuilder(cmd);
            pb.redirectErrorStream(true);
            if (this.cfg.getLibreTranslateManagedLoadOnly() != null && !this.cfg.getLibreTranslateManagedLoadOnly().isBlank()) {
                pb.environment().put("LT_LOAD_ONLY", this.cfg.getLibreTranslateManagedLoadOnly());
            }
            pb.environment().put("LT_DOWNLOAD_MODELS", String.valueOf(this.cfg.isLibreTranslateManagedDownloadModels()));
            this.startProcess(pb);
        }
        catch (Throwable t) {
            this.log.log(Level.WARNING, "Failed to start LibreTranslate with bootstrapped command: " + t.getMessage(), t);
        }
    }

    private String bootstrapAndGetCommand() throws Exception {
        if (!this.venvDir.exists() && !this.venvDir.mkdirs()) {
            throw new IOException("Failed to create venv directory: " + this.venvDir.getAbsolutePath());
        }
        String venvPython = this.venvPythonPath();
        File venvPyFile = new File(venvPython);
        if (!venvPyFile.exists()) {
            String py = this.findWorkingPython();
            if (py == null) {
                throw new IOException("No suitable Python interpreter found. Set libretranslate.managed.bootstrap.python in config.yml");
            }
            this.runProcessAndLog(List.of(py, "-m", "venv", this.venvDir.getAbsolutePath()), this.plugin.getDataFolder(), 600L);
        }
        this.runProcessAndLog(List.of(venvPython, "-m", "pip", "install", "--upgrade", "pip", "wheel", "setuptools"), this.plugin.getDataFolder(), 600L);
        this.runProcessAndLog(List.of(venvPython, "-m", "pip", "install", "libretranslate", "argostranslate"), this.plugin.getDataFolder(), 1200L);
        String cmd = venvPython + " -m libretranslate";
        try {
            this.plugin.getConfig().set("libretranslate.managed.command", (Object)cmd);
            this.plugin.saveConfig();
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        return cmd;
    }

    private String venvPythonPath() {
        boolean windows = System.getProperty("os.name").toLowerCase(Locale.ROOT).contains("win");
        if (windows) {
            return new File(new File(this.venvDir, "Scripts"), "python.exe").getAbsolutePath();
        }
        return new File(new File(this.venvDir, "bin"), "python").getAbsolutePath();
    }

    private String findWorkingPython() {
        boolean windows;
        String configured = this.cfg.getLibreTranslateManagedBootstrapPython();
        ArrayList<String> candidates = new ArrayList<String>();
        if (configured != null && !configured.isBlank()) {
            candidates.add(configured);
        }
        if (windows = System.getProperty("os.name").toLowerCase(Locale.ROOT).contains("win")) {
            candidates.add("py -3");
            candidates.add("python");
        } else {
            candidates.add("python3");
            candidates.add("python");
        }
        for (String cand : candidates) {
            try {
                ProcessBuilder pb = new ProcessBuilder(LibreTranslateProcessManager.parseCommand(cand + " --version"));
                pb.redirectErrorStream(true);
                Process p = pb.start();
                if (!p.waitFor(8L, TimeUnit.SECONDS)) {
                    p.destroyForcibly();
                    continue;
                }
                if (p.exitValue() != 0) continue;
                return cand;
            }
            catch (Exception exception) {
            }
        }
        return null;
    }

    private void runProcessAndLog(List<String> command, File cwd, long timeoutSeconds) throws Exception {
        ProcessBuilder pb = new ProcessBuilder(LibreTranslateProcessManager.parseCommand(String.join((CharSequence)" ", command)));
        if (cwd != null) {
            pb.directory(cwd);
        }
        pb.redirectErrorStream(true);
        Process p = pb.start();
        try (BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream(), StandardCharsets.UTF_8));){
            String line;
            while ((line = br.readLine()) != null) {
                this.log.info("[LT Bootstrap] " + line);
            }
        }
        if (!p.waitFor(timeoutSeconds, TimeUnit.SECONDS)) {
            p.destroyForcibly();
            throw new IOException("Command timed out: " + String.join((CharSequence)" ", command));
        }
        int code = p.exitValue();
        if (code != 0) {
            throw new IOException("Command failed (" + code + "): " + String.join((CharSequence)" ", command));
        }
    }

    private static List<String> parseCommand(String cmd) {
        ArrayList<String> tokens = new ArrayList<String>();
        StringBuilder current = new StringBuilder();
        boolean inQuotes = false;
        char quoteChar = '\"';
        for (int i = 0; i < cmd.length(); ++i) {
            char c = cmd.charAt(i);
            if (c == '\"' || c == '\'') {
                if (!inQuotes) {
                    inQuotes = true;
                    quoteChar = c;
                    continue;
                }
                if (quoteChar == c) {
                    inQuotes = false;
                    continue;
                }
                current.append(c);
                continue;
            }
            if (Character.isWhitespace(c) && !inQuotes) {
                if (current.length() <= 0) continue;
                tokens.add(current.toString());
                current.setLength(0);
                continue;
            }
            current.append(c);
        }
        if (current.length() > 0) {
            tokens.add(current.toString());
        }
        return tokens;
    }
}

