package com.petrolpark.core.recipe.ingredient.advanced;

import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.minecraft.util.RandomSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.npc.VillagerTrades.ItemListing;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.trading.MerchantOffer;
import net.minecraft.world.level.storage.loot.LootContext;
import net.minecraft.world.level.storage.loot.functions.LootItemConditionalFunction;
import net.minecraft.world.level.storage.loot.functions.LootItemFunction;
import net.minecraft.world.level.storage.loot.functions.SequenceFunction;

public interface IForcingItemAdvancedIngredient {
    
    /**
     * Attempt to force the output of the given {@link LootItemFunction} to {@link IAdvancedIngredient#test(Object) conform} to this {@link IAdvancedIngredient}.
     * This should not return an ItemStack which could not have been generated by the {@link LootItemFunction} naturally.
     * It is not guaranteed that the returned ItemStack (if there is one) will {@link IAdvancedIngredient#test(Object) conform}, but this method should make a best effort.
     * @param function The {@link LootItemFunction} being called. This will never be a {@link SequenceFunction}, as those are unpacked automatically.
     * Also, if it is a {@link LootItemConditionalFunction}, then it will have already passed all conditions by the time this method is called.
     * @param context
     * @param stack The ItemStack to which the {@link LootItemFunction} is being applied
     * @return Non-{@code null} Optional, filled if an ItemStack could be generated matching this {@link IAdvancedIngredient}, or empty if we could not force the output of the {@link LootItemFunction} successfully and it should be called as normal.
     */
    public @Nonnull Optional<ItemStack> forceLootItemFunction(LootItemFunction function, LootContext context, ItemStack stack);
    
    /**
     * Attempt to force the output of the given {@link LootItemFunction} so it does not {@link IAdvancedIngredient#test(Object) conform} to this {@link IAdvancedIngredient}.
     * This should not return an ItemStack which could not have been generated by the {@link LootItemFunction} naturally.
     * It is not guaranteed that the returned ItemStack (if there is one) won't {@link IAdvancedIngredient#test(Object) conform}, but this method should make a best effort.
     * @param function The {@link LootItemFunction} being called. This will never be a {@link SequenceFunction}, as those are unpacked automatically.
     * Also, if it is a {@link LootItemConditionalFunction}, then it will have already passed all conditions by the time this method is called. 
     * @param context
     * @param stack The ItemStack to which the {@link LootItemFunction} is being applied
     * @return Non-{@code null} Optional, filled if an ItemStack could be generated not matching this {@link IAdvancedIngredient}, or empty if we could not force the output of the {@link LootItemFunction} successfully and it should be called as normal.
     */
    public @Nonnull Optional<ItemStack> forbidLootItemFunction(LootItemFunction function, LootContext context, ItemStack stack);

    /**
     * Attempt to force the output of the given {@link ItemListing} to a {@link MerchantOffer} whose {@link MerchantOffer#getResult() output} {@link IAdvancedIngredient#test(Object) conforms} to this {@link IAdvancedIngredient}.
     * This should not generate a {@link MerchantOffer} which could not have been generated by the {@link ItemListing} naturally.
     * It is not guaranteed that the result of the returned {@link MerchantOffer} (if there is one) will {@link IAdvancedIngredient#test(Object) conform}, but this method should make a best effort.
     * @param listing
     * @param trader
     * @param random
     * @return {@code null} if this {@link IAdvancedIngredient} could not generate a {@link MerchantOffer} with a matching result, an empty Optional if the {@link ItemListing} should generate no {@link MerchantOffer} at all (i.e. {@link ItemListing#getOffer(Entity, RandomSource) return} {@code null}), or a filled Optional containing a {@link MerchantOffer} with a matching result.
     */
    public @Nullable Optional<MerchantOffer> forceTradeListing(ItemListing listing, Entity trader, RandomSource random);

    /**
     * Attempt to force the output of the given {@link ItemListing} to a {@link MerchantOffer} whose {@link MerchantOffer#getResult() output} does not {@link IAdvancedIngredient#test(Object) conform} to this {@link IAdvancedIngredient}.
     * This should not generate a {@link MerchantOffer} which could not have been generated by the {@link ItemListing} naturally.
     * It is not guaranteed that the result of the returned {@link MerchantOffer} (if there is one) won't {@link IAdvancedIngredient#test(Object) conform}, but this method should make a best effort.
     * @param listing
     * @param trader
     * @param random
     * @return {@code null} if this {@link IAdvancedIngredient} could not generate a {@link MerchantOffer} with a non-matching result, an empty Optional if the {@link ItemListing} should generate no {@link MerchantOffer} at all (i.e. {@link ItemListing#getOffer(Entity, RandomSource) return} {@code null}), or a filled Optional containing a {@link MerchantOffer} with a non-matching result.
     */
    public @Nullable Optional<MerchantOffer> forbidTradeListing(ItemListing listing, Entity trader, RandomSource random);
};
