package com.petrolpark.core.data.loot.numberprovider;

import java.util.List;

import javax.annotation.Nonnull;

import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import com.petrolpark.PetrolparkNumberProviderTypes;

import net.minecraft.world.level.storage.loot.LootContext;
import net.minecraft.world.level.storage.loot.providers.number.LootNumberProviderType;
import net.minecraft.world.level.storage.loot.providers.number.NumberProvider;
import net.minecraft.world.level.storage.loot.providers.number.NumberProviders;

public record PolynomialNumberProvider(NumberProvider value, List<NumberProvider> coefficients) implements IEstimableNumberProvider {

    public static final MapCodec<PolynomialNumberProvider> CODEC = RecordCodecBuilder.mapCodec(instance -> instance.group(
        NumberProviders.CODEC.fieldOf("value").forGetter(PolynomialNumberProvider::value),
        NumberProviders.CODEC.listOf().fieldOf("coefficients").forGetter(PolynomialNumberProvider::coefficients)
    ).apply(instance, PolynomialNumberProvider::new));

    @Override
    public float getFloat(@Nonnull LootContext lootContext) {
        float total = 0f;
        float value = value().getFloat(lootContext);
        int power = 0;
        for (NumberProvider coeff : coefficients()) {
            total += coeff.getFloat(lootContext) * Math.pow(value, power);
            power++;
        };
        return total;
    };
    
    @Override
    public NumberEstimate getEstimate() {
        NumberEstimate total = NumberEstimate.ZERO;
        NumberEstimate value = NumberEstimate.get(value());
        int power = 0;
        for (NumberProvider coeff : coefficients()) {
            total = total.add(NumberEstimate.get(coeff).multiply(value.pow(power)));
            power++;
        };
        return total;
    };

    @Override
    public float getMaxFloat(LootContext context) {
        float total = 0f;
        float value = NumberEstimate.getMax(context, value());
        int power = 0;
        for (NumberProvider coeff : coefficients()) {
            total += NumberEstimate.getMax(context, coeff) * Math.pow(value, power);
            power++;
        };
        return total;
    };

    @Override
    public LootNumberProviderType getType() {
        return PetrolparkNumberProviderTypes.POLYNOMIAL.get();
    };
    
};
