package fr.estecka.variantscit.reload;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import net.minecraft.class_1792;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_7923;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import fr.estecka.variantscit.BakedModule;
import fr.estecka.variantscit.IItemModelProvider;
import fr.estecka.variantscit.ModuleRegistry;
import fr.estecka.variantscit.VariantLibrary;
import fr.estecka.variantscit.VariantsCitMod;
import fr.estecka.variantscit.api.ICitModule;

public final class ModuleLoader
{
	static public class Result {
		public final Map<class_1792, IItemModelProvider> itemModules  = new HashMap<>();
		public final Map<class_1792, IItemModelProvider> equipModules = new HashMap<>();
		public final ItemVariantAggregator  itemAggregator  = new ItemVariantAggregator ();
		public final EquipVariantAggregator equipAggregator = new EquipVariantAggregator();
	}

	static record ProtoModule (
		ModuleDefinition definition,
		JsonObject parameters
	){}

	/**
	 * Contains all the processed data about a module. Some of that data is only
	 * relevant to the resource-loading phase, and will be discarded at the end.
	 */
	static public record MetaModule (
		class_2960 id,
		int priority,
		Set<class_1792> targets,
		Optional<VariantLibrary> itemLibrary,
		Optional<VariantLibrary> equipLibrary,
		ICitModule logic
		
	){}

	static public ModuleLoader.Result ReloadModules(class_3300 manager)
	{
		final ModuleLoader.Result result = new ModuleLoader.Result();
		final List<MetaModule> modules = new ArrayList<>();

		Map<class_2960, class_3298> resources = new HashMap<>();
		resources.putAll(manager.method_14488("variant-cits/item", id->id.method_12832().endsWith(".json")));
		ObsoletePathWarning(resources);
		resources.putAll(manager.method_14488("variants-cit/item", id->id.method_12832().endsWith(".json")));
		// resources.putAll(manager.findResources("variants-cit/equipment", id->id.getPath().endsWith(".json")));

		for (var entry : resources.entrySet())
		try {
			class_2960 moduleId = ModuleIdFromResourceId(entry.getKey());
			ProtoModule prototype = DefinitionFromResource(entry.getValue()).getOrThrow();

			List<EModuleContext> contexts = prototype.definition().contexts();
			if (contexts.isEmpty()){
				VariantsCitMod.LOGGER.warn("Ignored VCIT module with no context: {}", moduleId);
				continue;
			}

			Set<class_1792> targets = prototype.definition.targets()
				.map(ModuleLoader::ItemsFromTarget)
				.orElseGet(()->ItemsFromModuleId(moduleId))
				;
			if (targets.isEmpty()){
				VariantsCitMod.LOGGER.warn("Skipped VCIT module with no valid item: {}", moduleId);
				continue;
			}

			if (prototype.definition.modelPrefix().isEmpty())
				VariantsCitMod.LOGGER.error("VCIT module `{}` has an empty model prefix. This can lead to unexpected behaviours and performance loss.", moduleId);

			ICitModule moduleLogic = ModuleRegistry.CreateModule(prototype.definition.type(), prototype.parameters);
			VariantLibrary itemLibrary = null;
			VariantLibrary equipLibrary = null;

			for (EModuleContext c : contexts)
			switch (c) {
				case ITEM_MODEL: itemLibrary  = result.itemAggregator .CreateLibrary(prototype.definition, manager); break;
				case EQUIPPABLE: equipLibrary = result.equipAggregator.CreateLibrary(prototype.definition, manager); break;
			}

			MetaModule meta = new MetaModule(
				moduleId,
				prototype.definition.priority(),
				targets,
				Optional.ofNullable(itemLibrary),
				Optional.ofNullable(equipLibrary),
				moduleLogic
			);

			modules.add(meta);
		}
		catch (IllegalStateException e){
			VariantsCitMod.LOGGER.error("Error in VCIT module {}: {}", entry.getKey(), e);
		}

		// Sort highest priorities first.
		modules.sort((a,b) -> -Integer.compare(a.priority(), b.priority()));

		BakeModules(result, modules);
		return result;
	}

	static private void ObsoletePathWarning(Map<class_2960, class_3298> resources){
		if (!resources.isEmpty()){
			String names = "";
			for (class_2960 id : resources.keySet()) {
				names += ' ';
				names += ModuleIdFromResourceId(id).toString();
			}
			VariantsCitMod.LOGGER.warn("Some VCIT modules are using the old mispelled directory `variant-cits`, those should be moved to `variants-cit` instead:{}", names);
		}
	}

	static private Set<class_1792> ItemsFromTarget(List<class_2960> targets){
		Set<class_1792> result = new HashSet<>();
		targets.stream()
			.map(id->class_7923.field_41178.method_10223(id))
			.filter(Optional::isPresent)
			.map(opt->opt.get().comp_349())
			.forEach(result::add)
			;
		return result;
	}

	static private Set<class_1792> ItemsFromModuleId(class_2960 moduleId){
		class_2960 itemId = ItemIdFromModuleId(moduleId);

		if (class_7923.field_41178.method_10250(itemId))
			return Set.of(class_7923.field_41178.method_10223(itemId).get().comp_349());
		else
			return Set.of();
	}

	static private class_2960 ItemIdFromModuleId(class_2960 resource){
		String path = resource.method_12832();
		path = path.substring("item/".length());
		return class_2960.method_60655(resource.method_12836(), path);
	}

	static private class_2960 ModuleIdFromResourceId(class_2960 resource){
		String path = resource.method_12832();
		path = path.substring("variant-cits/".length(), path.length()-".json".length());
		return class_2960.method_60655(resource.method_12836(), path);
	}

	static private DataResult<ProtoModule> DefinitionFromResource(class_3298 resource){
		JsonObject json;
		try {
			json = class_3518.method_15255(resource.method_43039());
		}
		catch (IOException|JsonParseException e){
			return DataResult.error(e::toString);
		}

		var dataResult = ModuleDefinition.CODEC.decoder().decode(JsonOps.INSTANCE, json);
		if (dataResult.isError()){
			return DataResult.error(dataResult.error().get()::message);
		}

		try {
			ModuleDefinition definition = dataResult.getOrThrow().getFirst();
			JsonObject parameters = json.getAsJsonObject("parameters");
			if (parameters == null)
				parameters = new JsonObject();

			return DataResult.success(new ProtoModule(definition, parameters));
		}
		catch (IllegalStateException|ClassCastException e){
			return DataResult.error(e::toString);
		}
	}

	static public void BakeModules(ModuleLoader.Result result, List<MetaModule> modules){
		Map<class_1792, List<BakedModule>> itemModules  = new HashMap<>();
		Map<class_1792, List<BakedModule>> equipModules = new HashMap<>();
	
		for (MetaModule meta : modules)
		{
			if (meta.itemLibrary() .isPresent()) BakeModuleContext("item_model", meta, meta.itemLibrary ().get(), itemModules );
			if (meta.equipLibrary().isPresent()) BakeModuleContext("equippable", meta, meta.equipLibrary().get(), equipModules);
		}

		BakeItem(result.itemModules,  itemModules );
		BakeItem(result.equipModules, equipModules);
	}

	static private void BakeModuleContext(String contextName, MetaModule meta, VariantLibrary lib, Map<class_1792, List<BakedModule>> output){
		if (lib.isEmpty())
			VariantsCitMod.LOGGER.warn("Empty {} VCIT module {}", contextName, meta.id());
		else
			VariantsCitMod.LOGGER.info("Found {} {} variants for VCIT module {}", lib.GetVariantCount(), contextName, meta.id());

		for (class_1792 itemType : meta.targets()){
			output.computeIfAbsent(itemType, __->new ArrayList<>()).add(new BakedModule(lib, meta.logic()));
		}
	}

	static private void BakeItem(Map<class_1792, IItemModelProvider> result, Map<class_1792, List<BakedModule>> moduleListPerItem){
		for (var entry : moduleListPerItem.entrySet()){
			result.put(
				entry.getKey(),
				IItemModelProvider.OfList( entry.getValue() )
			);
		}
	}


}
