package fr.estecka.variantscit;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import fr.estecka.variantscit.api.ICitModule;
import net.fabricmc.fabric.api.client.model.loading.v1.PreparableModelLoadingPlugin.DataLoader;
import net.minecraft.class_1792;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_6880;
import net.minecraft.class_7923;

public final class ModuleLoader
implements DataLoader<ModuleLoader.Result>
{
	static public class Result {
		public final HashMap<class_2960, BakedModule> uniqueModules = new HashMap<>();
		public final HashSet<class_2960> ignoredModules = new HashSet<>();
		public final Map<class_6880<class_1792>,List<MetaModule>> modulesPerItem = new HashMap<>();
		public final ModelAggregator modelAggregator = new ModelAggregator();
	}

	static record ProtoModule (
		ModuleDefinition definition,
		JsonObject parameters
	){}

	/**
	 * Contains all the data  pertaining to  a module file. Most of this data is
	 * only  relevant  to  the  resource-loading  phase, and  will be  discarded
	 * afterward.
	 */
	static public record MetaModule (
		class_2960 id,
		ProtoModule prototype,
		BakedModule bakedModule
	){}

	@Override
	public CompletableFuture<ModuleLoader.Result> load(class_3300 resourceManager, Executor executor){
		return CompletableFuture.supplyAsync(()->ReloadModules(resourceManager), executor);
	}

	static private ModuleLoader.Result ReloadModules(class_3300 manager)
	{
		ModuleLoader.Result result = new ModuleLoader.Result();

		Map<class_2960, class_3298> resources = new HashMap<>();
		resources.putAll(manager.method_14488("variant-cits/item", id->id.method_12832().endsWith(".json")));
		Warn(resources);
		resources.putAll(manager.method_14488("variants-cit/item", id->id.method_12832().endsWith(".json")));

		for (var entry : resources.entrySet())
		try {
			class_2960 moduleId = ModuleIdFromResourceId(entry.getKey());
			ProtoModule prototype = DefinitionFromResource(entry.getValue()).getOrThrow();

			Set<class_6880<class_1792>> targets = prototype.definition.targets()
				.map(ModuleLoader::ItemsFromTarget)
				.orElseGet(()->ItemsFromModuleId(moduleId))
				;
			if (targets.isEmpty()){
				result.ignoredModules.add(moduleId);
				VariantsCitMod.LOGGER.warn("Skipped VCIT module with no valid item: {}", moduleId);
				continue;
			}

			if (prototype.definition.modelPrefix().equals("item/"))
				VariantsCitMod.LOGGER.error("VCIT module `{}` has an empty model prefix. This can lead to unexpected behaviours and performance loss.", moduleId);

			ICitModule moduleLogic = ModuleRegistry.CreateModule(prototype.definition, prototype.parameters);
			VariantLibrary library = result.modelAggregator.CreateLibrary(prototype, manager);
			MetaModule meta = new MetaModule(
				moduleId,
				prototype,
				new BakedModule(library, moduleLogic)
			);

			result.uniqueModules.put(moduleId, meta.bakedModule());
			for (var item : targets){
				result.modulesPerItem.computeIfAbsent(item, __->new ArrayList<>()).add(meta);
			}
		}
		catch (IllegalStateException e){
			VariantsCitMod.LOGGER.error("Error in VCIT module {}: {}", entry.getKey(), e);
		}

		// Sort highest priorities first.
		for (List<MetaModule> modules : result.modulesPerItem.values()){
			modules.sort((a,b) -> -Integer.compare(a.prototype.definition.priority(), b.prototype.definition.priority()));
		}
		return result;
	}

	static private void Warn(Map<class_2960, class_3298> resources){
		if (!resources.isEmpty()){
			String names = "";
			for (class_2960 id : resources.keySet()) {
				names += ' ';
				names += ModuleIdFromResourceId(id).toString();
			}
			VariantsCitMod.LOGGER.warn("Some VCIT modules are using the old mispelled directory `variant-cits`, those should be moved to `variants-cit` instead:{}", names);
		}
	}

	static private Set<class_6880<class_1792>> ItemsFromTarget(List<class_2960> targets){
		Set<class_6880<class_1792>> result = new HashSet<>();
		targets.stream()
			.map(id->class_7923.field_41178.method_10223(id).orElse(null))
			.filter(o->o!=null)
			.forEach(result::add)
			;
		return result;
	}

	static private Set<class_6880<class_1792>> ItemsFromModuleId(class_2960 moduleId){
		class_2960 itemId = ItemIdFromModuleId(moduleId);

		if (class_7923.field_41178.method_10250(itemId))
			return Set.of(class_7923.field_41178.method_10223(itemId).get());
		else
			return Set.of();
	}

	static private class_2960 ItemIdFromModuleId(class_2960 resource){
		String path = resource.method_12832();
		path = path.substring("item/".length());
		return class_2960.method_60655(resource.method_12836(), path);
	}

	static private class_2960 ModuleIdFromResourceId(class_2960 resource){
		String path = resource.method_12832();
		path = path.substring("variant-cits/".length(), path.length()-".json".length());
		return class_2960.method_60655(resource.method_12836(), path);
	}

	static private DataResult<ProtoModule> DefinitionFromResource(class_3298 resource){
		JsonObject json;
		try {
			json = class_3518.method_15255(resource.method_43039());
		}
		catch (IOException|JsonParseException e){
			return DataResult.error(e::toString);
		}

		var dataResult = ModuleDefinition.CODEC.decoder().decode(JsonOps.INSTANCE, json);
		if (dataResult.isError()){
			return DataResult.error(dataResult.error().get()::message);
		}

		try {
			ModuleDefinition definition = dataResult.getOrThrow().getFirst();
			JsonObject parameters = json.getAsJsonObject("parameters");
			if (parameters == null)
				parameters = new JsonObject();

			return DataResult.success(new ProtoModule(definition, parameters));
		}
		catch (IllegalStateException|ClassCastException e){
			return DataResult.error(e::toString);
		}
	}
}
