package fr.estecka.variantscit.reload;

import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import fr.estecka.variantscit.modules.libraries.VariantLibrary;
import fr.estecka.variantscit.VariantsCitMod;
import fr.estecka.variantscit.mixin.BakedModelManagerMixin;

/**
 * @implNote Texture and Baked Model ids usually start with `item/` however here
 * this prefix is stripped off so that their ids match those of the item states.
 * These `item/` need to be re-added in {@link BakedModelManagerMixin} for asset
 * generation.
 */
public class VariantAggregator
{
	static public record ModelToCreate(
		class_2960 parent,
		int priority
	){}

	private final Map<ModuleDefinition, class_2960> moduleIds = new IdentityHashMap<>();
	private final Map<ModuleDefinition, VariantLibrary> item_model = new IdentityHashMap<>();
	private final Map<ModuleDefinition, VariantLibrary> equippable = new IdentityHashMap<>();

	// item_model assetgen
	private final Set<class_2960> acceptedItemModels = new HashSet<>();
	public final Map<class_2960, ModelToCreate> modelsToCreate = new HashMap<>();
	public final Set<class_2960> itemStatesToCreate = new HashSet<>();
	public final Set<String> conflictingModelPrefixes = new HashSet<>();


	public VariantAggregator(Map<class_2960, ModuleDefinition> modules){
		for (var entry : modules.entrySet()){
			ModuleDefinition module = entry.getValue();
			this.moduleIds.put(module, entry.getKey());
			for (EModuleContext context : module.contexts())
				GetLibraryMap(context).put(module, EmptyLibrary(module));
		}
	}

	static private VariantLibrary EmptyLibrary(ModuleDefinition module) {
		return new VariantLibrary(
			module.fallbackModel().orElse(null),
			new HashMap<>(),
			module.specialModels()
		);
	}

	private Map<ModuleDefinition, VariantLibrary> GetLibraryMap(EModuleContext context){
		return switch (context){
			default -> throw new AssertionError("Invalid Context");
			case EQUIPPABLE -> this.equippable;
			case ITEM_MODEL -> this.item_model;
		};
	}

	public Optional<VariantLibrary> GetLibrary(EModuleContext context, ModuleDefinition module){
		return Optional.ofNullable(GetLibraryMap(context).get(module));
	}

	public void GatherAll(class_3300 manager){
		GatherType(EAssetType.ITEM_STATE,  manager);
		GatherType(EAssetType.BAKED_MODEL, manager);
		GatherType(EAssetType.TEXTURE,     manager);
		GatherType(EAssetType.EQUIPMENT,   manager);

		// Share generated assets accross modules
		GatherIds(EAssetType.BAKED_MODEL, this.modelsToCreate.keySet().stream());
		GatherIds(EAssetType.ITEM_STATE,  this.itemStatesToCreate.stream());
	}

	private void GatherType(EAssetType assetType, class_3300 manager){
		Set<class_2960> resources = manager.method_14488(assetType.directory, id->id.method_12832().endsWith(assetType.suffix)).keySet();

		Stream<class_2960> ids = resources.stream().map(
			id->id.method_45134(path->path.substring(
				assetType.directory.length() + 1,
				path.length() - assetType.suffix.length()
			))
		);

		GatherIds(assetType, ids);
	}

	private void GatherIds(EAssetType assetType, Stream<class_2960> assets){
		assets.forEach(assetId -> ApplyModelToAll(assetType, assetId));
	}

	private void ApplyModelToAll(EAssetType assetType, class_2960 assetId){
		for (var entry : GetLibraryMap(assetType.context).entrySet())
		if  (IsTypeAcceptable(assetType, entry.getKey()))
		{
			ModuleDefinition module = entry.getKey();
			VariantLibrary library = entry.getValue();
			VariantsCitMod.LOGGER.PushLabel(moduleIds.get(module));

			boolean accepted = this.ApplyModelToModule(module, library, assetId);
			if (accepted && assetType.context == EModuleContext.ITEM_MODEL) {
				switch (assetType) {
					default: /* no-op */;
					break;

					// Fallthrough
					case TEXTURE:     OnAcceptedTexture   (module, assetId);
					case BAKED_MODEL: OnAcceptedBakedModel(module, assetId);
					break;
				}

				this.acceptedItemModels.add(assetId);
			}
			VariantsCitMod.LOGGER.PopLabel();
		}
	}

	static private boolean IsTypeAcceptable(EAssetType type, ModuleDefinition module){
		switch (type) {
			default:          return true;
			case BAKED_MODEL: return module.itemGen();
			case TEXTURE:     return module.itemGen() && module.modelParent().isPresent();
		}
	}

	private boolean ApplyModelToModule(ModuleDefinition module, VariantLibrary library, class_2960 modelId){
		boolean accepted = false;

		if (modelId.equals(library.fallbackModel()))
			accepted = true;

		if (library.specialModels().containsValue(modelId))
			accepted = true;

		if (modelId.method_12832().startsWith(module.modelPrefix())){
			class_2960 variantId = class_2960.method_60655(
				modelId.method_12836(),
				modelId.method_12832().substring(module.modelPrefix().length())
			);

			if (module.parameters().AcceptsVariant(variantId)){
				accepted = true;
				library.variantModels().put(variantId, modelId);
			}

		}

		return accepted;
	}

	private void OnAcceptedBakedModel(ModuleDefinition module, class_2960 modelId){
		if (!this.acceptedItemModels.contains(modelId))
			this.itemStatesToCreate.add(modelId);
	}

	private void OnAcceptedTexture(ModuleDefinition module, class_2960 modelId){
		int priority = module.modelPrefix().length();
		class_2960 parent = module.modelParent().get();
		ModelToCreate oldModel = this.modelsToCreate.get(modelId);

		if ((oldModel != null && oldModel.priority < priority)
		|| (!this.acceptedItemModels.contains(modelId))
		){
			this.modelsToCreate.put(modelId, new ModelToCreate(parent, priority));
		}
		else if (oldModel != null && oldModel.priority == priority && !oldModel.parent.equals(parent)){
			this.conflictingModelPrefixes.add(module.modelPrefix());
		}
	}
}
