package io.wispforest.accessories.api;

import com.google.common.collect.Multimap;
import io.wispforest.accessories.api.slot.*;
import io.wispforest.accessories.pond.AccessoriesAPIAccess;
import it.unimi.dsi.fastutil.Pair;
import org.apache.commons.lang3.function.TriFunction;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import net.minecraft.class_1309;
import net.minecraft.class_1322;
import net.minecraft.class_1792;
import net.minecraft.class_1799;

public interface AccessoriesCapability {

    /**
     * @return The {@link AccessoriesCapability} Bound to the given living entity if present
     */
    @Nullable
    static AccessoriesCapability get(@NotNull class_1309 livingEntity){
        return ((AccessoriesAPIAccess) livingEntity).accessoriesCapability();
    }

    static Optional<AccessoriesCapability> getOptionally(@NotNull class_1309 livingEntity){
        return Optional.ofNullable(get(livingEntity));
    }

    //--

    /**
     * @return The entity bound to the given {@link AccessoriesCapability} instance
     */
    class_1309 entity();

    /**
     * @return The {@link AccessoriesHolder} bound to the given {@link class_1309}
     */
    AccessoriesHolder getHolder();

    //--

    /**
     * Method used to clear all containers bound to the given {@link class_1309}
     */
    void reset(boolean loadedFromTag);

    /**
     * @return A Map containing all the {@link AccessoriesContainer}s with their {@link SlotType#name()} as the key
     */
    Map<String, AccessoriesContainer> getContainers();

    /**
     * @return a given {@link AccessoriesContainer} if found on the given {@link class_1309} tied to the Capability or null if not
     */
    @Nullable
    default AccessoriesContainer getContainer(SlotType slotType){
        return getContainers().get(slotType.name());
    }

    @Nullable
    default AccessoriesContainer getContainer(SlotTypeReference reference){
        return getContainers().get(reference.slotName());
    }

    void updateContainers();

    //--

    /**
     * Used to attempt to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference and list within a pair. The given list may contain the overflow that could not fit based
     * on the containers max stack size.
     * <p>
     * <b>WARNING: THE GIVEN STACK PASSED WILL NOT BE MUTATED AT ALL!</b>
     *
     * @param stack          The given stack attempting to be equipped
     */
    @Deprecated
    @Nullable
    default Pair<SlotReference, List<class_1799>> equipAccessory(class_1799 stack){
        return equipAccessory(stack, false);
    }

    /**
     * Used to attempt to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference and list within a pair. The given list may contain the overflow that could not fit based
     * on the containers max stack size and the old stack found if swapping was allowed.
     * <p>
     * <b>WARNING: THE GIVEN STACK PASSED WILL NOT BE MUTATED AT ALL!</b>
     *
     * @param stack          The given stack attempting to be equipped
     * @param allowSwapping  If the given call can attempt to swap accessories
     */
    @Deprecated
    default Pair<SlotReference, List<class_1799>> equipAccessory(class_1799 stack, boolean allowSwapping) {
        var stackCopy = stack.method_7972();

        var result = attemptToEquipAccessory(stackCopy, allowSwapping);

        if(result == null) return null;

        var returnStacks = new ArrayList<class_1799>();

        if(!stackCopy.method_7960()) returnStacks.add(stackCopy);

        result.second().ifPresent(returnStacks::add);

        return Pair.of(result.first(), returnStacks);
    }

    /**
     * Attempts to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference to where it was equipped. The given passed stack <b>will</b> be adjusted passed on
     * the amount of room that can be found within the found container.
     *
     * @param stack The given stack attempting to be equipped
     */
    @Nullable
    default SlotReference attemptToEquipAccessory(class_1799 stack) {
        var result = attemptToEquipAccessory(stack, false);

        return result != null ? result.first() : null;
    }

    /**
     * Attempts to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference to where it was equipped and an {@link Optional} of the previous stack if swapped for
     * the passed stack. The given passed stack <b>will</b> be adjusted passed on the amount of room that
     * can be found within the found container.
     *
     * @param stack The given stack attempting to be equipped
     */
    @Nullable
    default Pair<SlotReference, Optional<class_1799>> attemptToEquipAccessory(class_1799 stack, boolean allowSwapping) {
        var result = canEquipAccessory(stack, allowSwapping, (slotStack, slotReference) -> true);

        return result != null ? Pair.of(result.first(), result.second().equipStack(stack)) : null;
    }

    default Pair<SlotReference, EquipAction> canEquipAccessory(class_1799 stack, boolean allowSwapping) {
        return canEquipAccessory(stack, allowSwapping, (slotStack, slotReference) -> true);
    }

    /**
     * Attempts to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference to where it can be equipped and a function to attempt equipping of the item which
     * may return an {@link Optional} of the previous stack if allowing for swapping.
     * <p>
     * Info: The passed stack will not be mutated in any way! Such only occurs on call of the possible
     * returned function.
     *
     * @param stack The given stack attempting to be equipped
     */
    @Nullable
    Pair<SlotReference, EquipAction> canEquipAccessory(class_1799 stack, boolean allowSwapping, EquipCheck extraCheck);

    //--

    /**
     * @return If any {@link class_1799} is equipped based on the given {@link class_1792} entry
     */
    default boolean isEquipped(class_1792 item){
        return isEquipped(item, EquipmentChecking.ACCESSORIES_ONLY);
    }

    default boolean isEquipped(class_1792 item, EquipmentChecking check){
        return isEquipped(stack -> stack.method_7909() == item, check);
    }

    /**
     * @return If any {@link class_1799} is equipped based on the passed predicate
     */
    default boolean isEquipped(Predicate<class_1799> predicate) {
        return isEquipped(predicate, EquipmentChecking.ACCESSORIES_ONLY);
    }

    default boolean isEquipped(Predicate<class_1799> predicate, EquipmentChecking check) {
        return getFirstEquipped(predicate, check) != null;
    }

    default boolean isAnotherEquipped(SlotReference slotReference, class_1792 item) {
        return isAnotherEquipped(slotReference, stack -> stack.method_7909().equals(item));
    }

    /**
     * @return If any {@link class_1799} is equipped based on the passed predicate while deduplicating
     * using the current {@link SlotReference}
     */
    default boolean isAnotherEquipped(SlotReference slotReference, Predicate<class_1799> predicate) {
        var equippedStacks = getEquipped(predicate);

        if (equippedStacks.size() > 2) {
            for (var equippedStack : equippedStacks) {
                if (!equippedStack.reference().equals(slotReference)) return true;
            }
        } else if(equippedStacks.size() == 1) {
            return !equippedStacks.get(0).reference().equals(slotReference);
        }

        return false;
    }

    /**
     * @return The first {@link class_1799} formatted within {@link SlotEntryReference} that matches the given {@link class_1792}
     */
    @Nullable
    default SlotEntryReference getFirstEquipped(class_1792 item){
        return getFirstEquipped(item, EquipmentChecking.ACCESSORIES_ONLY);
    }

    @Nullable
    default SlotEntryReference getFirstEquipped(class_1792 item, EquipmentChecking check){
        return getFirstEquipped(stack -> stack.method_7909() == item, check);
    }

    /**
     * @return The first {@link class_1799} formatted within {@link SlotEntryReference} that matches the given predicate
     */
    @Nullable
    default SlotEntryReference getFirstEquipped(Predicate<class_1799> predicate) {
        return getFirstEquipped(predicate, EquipmentChecking.ACCESSORIES_ONLY);
    }

    @Nullable
    SlotEntryReference getFirstEquipped(Predicate<class_1799> predicate, EquipmentChecking check);

    /**
     * @return A list of all {@link class_1799}'s formatted within {@link SlotEntryReference} matching the given {@link class_1792}
     */
    default List<SlotEntryReference> getEquipped(class_1792 item){
        return getEquipped(stack -> stack.method_7909() == item);
    }

    /**
     * @return A list of all {@link SlotEntryReference}'s formatted within {@link SlotEntryReference} matching the passed predicate
     */
    List<SlotEntryReference> getEquipped(Predicate<class_1799> predicate);

    /**
     * @return A list of all {@link class_1799}'s formatted within {@link SlotEntryReference}
     */
    default List<SlotEntryReference> getAllEquipped() {
        return getAllEquipped(true);
    }

    List<SlotEntryReference> getAllEquipped(boolean recursiveStackLookup);

    //--

    /**
     * Add map containing slot attributes to the given capability based on the keys used referencing specific slots
     * with being lost on reload
     * @param modifiers Slot Attribute Modifiers
     */
    void addTransientSlotModifiers(Multimap<String, class_1322> modifiers);

    /**
     * Add slot attribute attributes to the given capability based on the keys used referencing specific slots
     * with being persistent on a reload
     * @param modifiers Slot Attribute Modifiers
     */
    void addPersistentSlotModifiers(Multimap<String, class_1322> modifiers);

    /**
     * Add slot attribute attributes to the given capability based on the keys
     * @param modifiers Slot Attribute Modifiers
     */
    void removeSlotModifiers(Multimap<String, class_1322> modifiers);

    /**
     * Get all modifiers from the given containers bound to the capability
     */
    Multimap<String, class_1322> getSlotModifiers();

    /**
     * Remove all modifiers from the given containers bound to the capability
     */
    void clearSlotModifiers();

    /**
     * Remove all cached modifiers from the given containers bound to the capability
     */
    void clearCachedSlotModifiers();

    //--

    @Nullable
    @Deprecated
    default Pair<SlotReference, List<class_1799>> equipAccessory(class_1799 stack, boolean allowSwapping, TriFunction<Accessory, class_1799, SlotReference, Boolean> additionalCheck) {
        return equipAccessory(stack, allowSwapping);
    }
}
