package io.wispforest.accessories.api;

import io.wispforest.accessories.api.attributes.AccessoryAttributeBuilder;
import io.wispforest.accessories.api.components.AccessoriesDataComponents;
import io.wispforest.accessories.api.components.AccessoryItemAttributeModifiers;
import io.wispforest.accessories.api.slot.SlotReference;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.impl.AccessoriesEventHandler;
import io.wispforest.accessories.mixin.LivingEntityAccessor;
import io.wispforest.accessories.networking.client.AccessoryBreak;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import net.minecraft.class_1282;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_1890;
import net.minecraft.class_2561;
import net.minecraft.class_3417;

/**
 * Main interface for implementing accessory functionality within the main API.
 */
public interface Accessory {

    /**
     * Called on every tick of the wearing {@link class_1309} on both client and server.
     *
     * @param stack the stack being ticked
     * @param reference the slot the accessory is in
     */
    default void tick(class_1799 stack, SlotReference reference){}

    /**
     * Called when the accessory is equipped
     *
     * @param stack the stack being equipped
     * @param reference the slot the accessory is in
     */
    default void onEquip(class_1799 stack, SlotReference reference){}

    /**
     * Called when the accessory is unequipped
     *
     * @param stack the stack being unequipped
     * @param reference the slot the accessory is in
     */
    default void onUnequip(class_1799 stack, SlotReference reference){}

    /**
     * @param stack the stack to be equipped
     * @param reference the slot the accessory is in
     * @return whether the given stack can be equipped
     */
    default boolean canEquip(class_1799 stack, SlotReference reference){
        return true;
    }

    /**
     * @param stack the stack to be unequipped
     * @param reference the slot the accessory is in
     * @return whether the given stack can be unequipped
     */
    default boolean canUnequip(class_1799 stack, SlotReference reference){
        if(class_1890.method_8224(stack)) {
            return reference.entity() instanceof class_1657 player && player.method_7337();
        }

        return true;
    }

    /**
     * Helper method used to fill the passed {@link AccessoryAttributeBuilder} for every call from
     * {@link AccessoriesAPI#getAttributeModifiers}
     *
     * @param stack     The Stack attempting to be unequipped
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     * @param builder   The builder to which attributes are to be added
     */
    default void getDynamicModifiers(class_1799 stack, SlotReference reference, AccessoryAttributeBuilder builder){
        getModifiers(stack, reference, builder);
    }

    /**
     * Helper method used to fill the passed {@link AccessoryItemAttributeModifiers.Builder} when called to modify
     * the given default {@link AccessoriesDataComponents#ATTRIBUTES} right before Registry freeze occurs. This is
     * useful for attributes that are not meant to be changed and remain mostly static.
     *
     * @param item      The item to which the given attributes will be defaulted for
     * @param builder   The builder to which attributes are to be added
     */
    default void getStaticModifiers(class_1792 item, AccessoryItemAttributeModifiers.Builder builder){

    }

    /**
     * Returns the following drop rule for the given Item
     *
     * @param stack     The Stack being prepared for dropping
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     * @param source    The specific {@link class_1282} that lead to the drop rule evaluation
     */
    default DropRule getDropRule(class_1799 stack, SlotReference reference, class_1282 source){
        return DropRule.DEFAULT;
    }

    //--

    /**
     * Method called when equipping the given accessory from hotbar by right-clicking
     *
     * @param stack The Stack being prepared for dropping
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     */
    default void onEquipFromUse(class_1799 stack, SlotReference reference){
        var sound = getEquipSound(stack, reference);

        if(sound == null) return;

        reference.entity().method_5783(sound.event(), sound.volume(), sound.pitch());
    }

    /**
     * Returns the equipping sound from use for a given stack
     *
     * @param stack The Stack being prepared for dropping
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     */
    @Nullable
    default SoundEventData getEquipSound(class_1799 stack, SlotReference reference){
        return new SoundEventData(class_3417.field_14883, 1.0f, 1.0f);
    }

    /**
     * Returns whether the given stack can be equipped from use
     *
     * @param stack The Stack attempted to be equipped
     */
    default boolean canEquipFromUse(class_1799 stack){
        try {
            return canEquipFromUse(stack, null);
        } catch (NullPointerException e) {
            return false;
        }
    }

    @Deprecated(forRemoval = true)
    default boolean canEquipFromUse(class_1799 stack, SlotReference reference){
        return true;
    }

    /**
     * Method used to render client based particles when {@link AccessoriesAPI#breakStack(SlotReference)} is
     * called on the server and the {@link AccessoryBreak} packet is received
     *
     * @param stack The Stack that broke
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     */
    default void onBreak(class_1799 stack, SlotReference reference) {
        ((LivingEntityAccessor) reference.entity()).accessors$breakItem(stack);
    }

    /**
     * @return Return the max stack amount allowed when equipping a given stack into an accessories inventory
     */
    default int maxStackSize(class_1799 stack){
        return stack.method_7914();
    }

    //--

    /**
     * Method used to add tooltip info for attribute like data based on a given slot type
     *
     * @param stack The Stack being referenced
     * @param type The SlotType being referenced
     * @param tooltips Final list containing the tooltip info
     * @param tooltipType Current tooltipFlag
     */
    default void getAttributesTooltip(class_1799 stack, SlotType type, List<class_2561> tooltips, class_1836 tooltipType){
        getAttributesTooltip(stack, type, tooltips);
    }

    /**
     * Method used to add any additional tooltip information to a given {@link Accessory} tooltip after
     * {@link AccessoriesEventHandler#addEntityBasedTooltipData} if called and at the
     * end of the {@link AccessoriesEventHandler#getTooltipData} call.
     *
     * <p>
     *     Do note that <b>if the given entity</b> is found to not be null, the list passed
     *     contains all tooltip info allowing for positioning before or after the tooltip info
     * </p>
     *
     * @param stack The Stack being referenced
     * @param tooltips Final list containing the tooltip info
     * @param tooltipType Current tooltipFlag
     */
    default void getExtraTooltip(class_1799 stack, List<class_2561> tooltips, class_1836 tooltipType){
        getExtraTooltip(stack, tooltips);
    }

    //--

    /**
     * @deprecated Use {@link #getDynamicModifiers} instead
     */
    @ApiStatus.ScheduledForRemoval(inVersion = "1.22")
    @Deprecated(forRemoval = true)
    default void getModifiers(class_1799 stack, SlotReference reference, AccessoryAttributeBuilder builder){}

    /**
     * @deprecated Use {@link #getAttributesTooltip(class_1799, SlotType, List, class_1836)}
     */
    @Deprecated(forRemoval = true)
    @ApiStatus.ScheduledForRemoval(inVersion = "1.22")
    default void getAttributesTooltip(class_1799 stack, SlotType type, List<class_2561> tooltips){}

    /**
     * @deprecated Use {@link #getExtraTooltip(class_1799, List, class_1836)}
     */
    @Deprecated(forRemoval = true)
    @ApiStatus.ScheduledForRemoval(inVersion = "1.22")
    default void getExtraTooltip(class_1799 stack, List<class_2561> tooltips){}
}