package io.wispforest.accessories.api;

import io.wispforest.accessories.api.attributes.AccessoryAttributeBuilder;
import io.wispforest.accessories.api.components.AccessoriesDataComponents;
import io.wispforest.accessories.api.components.AccessoryNestContainerContents;
import io.wispforest.accessories.api.slot.SlotEntryReference;
import io.wispforest.accessories.api.slot.SlotReference;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.impl.AccessoryNestUtils;
import it.unimi.dsi.fastutil.Pair;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import net.minecraft.class_1282;
import net.minecraft.class_1309;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_2561;

/**
 * An {@link Accessory} that contains and delegates to other accessories in some way
 */
public interface AccessoryNest extends Accessory {

    /**
     * @return all inner accessory stacks
     */
    default List<class_1799> getInnerStacks(class_1799 holderStack) {
        var data = AccessoriesDataComponents.readOrDefault(AccessoriesDataComponents.NESTED_ACCESSORIES, holderStack);

        return data == null ? List.of() : data.accessories();
    }

    /**
     * Sets a given stack at the specified index for the passed holder stack
     *
     * @param holderStack The given HolderStack
     * @param index       The target index
     * @param newStack    The new stack replacing the given index
     */
    default boolean setInnerStack(class_1799 holderStack, int index, class_1799 newStack) {
        if(!AccessoryNest.isAccessoryNest(holderStack)) return false;
        if(AccessoryNest.isAccessoryNest(newStack) && !this.allowDeepRecursion()) return false;

        AccessoriesDataComponents.update(
                AccessoriesDataComponents.NESTED_ACCESSORIES,
                holderStack,
                contents -> contents.setStack(index, newStack));

        return true;
    }

    /**
     * By default, accessory nests can only go one layer deep as it's hard to track the stack modifications any further
     *
     * @return Whether this implementation of the Accessory nest allows for further nesting of other Nests
     */
    default boolean allowDeepRecursion() {
        return false;
    }

    default List<Pair<DropRule, class_1799>> getDropRules(class_1799 stack, SlotReference reference, class_1282 source) {
        var innerRules = new ArrayList<Pair<DropRule, class_1799>>();

        var innerStacks = getInnerStacks(stack);

        for (int i = 0; i < innerStacks.size(); i++) {
            var innerStack = innerStacks.get(i);

            var rule = AccessoriesAPI.getOrDefaultAccessory(innerStack).getDropRule(innerStack, reference, source);

            innerRules.add(Pair.of(rule, innerStack));
        }

        return innerRules;
    }

    //--

    /**
     * Method used to perform some action on a possible {@link AccessoryNest} and return a result from that action or a default value if none found
     *
     * @param holderStack   Potential stack linked to a AccessoryNest
     * @param slotReference The primary SlotReference used from the given call
     * @param func          Action being done
     * @param defaultValue  Default value if stack is not a AccessoryNest
     */
    static <T> T attemptFunction(class_1799 holderStack, SlotReference slotReference, Function<Map<SlotEntryReference, Accessory>, T> func, T defaultValue){
        var data = AccessoryNestUtils.getData(holderStack);

        if(data == null) return defaultValue;

        var nest = (AccessoryNest) AccessoriesAPI.getAccessory(holderStack);

        var t = func.apply(data.getMap(slotReference));

        if(data.hasChangesOccured(holderStack)) nest.onStackChanges(holderStack, AccessoriesDataComponents.readOrDefault(AccessoriesDataComponents.NESTED_ACCESSORIES, holderStack), slotReference.entity());

        return t;
    }

    /**
     * Method used to perform some action on a possible {@link AccessoryNest} and return a result from that action or a default value if none found
     *
     * @param holderStack   Potential stack linked to a AccessoryNest
     * @param livingEntity Potential Living Entity involved with any stack changes
     * @param func          Action being done
     * @param defaultValue  Default value if stack is not a AccessoryNest
     */
    static <T> T attemptFunction(class_1799 holderStack, @Nullable class_1309 livingEntity, Function<Map<class_1799, Accessory>, T> func, T defaultValue){
        var data = AccessoryNestUtils.getData(holderStack);

        if(data == null) return defaultValue;

        var nest = (AccessoryNest) AccessoriesAPI.getAccessory(holderStack);

        var t = func.apply(data.getMap());

        if(data.hasChangesOccured(holderStack)) nest.onStackChanges(holderStack, AccessoriesDataComponents.readOrDefault(AccessoriesDataComponents.NESTED_ACCESSORIES, holderStack), livingEntity);

        return t;
    }

    /**
     * Method used to perform some action on a possible {@link AccessoryNest}
     *
     * @param holderStack   Potential stack linked to a AccessoryNest
     * @param slotReference Potential Living Entity involved with any stack changes
     * @param consumer      Action being done
     */
    static void attemptConsumer(class_1799 holderStack, SlotReference slotReference, Consumer<Map<SlotEntryReference, Accessory>> consumer){
        var data = AccessoryNestUtils.getData(holderStack);

        if(data == null) return;

        var nest = (AccessoryNest) AccessoriesAPI.getAccessory(holderStack);

        consumer.accept(data.getMap(slotReference));

        if(data.hasChangesOccured(holderStack)) nest.onStackChanges(holderStack, AccessoriesDataComponents.readOrDefault(AccessoriesDataComponents.NESTED_ACCESSORIES, holderStack), slotReference.entity());
    }

    /**
     * Method used to perform some action on a possible {@link AccessoryNest}
     *
     * @param holderStack  Potential stack linked to a AccessoryNest
     * @param livingEntity Potential Living Entity involved with any stack changes
     * @param consumer     Action being done
     */
    static void attemptConsumer(class_1799 holderStack, @Nullable class_1309 livingEntity, Consumer<Map<class_1799, Accessory>> consumer) {
        var data = AccessoryNestUtils.getData(holderStack);

        if (data == null) return;

        var nest = (AccessoryNest) AccessoriesAPI.getAccessory(holderStack);

        consumer.accept(data.getMap());

        if(data.hasChangesOccured(holderStack)) nest.onStackChanges(holderStack, AccessoriesDataComponents.readOrDefault(AccessoriesDataComponents.NESTED_ACCESSORIES, holderStack), livingEntity);
    }

    //--

    static boolean isAccessoryNest(class_1799 holderStack) {
        return AccessoriesAPI.getAccessory(holderStack) instanceof AccessoryNest;
    }

    /**
     * Check and handle any inner stack changes that may have occurred from an action performed on the stacks within the nest
     *
     * @param holderStack  HolderStack containing the nest of stacks
     * @param data         StackData linked to the given HolderStack
     * @param livingEntity Potential Living Entity involved with any stack changes
     */
    default void onStackChanges(class_1799 holderStack, AccessoryNestContainerContents data,  @Nullable class_1309 livingEntity){}

    //--

    @Override
    default void tick(class_1799 stack, SlotReference reference) {
        attemptConsumer(stack, reference, map -> map.forEach((entryRef, accessory) -> accessory.tick(entryRef.stack(), entryRef.reference())));
    }

    @Override
    default void onEquip(class_1799 stack, SlotReference reference) {
        attemptConsumer(stack, reference, map -> map.forEach((entryRef, accessory) -> accessory.onEquip(entryRef.stack(), entryRef.reference())));
    }

    @Override
    default void onUnequip(class_1799 stack, SlotReference reference) {
        attemptConsumer(stack, reference, map -> map.forEach((entryRef, accessory) -> accessory.onUnequip(entryRef.stack(), entryRef.reference())));
    }

    @Override
    default boolean canEquip(class_1799 stack, SlotReference reference) {
        return attemptFunction(stack, reference, map -> {
            MutableBoolean canEquip = new MutableBoolean(true);

            map.forEach((entryRef, accessory) -> canEquip.setValue(canEquip.booleanValue() && accessory.canEquip(entryRef.stack(), entryRef.reference())));

            return canEquip.getValue();
        }, false);
    }

    @Override
    default boolean canUnequip(class_1799 stack, SlotReference reference) {
        return attemptFunction(stack, reference, map -> {
            MutableBoolean canUnequip = new MutableBoolean(true);

            map.forEach((entryRef, accessory) -> canUnequip.setValue(canUnequip.booleanValue() && accessory.canUnequip(entryRef.stack(), entryRef.reference())));

            return canUnequip.getValue();
        }, true);
    }

    @Override
    default void getDynamicModifiers(class_1799 stack, SlotReference reference, AccessoryAttributeBuilder builder) {
        attemptConsumer(stack, reference, innerMap -> innerMap.forEach((entryRef, accessory) -> {
            var innerBuilder = new AccessoryAttributeBuilder(entryRef.reference());

            accessory.getDynamicModifiers(entryRef.stack(), entryRef.reference(), innerBuilder);

            builder.addFrom(innerBuilder);
        }));
    }

    @Override
    default void getAttributesTooltip(class_1799 stack, SlotType type, List<class_2561> tooltips, class_1836 tooltipType) {
        attemptConsumer(stack, (class_1309) null, map -> map.forEach((stack1, accessory) -> accessory.getAttributesTooltip(stack1, type, tooltips, tooltipType)));
    }

    @Override
    default void getExtraTooltip(class_1799 stack, List<class_2561> tooltips,class_1836 tooltipType) {
        attemptConsumer(stack, (class_1309) null, map -> map.forEach((stack1, accessory) -> accessory.getExtraTooltip(stack1, tooltips, tooltipType)));
    }
}
