package io.wispforest.accessories.api.attributes;

import com.google.common.collect.*;
import io.wispforest.accessories.Accessories;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.AccessoriesInternals;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.api.slot.NestedSlotReferenceImpl;
import io.wispforest.accessories.api.slot.SlotReference;
import io.wispforest.accessories.utils.AttributeUtils;
import it.unimi.dsi.fastutil.Pair;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

import java.util.*;
import java.util.Map.Entry;
import java.util.function.Function;
import net.minecraft.class_1320;
import net.minecraft.class_1322;
import net.minecraft.class_2960;

/**
 * Builder used to collect the attribute modifications from a given Accessory with the ability
 * to specified if an Attribute modification can be stacked or is exclusive to one version
 */
public final class AccessoryAttributeBuilder {

    private static final Logger LOGGER = LogUtils.getLogger();

    private final Map<class_1320, Map<class_2960, AttributeModificationData>> exclusiveAttributes = new HashMap<>();
    private final Multimap<class_1320, AttributeModificationData> stackedAttributes = LinkedHashMultimap.create();

    private final SlotReference slotReference;

    @ApiStatus.Internal
    public AccessoryAttributeBuilder(SlotReference slotReference) {
        this.slotReference = slotReference;
    }

    @ApiStatus.Internal
    public AccessoryAttributeBuilder(String slotName, int slot) {
        this.slotReference = SlotReference.of(null, slotName, slot);
    }

    @ApiStatus.Internal
    public AccessoryAttributeBuilder() {
        this.slotReference = SlotReference.of(null, "", 0);
    }

    /**
     * Adds a given attribute modifier as an exclusive modifier meaning that only one instance should ever exist
     */
    public AccessoryAttributeBuilder addExclusive(class_1320 attribute, class_2960 location, double amount, class_1322.class_1323 operation) {
        var data = AttributeUtils.getModifierData(location);

        this.addExclusive(attribute, new class_1322(data.second(), data.first(), amount, operation));

        return this;
    }

    /**
     * Adds a given attribute modifier as a stackable modifier meaning variants based on slot position is allowed. This is done by post process
     * step of appending slot information when adding to the living entity
     */
    public AccessoryAttributeBuilder addStackable(class_1320 attribute, class_2960 location, double amount, class_1322.class_1323 operation) {
        var data = AttributeUtils.getModifierData(location);

        this.addStackable(attribute, new class_1322(data.second(), data.first(), amount, operation));

        return this;
    }

    private final Set<class_2960> previouslyWarnedLocations = new HashSet<>();

    /**
     * Adds a given attribute modifier as an exclusive modifier meaning that only one instance should ever exist
     */
    public AccessoryAttributeBuilder addExclusive(class_1320 attribute, class_1322 modifier) {
        var id = AttributeUtils.getLocation(modifier.method_6185());

        var innerMap = this.exclusiveAttributes.computeIfAbsent(attribute, attributeHolder -> new HashMap<>());

        if(AccessoriesInternals.isDevelopmentEnv() && innerMap.containsKey(id) && !this.previouslyWarnedLocations.contains(id)) {
            LOGGER.warn("A given Modifier was found to have a duplicate location but was added as exclusive, was such on purpose as such will not stack with the other: {}", id);

            this.previouslyWarnedLocations.add(id);
        }

        innerMap.putIfAbsent(id, new AttributeModificationData(attribute, modifier));

        return this;
    }

    /**
     * Adds a given attribute modifier as a stackable modifier meaning variants based on slot position is allowed. This is done by post process
     * step of appending slot information when adding to the living entity
     */
    private AccessoryAttributeBuilder addStackable(class_1320 attribute, class_1322 modifier) {
        var location = AttributeUtils.getLocation(modifier.method_6185());

        this.stackedAttributes.put(attribute, new AttributeModificationData(this.slotReference.createSlotPath(), attribute, modifier));

        return this;
    }

    //--

    @ApiStatus.Internal
    public AccessoryAttributeBuilder addModifier(class_1320 attribute, class_1322 modifier, SlotReference slotReference, Function<String, class_2960> locationBuilder) {
        var data = AttributeUtils.getModifierData(Accessories.of(AccessoryAttributeBuilder.createSlotPath(slotReference)));

        var validName = modifier.method_6185().toLowerCase()
                .replace(" ", "_")
                .replaceAll("([^a-z0-9/._-])", "");

        if(modifier.method_6189().equals(data.right())) {
            this.addStackable(attribute, locationBuilder.apply(validName), modifier.method_6186(), modifier.method_6182());
        } else {
            this.addExclusive(attribute, locationBuilder.apply(validName), modifier.method_6186(), modifier.method_6182());
        }

        return this;
    }

    @Nullable
    public AttributeModificationData getExclusive(class_1320 attribute, class_2960 location) {
        var innerMap = this.exclusiveAttributes.get(attribute);

        if(innerMap == null) return null;

        return innerMap.get(location);
    }

    public Collection<AttributeModificationData> getStacks(class_1320 attribute, class_2960 location) {
        return this.stackedAttributes.get(attribute).stream().filter(data -> {
            var id = AttributeUtils.getLocation(data.modifier().method_6185());

            return id.equals(location);
        }).toList();
    }

    @Nullable
    public AttributeModificationData removeExclusive(class_1320 attribute, class_2960 location) {
        var innerMap = this.exclusiveAttributes.get(attribute);

        if(innerMap == null) return null;

        return innerMap.remove(location);
    }

    public Collection<AttributeModificationData> removeStacks(class_1320 attribute, class_2960 location) {
        Set<AttributeModificationData> removedData = new HashSet<>();

        for (var data : List.copyOf(this.stackedAttributes.get(attribute))) {
            var id = AttributeUtils.getLocation(data.modifier().method_6185());

            if(!id.equals(location)) continue;

            removedData.add(data);

            this.stackedAttributes.remove(attribute, data);
        }

        return removedData;
    }

    //--

    public Multimap<String, class_1322> getSlotModifiers() {
        var map = LinkedHashMultimap.<String, class_1322>create();

        this.exclusiveAttributes.forEach((attribute, innerMap) -> {
            innerMap.forEach((location, uniqueInstance) -> {
                if(!(uniqueInstance.attribute() instanceof SlotAttribute slotAttribute)) return;

                map.put(slotAttribute.slotName(), uniqueInstance.modifier());
            });
        });

        this.stackedAttributes.forEach((location, stackedInstance) -> {
            if(!(stackedInstance.attribute() instanceof SlotAttribute slotAttribute)) return;

            map.put(slotAttribute.slotName(), stackedInstance.modifier());
        });

        return map;
    }

    public Multimap<class_1320, class_1322> getAttributeModifiers(boolean filterSlots) {
        var map = LinkedHashMultimap.<class_1320, class_1322>create();

        this.exclusiveAttributes.forEach((attribute, innerMap) -> {
            innerMap.forEach((location, uniqueInstance) -> {
                if (filterSlots && uniqueInstance.attribute() instanceof SlotAttribute) return;

                map.put(uniqueInstance.attribute(), uniqueInstance.modifier());
            });
        });

        this.stackedAttributes.forEach((location, stackedInstance) -> {
            if(filterSlots && stackedInstance.attribute() instanceof SlotAttribute) return;

            map.put(stackedInstance.attribute(), stackedInstance.modifier());
        });

        return map;
    }

    public boolean isEmpty() {
        return this.exclusiveAttributes.isEmpty() && this.stackedAttributes.isEmpty();
    }

    public Map<class_1320, Map<class_2960, AttributeModificationData>> exclusiveAttributes() {
        return ImmutableMap.copyOf(this.exclusiveAttributes);
    }

    public Multimap<class_1320, AttributeModificationData> stackedAttributes() {
        return ImmutableMultimap.copyOf(this.stackedAttributes);
    }

    public AccessoryAttributeBuilder addFrom(AccessoryAttributeBuilder builder) {
        builder.exclusiveAttributes.forEach(this.exclusiveAttributes::putIfAbsent);
        this.stackedAttributes.putAll(builder.stackedAttributes);

        return this;
    }

    @Override
    public boolean equals(Object obj) {
        if(!(obj instanceof AccessoryAttributeBuilder otherBuilder)) return false;
        if(!areMapsEqual(this.stackedAttributes, otherBuilder.stackedAttributes)) return false;

        return this.exclusiveAttributes.equals(otherBuilder.exclusiveAttributes);
    }

    private static <K, V> boolean areMapsEqual(Multimap<K, V> multimap1, Multimap<K, V> multimap2) {
        for (var entry : multimap1.asMap().entrySet()) {
            if(entry.getValue().equals(multimap2.get(entry.getKey()))) return false;
        }

        return true;
    }

    //--

    // slotPath          = {slot_name}/{slot_index}[{nested_layer_info}]
    // nested_layer_info = /nest_{layer_index}_{slot_index}
    @Deprecated
    public static String createSlotPath(SlotReference ref) {
        return ref.createSlotPath();
    }

    @Deprecated
    public static String createSlotPath(String slotname, int slot) {
        return slotname.replace(":", "-") + "/" + slot;
    }
}
