package io.wispforest.accessories.api.client;

import com.mojang.datafixers.util.Pair;
import io.wispforest.accessories.api.AccessoriesContainer;
import io.wispforest.accessories.api.slot.SlotReference;
import io.wispforest.accessories.client.AccessoriesRenderLayer;
import io.wispforest.accessories.mixin.client.ModelPartAccessor;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.stream.Stream;
import net.minecraft.class_1306;
import net.minecraft.class_1309;
import net.minecraft.class_1799;
import net.minecraft.class_243;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4587;
import net.minecraft.class_4597;
import net.minecraft.class_572;
import net.minecraft.class_583;
import net.minecraft.class_630;
import net.minecraft.class_7833;
import net.minecraft.class_897;
import net.minecraft.class_922;

/**
 * Main Render Interface used to render Accessories
 * <p>
 * All Translation code is based on <a href="https://github.com/emilyploszaj/trinkets/blob/main/src/main/java/dev/emi/trinkets/api/client/TrinketRenderer.java">TrinketRenderer</a>
 * with adjustments to allow for any {@link class_1309} extending {@link class_572} in which
 * credit goes to <a href="https://github.com/TheIllusiveC4">TheIllusiveC4</a>, <a href="https://github.com/florensie">florensie</a>, and <a href="https://github.com/emilyploszaj">Emi</a>
 */
public interface AccessoryRenderer {

    /**
     * Render method called within the {@link AccessoriesRenderLayer#render} when rendering a given Accessory on a given {@link class_1309}.
     * The given {@link SlotReference} refers to the slot based on its type, entity and index within the {@link AccessoriesContainer}.
     */
    <M extends class_1309> void render(
            class_1799 stack,
            SlotReference reference,
            class_4587 matrices,
            class_583<M> model,
            class_4597 multiBufferSource,
            int light,
            float limbSwing,
            float limbSwingAmount,
            float partialTicks,
            float ageInTicks,
            float netHeadYaw,
            float headPitch
    );

    /**
     * @return if the given Accessory should render or not based on the boolean provided
     */
    default boolean shouldRender(boolean isRendering) {
        return isRendering;
    }

    /**
     * Determines if this accessory should render in first person
     * Override to return true for whichever arm this accessory renders on
     */
    default boolean shouldRenderInFirstPerson(class_1306 arm, class_1799 stack, SlotReference reference) {
        return false;
    }

    /**
     * Attempt to render the given Accessory on the first person player model if found to be able to from the {@link #shouldRenderInFirstPerson}
     * invocation.
     */
    default <M extends class_1309> void renderOnFirstPerson(
            class_1306 arm,
            class_1799 stack,
            SlotReference reference,
            class_4587 matrices,
            class_583<M> model,
            class_4597 multiBufferSource,
            int light
    ) {
        if (!shouldRenderInFirstPerson(arm, stack, reference)) return;

        this.render(stack, reference, matrices, model, multiBufferSource, light, 0, 0, 0, 0, 0, 0);
    }

    /**
     * Rotates the rendering for the models based on the entity's poses and movements. This will do
     * nothing if the entity render object does not implement {@link class_922} or if the
     * model does not implement {@link class_572}).
     *
     * @param entity The wearer of the trinket
     * @param model  The model to align to the body movement
     *
     * @deprecated Use {@link #transformToFace(class_4587, class_630, Side)} or {@link #transformToModelPart(class_4587, class_630)} instead
     */
    @SuppressWarnings("unchecked")
    @Deprecated(forRemoval = true)
    static void followBodyRotations(final class_1309 entity, final class_572<class_1309> model) {
        class_897<? super class_1309> render = class_310.method_1551().method_1561().method_3953(entity);

        if (render instanceof class_922 renderer && renderer.method_4038() instanceof class_572 entityModel) {
            entityModel.method_2818(model);
        }
    }

    /**
     * Translates the rendering context to the center of the player's face
     *
     * @deprecated Use {@link #transformToFace(class_4587, class_630, Side)} or {@link #transformToModelPart(class_4587, class_630)} instead
     */
    @Deprecated
    static void translateToFace(class_4587 poseStack, class_572<? extends class_1309> model, class_1309 entity) {
        transformToFace(poseStack, model.field_3398, Side.FRONT);
    }

    /**
     * Translates the rendering context to the center of the player's chest/torso segment
     *
     * @deprecated Use {@link #transformToFace(class_4587, class_630, Side)} or {@link #transformToModelPart(class_4587, class_630)} instead
     */
    @Deprecated(forRemoval = true)
    static void translateToChest(class_4587 poseStack, class_572<? extends class_1309> model, class_1309 livingEntity) {
        transformToModelPart(poseStack, model.field_3391);
    }

    /**
     * Translates the rendering context to the center of the bottom of the player's right arm
     *
     * @deprecated Use {@link #transformToFace(class_4587, class_630, Side)} or {@link #transformToModelPart(class_4587, class_630)} instead
     */
    @Deprecated(forRemoval = true)
    static void translateToRightArm(class_4587 poseStack, class_572<? extends class_1309> model, class_1309 player) {
        transformToFace(poseStack, model.field_3401, Side.BOTTOM);
    }

    /**
     * Translates the rendering context to the center of the bottom of the player's left arm
     *
     * @deprecated Use {@link #transformToFace(class_4587, class_630, Side)} or {@link #transformToModelPart(class_4587, class_630)} instead
     */
    @Deprecated(forRemoval = true)
    static void translateToLeftArm(class_4587 poseStack, class_572<? extends class_1309> model, class_1309 player) {
        transformToFace(poseStack, model.field_27433, Side.BOTTOM);
    }

    /**
     * Translates the rendering context to the center of the bottom of the player's right leg
     *
     * @deprecated Use {@link #transformToFace(class_4587, class_630, Side)} or {@link #transformToModelPart(class_4587, class_630)} instead
     */
    @Deprecated(forRemoval = true)
    static void translateToRightLeg(class_4587 poseStack, class_572<? extends class_1309> model, class_1309 player) {
        transformToFace(poseStack, model.field_3392, Side.BOTTOM);
    }

    /**
     * Translates the rendering context to the center of the bottom of the player's left leg
     *
     * @deprecated Use {@link #transformToFace(class_4587, class_630, Side)} or {@link #transformToModelPart(class_4587, class_630)} instead
     */
    @Deprecated(forRemoval = true)
    static void translateToLeftLeg(class_4587 poseStack, class_572<? extends class_1309> model, class_1309 player) {
        transformToFace(poseStack, model.field_3397, Side.BOTTOM);
    }

    /**
     * Transforms the rendering context to a specific face on a ModelPart
     *
     * @param poseStack the pose stack to apply the transformation(s) to
     * @param part      The ModelPart to transform to
     * @param side      The side of the ModelPart to transform to
     */
    static void transformToFace(class_4587 poseStack, class_630 part, Side side) {
        transformToModelPart(poseStack, part, side.direction.method_10163().method_10263(), side.direction.method_10163().method_10264(), side.direction.method_10163().method_10260());
    }

    /**
     * Transforms the rendering context to the center of a ModelPart
     *
     * @param poseStack the pose stack to apply the transformation(s) to
     * @param part      The ModelPart to transform to
     */
    static void transformToModelPart(class_4587 poseStack, class_630 part) {
        transformToModelPart(poseStack, part, 0, 0, 0);
    }

    /**
     * Transforms the rendering context to a specific place relative to a ModelPart
     *
     * @param poseStack the pose stack to apply the transformation(s) to
     * @param part      The ModelPart to transform to
     * @param xPercent  The percentage of the x-axis to translate to
     *                  <p>
     *                  (-1 being the left side and 1 being the right side)
     *                  <p>
     *                  If null, will be ignored
     * @param yPercent  The percentage of the y-axis to translate to
     *                  <p>
     *                  (-1 being the bottom and 1 being the top)
     *                  <p>
     *                  If null, will be ignored
     * @param zPercent  The percentage of the z-axis to translate to
     *                  <p>
     *                  (-1 being the back and 1 being the front)
     *                  <p>
     *                  If null, will be ignored
     */
    static void transformToModelPart(class_4587 poseStack, class_630 part, @Nullable Number xPercent, @Nullable Number yPercent, @Nullable Number zPercent) {
        part.method_22703(poseStack);
        var aabb = getAABB(part);
        poseStack.method_22905(1 / 16f, 1 / 16f, 1 / 16f);
        poseStack.method_22904(
                xPercent != null ? class_3532.method_16436((-xPercent.doubleValue() + 1) / 2, aabb.getFirst().field_1352, aabb.getSecond().field_1352) : 0,
                yPercent != null ? class_3532.method_16436((-yPercent.doubleValue() + 1) / 2, aabb.getFirst().field_1351, aabb.getSecond().field_1351) : 0,
                zPercent != null ? class_3532.method_16436((-zPercent.doubleValue() + 1) / 2, aabb.getFirst().field_1350, aabb.getSecond().field_1350) : 0
        );
        poseStack.method_22905(8, 8, 8);
        poseStack.method_22907(class_7833.field_40714.rotationDegrees(180));
    }

    private static Pair<class_243, class_243> getAABB(class_630 part) {
        class_243 min = new class_243(0, 0, 0);
        class_243 max = new class_243(0, 0, 0);

        if (part.getClass().getSimpleName().contains("EMFModelPart")) {
            var parts = new ArrayList<class_630>();

            parts.add(part);
            parts.addAll(((ModelPartAccessor) (Object) part).getChildren().values());

            for (var modelPart : parts) {
                for (class_630.class_628 cube : ((ModelPartAccessor) (Object) modelPart).getCubes()) {
                    min = new class_243(
                            Math.min(min.field_1352, Math.min(cube.field_3645 + modelPart.field_3657, cube.field_3648 + modelPart.field_3657)),
                            Math.min(min.field_1351, Math.min(cube.field_3644 + modelPart.field_3656, cube.field_3647 + modelPart.field_3656)),
                            Math.min(min.field_1350, Math.min(cube.field_3643 + modelPart.field_3655, cube.field_3646 + modelPart.field_3655))
                    );
                    max = new class_243(
                            Math.max(max.field_1352, Math.max(cube.field_3645 + modelPart.field_3657, cube.field_3648 + modelPart.field_3657)),
                            Math.max(max.field_1351, Math.max(cube.field_3644 + modelPart.field_3656, cube.field_3647 + modelPart.field_3656)),
                            Math.max(max.field_1350, Math.max(cube.field_3643 + modelPart.field_3655, cube.field_3646 + modelPart.field_3655))
                    );
                }
            }
        } else {
            for (class_630.class_628 cube : ((ModelPartAccessor) (Object) part).getCubes()) {
                min = new class_243(
                        Math.min(min.field_1352, Math.min(cube.field_3645, cube.field_3648)),
                        Math.min(min.field_1351, Math.min(cube.field_3644, cube.field_3647)),
                        Math.min(min.field_1350, Math.min(cube.field_3643, cube.field_3646))
                );
                max = new class_243(
                        Math.max(max.field_1352, Math.max(cube.field_3645, cube.field_3648)),
                        Math.max(max.field_1351, Math.max(cube.field_3644, cube.field_3647)),
                        Math.max(max.field_1350, Math.max(cube.field_3643, cube.field_3646))
                );
            }
        }

        return Pair.of(min, max);
    }
}
