package io.wispforest.accessories.api.slot;

import com.google.common.collect.Lists;
import io.wispforest.accessories.api.AccessoriesAPI;
import io.wispforest.accessories.api.Accessory;
import io.wispforest.accessories.api.AccessoryNest;
import it.unimi.dsi.fastutil.Pair;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_1309;
import net.minecraft.class_1799;

/**
 * Special {@link SlotReference} implementation that references an accessory nested inside an {@link AccessoryNest}.
 *
 * @param innerSlotIndices the path used to get to the nested accessory
 */
@ApiStatus.Internal
public record NestedSlotReferenceImpl(class_1309 entity, String slotName, int initialHolderSlot, List<Integer> innerSlotIndices) implements SlotReference {

    public NestedSlotReferenceImpl {
        if(initialHolderSlot < -1) {
            throw new IndexOutOfBoundsException("A given Nested Slot Reference was attempted to be created with a negative initialHolderSlot value!");
        }
    }

    public String createSlotPath() {
        var slotPath = new StringBuilder(SlotReference.super.createSlotPath());

        var innerSlotIndices = this.innerSlotIndices();

        for (int i = 0; i < innerSlotIndices.size(); i++) {
            var innerIndex = innerSlotIndices.get(i);
            slotPath.append("/nest_")
                    .append(i)
                    .append("_")
                    .append(innerIndex);
        }

        return slotPath.toString();
    }

    @Override
    public boolean isValid() {
        if(!SlotReference.super.isValid()) return false;

        var selectedStack = SlotReference.super.getStack();

        for (var innerSlotIndex : this.innerSlotIndices()) {
            var innerData = tryAndGet(selectedStack, innerSlotIndex);

            if(innerData == null) return false;

            selectedStack = innerData.right();
        }

        return true;
    }

    @Override
    public int slot() {
        return initialHolderSlot();
    }

    @Override
    @Nullable
    public class_1799 getStack() {
        var selectedStack = SlotReference.super.getStack();

        for (var innerSlotIndex : this.innerSlotIndices()) {
            var innerData = tryAndGet(selectedStack, innerSlotIndex);

            if(innerData == null) return null;

            selectedStack = innerData.right();
        }

        return selectedStack;
    }

    @Nullable
    private static Pair<NestLayer, class_1799> tryAndGet(class_1799 holderStack, int innerIndex) {
        var accessory = AccessoriesAPI.getAccessory(holderStack);

        if(!(accessory instanceof AccessoryNest accessoryNest)) return null;

        return Pair.of(new NestLayer(accessoryNest, holderStack, innerIndex), accessoryNest.getInnerStacks(holderStack).get(innerIndex));
    }

    @Override
    public boolean setStack(class_1799 stack) {
        var selectedStack = SlotReference.super.getStack();

        if(selectedStack == null) return false;

        var layerStack = new ArrayList<NestLayer>();

        for (var innerSlotIndex : innerSlotIndices()) {
            var innerData = tryAndGet(selectedStack, innerSlotIndex);

            if(innerData == null) return false;

            layerStack.add(innerData.first());
            selectedStack = innerData.right();
        }

        var innerStack = stack;

        for (var layer : Lists.reverse(layerStack)){
            if(!layer.setStack(innerStack)) return false;

            innerStack = layer.holderStack();
        }

        SlotReference.super.setStack(innerStack);

        return true;
    }

    private record NestLayer(AccessoryNest accessoryNest, class_1799 holderStack, int index) {
        private boolean setStack(class_1799 innerStack) {
            return accessoryNest.setInnerStack(holderStack, index, innerStack);
        }
    }
}
