package io.wispforest.accessories.data;

import com.google.common.collect.ImmutableMap;
import com.google.gson.*;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.AccessoriesInternals;
import io.wispforest.accessories.api.slot.ExtraSlotTypeProperties;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.api.slot.UniqueSlotHandling;
import it.unimi.dsi.fastutil.Pair;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

import java.util.*;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import net.minecraft.class_1299;
import net.minecraft.class_1309;
import net.minecraft.class_1937;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_3695;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import net.minecraft.class_7924;

/**
 * Resource Reload in which handles the loading of {@link SlotType}'s bindings
 * to the targeted {@link class_1299} though a {@link class_6862} or {@link class_2960}
 */
public class EntitySlotLoader extends ReplaceableJsonResourceReloadListener {

    private static final Logger LOGGER = LogUtils.getLogger();
    private static final Gson GSON = new GsonBuilder().setLenient().setPrettyPrinting().create();

    public static final EntitySlotLoader INSTANCE = new EntitySlotLoader();

    private Map<class_1299<?>, Map<String, SlotType>> server = new HashMap<>();
    private Map<class_1299<?>, Map<String, SlotType>> client = new HashMap<>();

    protected EntitySlotLoader() {
        super(GSON, LOGGER, "accessories/entity");
    }

    //--

    /**
     * @return The valid {@link SlotType}'s for given {@link class_1309} based on its {@link class_1299}
     */
    public static Map<String, SlotType> getEntitySlots(class_1309 livingEntity){
        return getEntitySlots(livingEntity.method_37908(), livingEntity.method_5864());
    }

    /**
     * @return The valid {@link SlotType}'s for given {@link class_1299}
     */
    public static Map<String, SlotType> getEntitySlots(class_1937 level, class_1299<?> entityType){
        var map = EntitySlotLoader.INSTANCE.getSlotTypes(level.field_9236, entityType);

        return map != null ? map : Map.of();
    }

    //--

    @Nullable
    public final Map<String, SlotType> getSlotTypes(boolean isClientSide, class_1299<?> entityType){
        return this.getEntitySlotData(isClientSide).get(entityType);
    }

    @ApiStatus.Internal
    public final Map<class_1299<?>, Map<String, SlotType>> getEntitySlotData(boolean isClientSide){
        return isClientSide ? this.client : this.server;
    }

    @ApiStatus.Internal
    public final void setEntitySlotData(Map<class_1299<?>, Map<String, SlotType>> data){
        this.client = ImmutableMap.copyOf(data);
    }

    //--

    @Override
    protected void apply(Map<class_2960, JsonObject> data, class_3300 resourceManager, class_3695 profiler) {
        var allSlotTypes = SlotTypeLoader.INSTANCE.getSlotTypes(false);

        var tempMap = new HashMap<class_1299<?>, Map<String, SlotType>>();

        for (var resourceEntry : data.entrySet()) {
            var location = resourceEntry.getKey();
            var jsonObject = resourceEntry.getValue();

            if(!AccessoriesInternals.isValidOnConditions(jsonObject, this.directory, location, null)) continue;

            var slots = new HashMap<String, SlotType>();

            var slotElements = this.safeHelper(class_3518::method_15261, jsonObject, "slots", new JsonArray(), location);

            this.decodeJsonArray(slotElements, "slot", location, element -> {
                var slotName = element.getAsString();

                return Pair.of(slotName, allSlotTypes.get(slotName));
            }, slotInfo -> {
                var slotType = slotInfo.right();

                if(slotType != null) {
                    if(!ExtraSlotTypeProperties.getProperty(slotInfo.left(), false).strictMode()) {
                        slots.put(slotType.name(), slotType);
                    } else {
                        LOGGER.warn("Unable to add the given slot to the given group due to it being in strict mode! [Slot: {}]", slotInfo.left());
                    }
                } else if (slotType == null) {
                    LOGGER.warn("Unable to locate a given slot to add to a given entity('s) as it was not registered: [Slot: {}]", slotInfo.first());
                }
            });

            //--

            var entities = new ArrayList<class_1299<?>>();

            var entityElements = this.safeHelper(class_3518::method_15261, jsonObject, "entities", new JsonArray(), location);

            this.decodeJsonArray(entityElements, "entity", location, element -> {
                var string = element.getAsString();

                if(string.contains("#")){
                    var entityTypeTagLocation = class_2960.method_12829(string.replace("#", ""));

                    var entityTypeTag = class_6862.method_40092(class_7924.field_41266, entityTypeTagLocation);

                    return AccessoriesInternals.getHolder(entityTypeTag)
                            .map(holders -> holders.stream().map(class_6880::comp_349).collect(Collectors.toSet()))
                            .orElseGet(() -> {
                                LOGGER.warn("[EntitySlotLoader]: Unable to locate the given EntityType Tag used within a slot entry: [Location: {}]", string);
                                return Set.of();
                            });
                } else {
                    return Optional.ofNullable(class_2960.method_12829(string))
                            .map(location1 -> class_7923.field_41177.method_17966(location1).map(Set::of).orElse(Set.of()))
                            .orElseGet(() -> {
                                LOGGER.warn("[EntitySlotLoader]: Unable to locate the given EntityType within the registries for a slot entry: [Location: {}]", string);
                                return Set.of();
                            });
                }
            }, entities::addAll);

            for (class_1299<?> entityType : entities) {
                tempMap.computeIfAbsent(entityType, entityType1 -> new HashMap<>())
                        .putAll(slots);
            }
        }

        for (var entry : UniqueSlotHandling.getSlotToEntities().entrySet()) {
            var slotType = SlotTypeLoader.INSTANCE.getSlotTypes(false).get(entry.getKey());

            for (var entityType : entry.getValue()) {
                tempMap.computeIfAbsent(entityType, entityType1 -> new HashMap<>())
                        .put(slotType.name(), slotType);
            }
        }
        
        this.server = ImmutableMap.copyOf(tempMap);
    }
}
