package io.wispforest.accessories.data;

import ;
import Z;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import com.google.gson.*;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.Accessories;
import io.wispforest.accessories.AccessoriesInternals;
import io.wispforest.accessories.api.slot.SlotGroup;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.api.slot.UniqueSlotHandling;
import io.wispforest.accessories.impl.SlotGroupImpl;
import org.jetbrains.annotations.ApiStatus;
import org.slf4j.Logger;

import java.util.*;
import java.util.Map.Entry;
import net.minecraft.class_1937;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_3695;

public class SlotGroupLoader extends ReplaceableJsonResourceReloadListener {

    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().setLenient().create();
    private static final Logger LOGGER = LogUtils.getLogger();

    public static final SlotGroupLoader INSTANCE = new SlotGroupLoader();

    private Map<String, SlotGroup> server = new HashMap<>();
    private Map<String, SlotGroup> client = new HashMap<>();

    protected SlotGroupLoader() {
        super(GSON, LOGGER, "accessories/group");
    }

    //--

    public static List<SlotGroup> getGroups(class_1937 level){
        return INSTANCE.getGroups(level.method_8608(), true);
    }

    public static List<SlotGroup> getGroups(class_1937 level, boolean filterUniqueGroups){
        return INSTANCE.getGroups(level.method_8608(), filterUniqueGroups);
    }

    public static Optional<SlotGroup> getGroup(class_1937 level, String group){
        return Optional.ofNullable(INSTANCE.getGroup(level.method_8608(), group));
    }

    //--

    @ApiStatus.Internal
    public final Map<String, SlotGroup> getGroupMap(boolean isClientSide) {
        return (isClientSide ? this.client : this.server);
    }

    public final List<SlotGroup> getGroups(boolean isClientSide, boolean filterUniqueGroups){
        var groups = getGroupMap(isClientSide).values();

        if(filterUniqueGroups) groups = groups.stream().filter(group -> !UniqueSlotHandling.isUniqueGroup(group.name(), isClientSide)).toList();

        return List.copyOf(groups);
    }

    public final SlotGroup getGroup(boolean isClientSide, String group){
        return getGroupMap(isClientSide).get(group);
    }

    public final Optional<SlotGroup> findGroup(boolean isClientSide, String slot){
        for (var entry : getGroups(isClientSide, false)) {
            if(entry.slots().contains(slot)) return Optional.of(entry);
        }

        return Optional.empty();
    }

    public final SlotGroup getOrDefaultGroup(boolean isClientSide, String slot){
        var groups = getGroupMap(isClientSide);

        for (var entry : groups.values()) {
            if(entry.slots().contains(slot)) return entry;
        }

        return groups.get("any");
    }

    @ApiStatus.Internal
    public final void setGroups(Map<String, SlotGroup> groups){
        this.client = ImmutableMap.copyOf(groups);
    }

    @Override
    protected void apply(Map<class_2960, JsonObject> data, class_3300 resourceManager, class_3695 profiler) {
        var slotGroups = new HashMap<String, SlotGroupBuilder>();

        slotGroups.put("unsorted", new SlotGroupBuilder("unsorted").order(30));

        var allSlots = new HashMap<>(SlotTypeLoader.INSTANCE.getSlotTypes(false));

        for (var resourceEntry : data.entrySet()) {
            var location = resourceEntry.getKey();
            var jsonObject = resourceEntry.getValue();

            if(!AccessoriesInternals.isValidOnConditions(jsonObject, this.directory, location, null)) continue;

            var pathParts = location.method_12832().split("/");

            String groupName = pathParts[pathParts.length - 1];
            String namespace = pathParts.length > 1 ? pathParts[0] + ":" : "";

            var isShared = namespace.isBlank();

            if(!isShared) groupName = namespace + ":" + groupName;

            var group = slotGroups.computeIfAbsent(groupName, SlotGroupBuilder::new);

            if(isShared) {
                var slotElements = safeHelper(class_3518::method_15261, jsonObject, "slots", new JsonArray(), location);

                decodeJsonArray(slotElements, "slot", location, JsonElement::getAsString, s -> {
                    for (var builderEntry : slotGroups.entrySet()) {
                        if (builderEntry.getValue().slots.contains(s)) {
                            LOGGER.error("Unable to assign a give slot [{}] to the group [{}] as it already exists within the group [{}]", s, group, builderEntry.getKey());
                            return;
                        }
                    }

                    var slotType = allSlots.remove(s);

                    if (slotType == null) {
                        LOGGER.warn("SlotType added to a given group without being in the main map for slots! [Name: {}]", slotType.name());
                    } else {
                        group.addSlot(s);
                    }
                });

                group.order(safeHelper(class_3518::method_15260, jsonObject, "order", 100, location));
            }

            var icon = safeHelper(class_3518::method_15265, jsonObject, "icon", location);

            if(icon != null){
                var iconLocation = class_2960.method_12829(icon);

                if(iconLocation != null){
                    group.icon(iconLocation);
                } else {
                    LOGGER.warn("A given SlotGroup was found to have a invalid Icon Location. [Location: {}]", location);
                }
            }
        }

        var remainSlots = new HashSet<String>();

        for (var value : allSlots.values()) {
            var slotName = value.name();

            if(!UniqueSlotHandling.isUniqueSlot(slotName)) {
                remainSlots.add(slotName);

                continue;
            }

            var group = slotName.split(":")[0];

            slotGroups.computeIfAbsent(group, SlotGroupBuilder::new)
                    .order(5)
                    .addSlot(slotName);

            UniqueSlotHandling.addGroup(group);
        }

        slotGroups.get("unsorted").addSlots(remainSlots);

        var tempMap = ImmutableMap.<String, SlotGroup>builder();

        slotGroups.forEach((s, builder) -> tempMap.put(s, builder.build()));

        this.server = tempMap.build();
    }

    public static class SlotGroupBuilder {
        private final String name;

        private Integer order = null;
        private final Set<String> slots = new LinkedHashSet<>();

        private class_2960 iconLocation = SlotGroup.UNKNOWN;

        public SlotGroupBuilder(String name){
            this.name = name;
        }

        public SlotGroupBuilder order(Integer value){
            this.order = value;

            return this;
        }

        public SlotGroupBuilder addSlot(String value){
            this.slots.add(value);

            return this;
        }

        public SlotGroupBuilder addSlots(Collection<String> values){
            this.slots.addAll(values);

            return this;
        }

        public SlotGroupBuilder icon(class_2960 location) {
            this.iconLocation = location;

            return this;
        }

        public SlotGroup build(){
            return new SlotGroupImpl(
                    name,
                    Optional.ofNullable(order).orElse(0),
                    slots,
                    iconLocation
            );
        }
    }
}
