package io.wispforest.accessories.api.slot;

import com.google.common.collect.ImmutableList;
import io.wispforest.accessories.api.AccessoriesCapability;
import io.wispforest.accessories.api.AccessoriesContainer;
import io.wispforest.accessories.api.Accessory;
import io.wispforest.accessories.api.AccessoryNest;
import io.wispforest.accessories.data.SlotTypeLoader;
import io.wispforest.endec.Endec;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.item.ItemStack;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.UUID;

/**
 * A reference to a specific accessory slot of a {@link LivingEntity}.
 */
public sealed interface SlotReference permits NestedSlotReferenceImpl, SlotReferenceImpl {

    static SlotReference of(LivingEntity livingEntity, String slotName, int slot) {
        return new SlotReferenceImpl(livingEntity, slotName, slot);
    }

    static SlotReference ofNest(LivingEntity livingEntity, String slotName, int initialHolderSlot, List<Integer> innerSlotIndices) {
        return new NestedSlotReferenceImpl(livingEntity, slotName, initialHolderSlot, ImmutableList.copyOf(innerSlotIndices));
    }

    /**
     * @return the referenced slot name
     */
    String slotName();

    /**
     * @return the referenced entity
     */
    LivingEntity entity();

    /**
     * @return the referenced slot index
     */
    int slot();

    //--

    default boolean isValid() {
        var capability = this.capability();

        if(capability == null) return false;

        var container = capability.getContainers().get(this.slotName());

        if(container == null) return false;

        return slot() < container.getSize();
    }

    default String createSlotPath() {
        return this.slotName().replace(":", "-") + "/" + this.slot();
    }

    @Nullable
    default SlotType type(){
        return SlotTypeLoader.getSlotType(entity().m_9236_(), slotName());
    }

    @Nullable
    default AccessoriesCapability capability() {
        return this.entity().accessoriesCapability();
    }

    @Nullable
    default AccessoriesContainer slotContainer() {
        var capability = this.capability();

        if(capability == null) return null;

        return capability.getContainers().get(slotName());
    }

    /**
     * @return the current referenced stack
     */
    @Nullable
    default ItemStack getStack() {
        var container = this.slotContainer();

        if(container == null) return null;

        return container.getAccessories().m_8020_(slot());
    }

    /**
     * @return {@code true} if the stack was successfully set, {@code false} otherwise
     */
    default boolean setStack(ItemStack stack) {
        var container = this.slotContainer();

        if(container == null) return false;

        container.getAccessories().m_6836_(slot(), stack);

        return true;
    }
}
