package io.wispforest.accessories.data;

import com.google.common.collect.ImmutableMap;
import com.google.gson.*;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.AccessoriesInternals;
import io.wispforest.accessories.api.slot.ExtraSlotTypeProperties;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.api.slot.UniqueSlotHandling;
import it.unimi.dsi.fastutil.Pair;
import net.minecraft.core.Holder;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.core.registries.Registries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.packs.resources.ResourceManager;
import net.minecraft.tags.TagKey;
import net.minecraft.util.GsonHelper;
import net.minecraft.util.profiling.ProfilerFiller;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.level.Level;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

import java.util.*;
import java.util.Map.Entry;
import java.util.stream.Collectors;

/**
 * Resource Reload in which handles the loading of {@link SlotType}'s bindings
 * to the targeted {@link EntityType} though a {@link TagKey} or {@link ResourceLocation}
 */
public class EntitySlotLoader extends ReplaceableJsonResourceReloadListener {

    private static final Logger LOGGER = LogUtils.getLogger();
    private static final Gson GSON = new GsonBuilder().setLenient().setPrettyPrinting().create();

    public static final EntitySlotLoader INSTANCE = new EntitySlotLoader();

    private Map<EntityType<?>, Map<String, SlotType>> server = new HashMap<>();
    private Map<EntityType<?>, Map<String, SlotType>> client = new HashMap<>();

    protected EntitySlotLoader() {
        super(GSON, LOGGER, "accessories/entity");
    }

    //--

    /**
     * @return The valid {@link SlotType}'s for given {@link LivingEntity} based on its {@link EntityType}
     */
    public static Map<String, SlotType> getEntitySlots(LivingEntity livingEntity){
        return getEntitySlots(livingEntity.m_9236_(), livingEntity.m_6095_());
    }

    /**
     * @return The valid {@link SlotType}'s for given {@link EntityType}
     */
    public static Map<String, SlotType> getEntitySlots(Level level, EntityType<?> entityType){
        var map = EntitySlotLoader.INSTANCE.getSlotTypes(level.f_46443_, entityType);

        return map != null ? map : Map.of();
    }

    //--

    @Nullable
    public final Map<String, SlotType> getSlotTypes(boolean isClientSide, EntityType<?> entityType){
        return this.getEntitySlotData(isClientSide).get(entityType);
    }

    @ApiStatus.Internal
    public final Map<EntityType<?>, Map<String, SlotType>> getEntitySlotData(boolean isClientSide){
        return isClientSide ? this.client : this.server;
    }

    @ApiStatus.Internal
    public final void setEntitySlotData(Map<EntityType<?>, Map<String, SlotType>> data){
        this.client = ImmutableMap.copyOf(data);
    }

    //--

    @Override
    protected void apply(Map<ResourceLocation, JsonObject> data, ResourceManager resourceManager, ProfilerFiller profiler) {
        var allSlotTypes = SlotTypeLoader.INSTANCE.getSlotTypes(false);

        var tempMap = new HashMap<EntityType<?>, Map<String, SlotType>>();

        for (var resourceEntry : data.entrySet()) {
            var location = resourceEntry.getKey();
            var jsonObject = resourceEntry.getValue();

            if(!AccessoriesInternals.isValidOnConditions(jsonObject, this.directory, location, null)) continue;

            var slots = new HashMap<String, SlotType>();

            var slotElements = this.safeHelper(GsonHelper::m_13933_, jsonObject, "slots", new JsonArray(), location);

            this.decodeJsonArray(slotElements, "slot", location, element -> {
                var slotName = element.getAsString();

                return Pair.of(slotName, allSlotTypes.get(slotName));
            }, slotInfo -> {
                var slotType = slotInfo.right();

                if(slotType != null) {
                    if(!ExtraSlotTypeProperties.getProperty(slotInfo.left(), false).strictMode()) {
                        slots.put(slotType.name(), slotType);
                    } else {
                        LOGGER.warn("Unable to add the given slot to the given group due to it being in strict mode! [Slot: {}]", slotInfo.left());
                    }
                } else if (slotType == null) {
                    LOGGER.warn("Unable to locate a given slot to add to a given entity('s) as it was not registered: [Slot: {}]", slotInfo.first());
                }
            });

            //--

            var entities = new ArrayList<EntityType<?>>();

            var entityElements = this.safeHelper(GsonHelper::m_13933_, jsonObject, "entities", new JsonArray(), location);

            this.decodeJsonArray(entityElements, "entity", location, element -> {
                var string = element.getAsString();

                if(string.contains("#")){
                    var entityTypeTagLocation = ResourceLocation.m_135820_(string.replace("#", ""));

                    var entityTypeTag = TagKey.m_203882_(Registries.f_256939_, entityTypeTagLocation);

                    return AccessoriesInternals.getHolder(entityTypeTag)
                            .map(holders -> holders.stream().map(Holder::m_203334_).collect(Collectors.toSet()))
                            .orElseGet(() -> {
                                LOGGER.warn("[EntitySlotLoader]: Unable to locate the given EntityType Tag used within a slot entry: [Location: {}]", string);
                                return Set.of();
                            });
                } else {
                    return Optional.ofNullable(ResourceLocation.m_135820_(string))
                            .map(location1 -> BuiltInRegistries.f_256780_.m_6612_(location1).map(Set::of).orElse(Set.of()))
                            .orElseGet(() -> {
                                LOGGER.warn("[EntitySlotLoader]: Unable to locate the given EntityType within the registries for a slot entry: [Location: {}]", string);
                                return Set.of();
                            });
                }
            }, entities::addAll);

            for (EntityType<?> entityType : entities) {
                tempMap.computeIfAbsent(entityType, entityType1 -> new HashMap<>())
                        .putAll(slots);
            }
        }

        for (var entry : UniqueSlotHandling.getSlotToEntities().entrySet()) {
            var slotType = SlotTypeLoader.INSTANCE.getSlotTypes(false).get(entry.getKey());

            for (var entityType : entry.getValue()) {
                tempMap.computeIfAbsent(entityType, entityType1 -> new HashMap<>())
                        .put(slotType.name(), slotType);
            }
        }
        
        this.server = ImmutableMap.copyOf(tempMap);
    }
}
