package io.wispforest.accessories.impl;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import io.wispforest.accessories.Accessories;
import io.wispforest.accessories.AccessoriesInternals;
import io.wispforest.accessories.api.*;
import io.wispforest.accessories.api.attributes.AccessoryAttributeBuilder;
import io.wispforest.accessories.api.components.AccessoriesDataComponents;
import io.wispforest.accessories.api.components.AccessoryItemAttributeModifiers;
import io.wispforest.accessories.api.components.AccessoryNestContainerContents;
import io.wispforest.accessories.api.events.*;
import io.wispforest.accessories.api.slot.SlotEntryReference;
import io.wispforest.accessories.api.slot.SlotReference;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.api.slot.UniqueSlotHandling;
import io.wispforest.accessories.client.AccessoriesMenu;
import io.wispforest.accessories.data.EntitySlotLoader;
import io.wispforest.accessories.data.SlotTypeLoader;
import io.wispforest.accessories.endec.NbtMapCarrier;
import io.wispforest.accessories.networking.client.SyncEntireContainer;
import io.wispforest.accessories.networking.base.BaseNetworkHandler;
import io.wispforest.accessories.networking.client.SyncContainerData;
import io.wispforest.accessories.networking.client.SyncData;
import io.wispforest.accessories.utils.AttributeUtils;
import io.wispforest.endec.SerializationContext;
import it.unimi.dsi.fastutil.Pair;
import net.fabricmc.fabric.api.util.TriState;
import net.minecraft.ChatFormatting;
import net.minecraft.core.registries.Registries;
import net.minecraft.network.chat.CommonComponents;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.server.players.PlayerList;
import net.minecraft.tags.TagKey;
import net.minecraft.world.*;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.ai.attributes.Attribute;
import net.minecraft.world.entity.ai.attributes.AttributeModifier;
import net.minecraft.world.entity.ai.attributes.Attributes;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.TooltipFlag;
import net.minecraft.world.item.enchantment.EnchantmentHelper;
import net.minecraft.world.level.GameRules;
import net.minecraft.world.level.Level;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import static io.wispforest.accessories.Accessories.ACCESSORY_EQUIPPED;
import static io.wispforest.accessories.Accessories.ACCESSORY_UNEQUIPPED;

import I;
import Z;

@ApiStatus.Internal
public class AccessoriesEventHandler {

    public static boolean dataReloadOccurred = false;

    public static void onWorldTick(Level level) {
        if (!(level instanceof ServerLevel serverLevel)) return;

        revalidatePlayersOnReload(serverLevel.m_7654_().m_6846_());
    }

    public static void revalidatePlayersOnReload(PlayerList playerList) {
        if(!dataReloadOccurred) return;

        for (var player : playerList.m_11314_()) revalidatePlayer(player);

        dataReloadOccurred = false;
    }

    public static void revalidatePlayer(ServerPlayer player) {
        var capability = AccessoriesCapability.get(player);

        if (capability == null) return;

        var validSlotTypes = EntitySlotLoader.getEntitySlots(player).values();

        for (var container : capability.getContainers().values()) {
            var slotType = container.slotType();

            if (slotType != null && validSlotTypes.contains(slotType)) {
                var baseSize = ((AccessoriesContainerImpl) container).getBaseSize();

                if (baseSize == null || baseSize != slotType.amount()) {
                    container.markChanged();
                    container.update();
                }

                var stacks = container.getAccessories();
                var cosmeticStacks = container.getCosmeticAccessories();

                for (int i = 0; i < container.getSize(); i++) {
                    var reference = container.createReference(i);

                    handleInvalidStacks(stacks, reference, player);
                    handleInvalidStacks(cosmeticStacks, reference, player);
                }
            } else {
                // TODO: DROP CONTAINER ?!
                var stacks = container.getAccessories();
                var cosmeticStacks = container.getCosmeticAccessories();

                for (int i = 0; i < container.getSize(); i++) {
                    var reference = container.createReference(i);

                    dropAndRemoveStack(stacks, reference, player);
                    dropAndRemoveStack(cosmeticStacks, reference, player);
                }
            }
        }
    }

    private static void handleInvalidStacks(Container container, SlotReference reference, ServerPlayer player) {
        var bl = !AccessoriesAPI.canInsertIntoSlot(container.m_8020_(reference.slot()), reference);

        if (bl) dropAndRemoveStack(container, reference, player);
    }

    private static void dropAndRemoveStack(Container container, SlotReference reference, ServerPlayer player) {
        var stack = container.m_8020_(reference.slot());

        container.m_6836_(reference.slot(), ItemStack.f_41583_);

        AccessoriesInternals.giveItemToPlayer(player, stack);
    }

    public static void entityLoad(LivingEntity entity, Level level) {
        if (!level.m_5776_() || !(entity instanceof ServerPlayer serverPlayer)) return;

        var capability = AccessoriesCapability.get(serverPlayer);

        if(capability == null) return;

        var carrier = NbtMapCarrier.of();

        ((AccessoriesHolderImpl) capability.getHolder()).write(carrier, SerializationContext.empty());

        AccessoriesInternals.getNetworkHandler().sendToTrackingAndSelf(serverPlayer, new SyncEntireContainer(capability.entity().m_19879_(), carrier));
    }

    public static void onTracking(LivingEntity entity, ServerPlayer serverPlayer) {
        var capability = AccessoriesCapability.get(entity);

        if(capability == null) return;

        var carrier = NbtMapCarrier.of();

        ((AccessoriesHolderImpl) capability.getHolder()).write(carrier, SerializationContext.empty());

        AccessoriesInternals.getNetworkHandler().sendToPlayer(serverPlayer, new SyncEntireContainer(capability.entity().m_19879_(), carrier));
    }

    public static void dataSync(@Nullable PlayerList list, @Nullable ServerPlayer player) {
        var networkHandler = AccessoriesInternals.getNetworkHandler();
        var syncPacket = SyncData.create();

        if (list != null && !list.m_11314_().isEmpty()) {
            revalidatePlayersOnReload(list);

            // TODO: OPTIMIZE THIS?
            for (var playerEntry : list.m_11314_()) {
                networkHandler.sendToPlayer(playerEntry, syncPacket);

                var capability = AccessoriesCapability.get(playerEntry);

                if(capability == null) return;

                var carrier = NbtMapCarrier.of();

                ((AccessoriesHolderImpl) capability.getHolder()).write(carrier, SerializationContext.empty());

                networkHandler.sendToTrackingAndSelf(playerEntry, new SyncEntireContainer(capability.entity().m_19879_(), carrier));

                if(playerEntry.f_36096_ instanceof AccessoriesMenu accessoriesMenu) {
                    Accessories.openAccessoriesMenu(playerEntry, accessoriesMenu.targetEntity());
                }
            }
        } else if (player != null) {
            networkHandler.sendToPlayer(player, syncPacket);

            revalidatePlayer(player);

            var capability = AccessoriesCapability.get(player);

            if(capability == null) return;

            var carrier = NbtMapCarrier.of();

            ((AccessoriesHolderImpl) capability.getHolder()).write(carrier, SerializationContext.empty());

            networkHandler.sendToPlayer(player, new SyncEntireContainer(capability.entity().m_19879_(), carrier));

            if(player.f_36096_ instanceof AccessoriesMenu accessoriesMenu) {
                Accessories.openAccessoriesMenu(player, accessoriesMenu.targetEntity());
            }
        }
    }

    public static void onLivingEntityTick(LivingEntity entity) {
        if(entity.m_213877_()) return;

        var capability = AccessoriesCapability.get(entity);

        if (capability != null) {
            var dirtyStacks = new HashMap<String, ItemStack>();
            var dirtyCosmeticStacks = new HashMap<String, ItemStack>();

            var removedAttributesBuilder = new AccessoryAttributeBuilder();
            var addedAttributesBuilder = new AccessoryAttributeBuilder();

            for (var containerEntry : capability.getContainers().entrySet()) {
                var container = containerEntry.getValue();
                var slotType = container.slotType();

                var accessories = (ExpandedSimpleContainer) container.getAccessories();
                var cosmetics = container.getCosmeticAccessories();

                for (int i = 0; i < accessories.m_6643_(); i++) {
                    var slotReference = container.createReference(i);

                    var slotId = slotType.name() + "/" + i;

                    var currentStack = accessories.m_8020_(i);

                    // TODO: Move ticking below checks?
                    if (!currentStack.m_41619_()) {
                        // TODO: Document this behavior to prevent double ticking maybe!!!
                        currentStack.m_41666_(entity.m_9236_(), entity, -1, false);

                        var accessory = AccessoriesAPI.getAccessory(currentStack);

                        if (accessory != null) accessory.tick(currentStack, slotReference);
                    }

                    var lastStack = accessories.getPreviousItem(i);

                    // Prevent attribute related logic on the client and if the entity
                    // is dead as such data should not be updated. Though we allow for
                    // ticking to occur at least for vanilla parity I guess.
                    if (entity.m_9236_().m_5776_() || entity.m_21224_()) continue;

                    if (!ItemStack.m_41728_(currentStack, lastStack)) {
                        container.getAccessories().setPreviousItem(i, currentStack.m_41777_());
                        dirtyStacks.put(slotId, currentStack.m_41777_());

                        if (!lastStack.m_41619_()) {
                            removedAttributesBuilder.addFrom(AccessoriesAPI.getAttributeModifiers(lastStack, slotReference));
                        }

                        if (!currentStack.m_41619_()) {
                            addedAttributesBuilder.addFrom(AccessoriesAPI.getAttributeModifiers(currentStack, slotReference));
                        }

                        boolean equipmentChange = false;

                        /*
                         * TODO: Does item check need to exist anymore?
                         */
                        if (!ItemStack.m_41656_(currentStack, lastStack) || accessories.isSlotFlagged(i)) {
                            AccessoriesAPI.getOrDefaultAccessory(lastStack.m_41720_()).onUnequip(lastStack, slotReference);
                            AccessoriesAPI.getOrDefaultAccessory(currentStack.m_41720_()).onEquip(currentStack, slotReference);

                            if (entity instanceof ServerPlayer serverPlayer) {
                                if (!currentStack.m_41619_()) {
                                    ACCESSORY_EQUIPPED.trigger(serverPlayer, currentStack, slotReference, false);
                                }

                                if (!lastStack.m_41619_()) {
                                    ACCESSORY_UNEQUIPPED.trigger(serverPlayer, lastStack, slotReference, false);
                                }
                            }

                            equipmentChange = true;
                        }

                        AccessoryChangeCallback.EVENT.invoker().onChange(lastStack, currentStack, slotReference, equipmentChange ? SlotStateChange.REPLACEMENT : SlotStateChange.MUTATION);

                        recursiveStackChange(slotReference, AccessoryNestUtils.getData(lastStack), AccessoryNestUtils.getData(currentStack));
                    }

                    var currentCosmeticStack = cosmetics.m_8020_(i);
                    var lastCosmeticStack = container.getCosmeticAccessories().getPreviousItem(i);

                    if (!ItemStack.m_41728_(currentCosmeticStack, lastCosmeticStack)) {
                        cosmetics.setPreviousItem(i, currentCosmeticStack.m_41777_());
                        dirtyCosmeticStacks.put(slotId, currentCosmeticStack.m_41777_());

                        if (entity instanceof ServerPlayer serverPlayer) {
                            if (!currentStack.m_41619_()) {
                                ACCESSORY_EQUIPPED.trigger(serverPlayer, currentStack, slotReference, true);
                            }
                            if (!lastStack.m_41619_()) {
                                ACCESSORY_UNEQUIPPED.trigger(serverPlayer, lastStack, slotReference, true);
                            }
                        }
                    }
                }
            }

            if (entity.m_9236_().m_5776_()) return;

            AttributeUtils.removeTransientAttributeModifiers(entity, removedAttributesBuilder);
            AttributeUtils.addTransientAttributeModifiers(entity, addedAttributesBuilder);

            //--

            var updatedContainers = ((AccessoriesCapabilityImpl) capability).getUpdatingInventories();

            capability.updateContainers();

            ContainersChangeCallback.EVENT.invoker().onChange(entity, capability, ImmutableMap.copyOf(updatedContainers));

            if (!dirtyStacks.isEmpty() || !dirtyCosmeticStacks.isEmpty() || !updatedContainers.isEmpty()) {
                var packet = SyncContainerData.of(entity, updatedContainers.keySet(), dirtyStacks, dirtyCosmeticStacks);

                var networkHandler = AccessoriesInternals.getNetworkHandler();

                networkHandler.sendToTrackingAndSelf(entity, packet);
            }

            updatedContainers.clear();
        }

        //--

        var holder = ((AccessoriesHolderImpl) AccessoriesInternals.getHolder(entity));

        if(holder.loadedFromTag && capability == null) {
            var tempCapability = new AccessoriesCapabilityImpl(entity);
        }

        var invalidStacks = (holder).invalidStacks;

        if (!invalidStacks.isEmpty()) {
            for (ItemStack invalidStack : invalidStacks) {
                if (entity instanceof ServerPlayer serverPlayer) {
                    AccessoriesInternals.giveItemToPlayer(serverPlayer, invalidStack);
                } else {
                    entity.m_19983_(invalidStack);
                }
            }

            invalidStacks.clear();
        }
    }

    private static void recursiveStackChange(SlotReference slotReference, @Nullable AccessoryNestContainerContents lastNestData, @Nullable AccessoryNestContainerContents currentNestData) {
        var lastInnerStacks = lastNestData != null ? List.copyOf(lastNestData.getMap(slotReference).entrySet()) : List.<Map.Entry<SlotEntryReference, Accessory>>of();
        var currentInnerStacks = currentNestData != null ? List.copyOf(currentNestData.getMap(slotReference).entrySet()) : List.<Map.Entry<SlotEntryReference, Accessory>>of();

        var maxIterationLength = Math.max(lastInnerStacks.size(), currentInnerStacks.size());

        for (int i = 0; i < maxIterationLength; i++) {
            var lastInnerEntry = (i < lastInnerStacks.size()) ? lastInnerStacks.get(i) : null;
            var currentInnerEntry = (i < currentInnerStacks.size()) ? currentInnerStacks.get(i) : null;

            if(lastInnerEntry == null && currentInnerEntry != null) {
                var currentRef = currentInnerEntry.getKey();

                AccessoryChangeCallback.EVENT.invoker().onChange(ItemStack.f_41583_, currentRef.stack(), currentRef.reference(), SlotStateChange.REPLACEMENT);
            } else if(currentInnerEntry == null && lastInnerEntry != null) {
                var lastRef = lastInnerEntry.getKey();

                AccessoryChangeCallback.EVENT.invoker().onChange(lastRef.stack(), ItemStack.f_41583_, lastRef.reference(), SlotStateChange.REPLACEMENT);
            } else if(lastInnerEntry != null && currentInnerEntry != null) {
                var currentRef = currentInnerEntry.getKey();
                var lastRef = lastInnerEntry.getKey();

                var innerRef = lastRef.reference();

                var currentInnerStack = currentRef.stack();
                var lastInnerStack = lastRef.stack();

                AccessoryChangeCallback.EVENT.invoker().onChange(lastInnerStack, currentInnerStack, innerRef, SlotStateChange.REPLACEMENT);

                recursiveStackChange(slotReference, AccessoryNestUtils.getData(lastInnerStack), AccessoryNestUtils.getData(currentInnerStack));
            }
        }
    }

    public static void getTooltipData(@Nullable LivingEntity entity, ItemStack stack, List<Component> tooltip, TooltipFlag tooltipType) {
        var accessory = AccessoriesAPI.getOrDefaultAccessory(stack);

        if (accessory != null) {
            if(entity != null && AccessoriesCapability.get(entity) != null) addEntityBasedTooltipData(entity, accessory, stack, tooltip, tooltipType);

            accessory.getExtraTooltip(stack, tooltip, tooltipType);
        }
    }

    // TODO: Rewrite for better handling of various odd cases
    private static void addEntityBasedTooltipData(LivingEntity entity, Accessory accessory, ItemStack stack, List<Component> tooltip, TooltipFlag tooltipType) {
        // TODO: MAYBE DEPENDING ON ENTITY OR SOMETHING SHOW ALL VALID SLOTS BUT COLOR CODE THEM IF NOT VALID FOR ENTITY?
        // TODO: ADD BETTER HANDLING FOR POSSIBLE SLOTS THAT ARE EQUIPABLE IN BUT IS AT ZERO SIZE
        var validSlotTypes = new HashSet<>(AccessoriesAPI.getValidSlotTypes(entity, stack));

        var validUniqueSlots = validSlotTypes.stream()
                .filter(slotType -> UniqueSlotHandling.isUniqueSlot(slotType.name()))
                .collect(Collectors.toSet());

        if (validSlotTypes.isEmpty()) return;

        validSlotTypes.removeAll(validUniqueSlots);

        var sharedSlotTypes = SlotTypeLoader.getSlotTypes(entity.m_9236_()).values()
                .stream()
                .filter(slotType -> /*slotType.amount() > 0 &&*/ !UniqueSlotHandling.isUniqueSlot(slotType.name()))
                .collect(Collectors.toSet());

        var slotInfoComponent = Component.m_237113_("");

        var slotsComponent = Component.m_237113_("");
        boolean allSlots = false;


        if (validSlotTypes.containsAll(sharedSlotTypes)) {
            slotsComponent.m_7220_(Component.m_237115_(Accessories.translation("slot.any")));
            allSlots = true;
        } else {
            var entitySlotTypes = Set.copyOf(EntitySlotLoader.getEntitySlots(entity).values());

            var differenceSlotTypes = Sets.difference(entitySlotTypes, validSlotTypes);

            if(differenceSlotTypes.size() < validSlotTypes.size()) {
                slotsComponent.m_7220_(Component.m_237115_(Accessories.translation("slot.any")));
                slotsComponent.m_7220_(Component.m_237113_(" except ").m_130940_(ChatFormatting.GRAY));

                var slotTypesList = List.copyOf(differenceSlotTypes);

                for (int i = 0; i < slotTypesList.size(); i++) {
                    var type = slotTypesList.get(i);

                    slotsComponent.m_7220_(Component.m_237115_(type.translation()).m_130940_(ChatFormatting.RED));

                    if (i + 1 != slotTypesList.size()) {
                        slotsComponent.m_7220_(Component.m_237113_(", ").m_130940_(ChatFormatting.GRAY));
                    }
                }
            } else {
                var slotTypesList = List.copyOf(validSlotTypes);

                for (int i = 0; i < slotTypesList.size(); i++) {
                    var type = slotTypesList.get(i);

                    slotsComponent.m_7220_(Component.m_237115_(type.translation()));

                    if (i + 1 != slotTypesList.size()) {
                        slotsComponent.m_7220_(Component.m_237113_(", ").m_130940_(ChatFormatting.GRAY));
                    }
                }
            }
        }

        if(!validUniqueSlots.isEmpty()) {
            var uniqueSlotTypes = List.copyOf(validUniqueSlots);

            for (int i = 0; i < uniqueSlotTypes.size(); i++) {
                var type = uniqueSlotTypes.get(i);

                slotsComponent.m_7220_(Component.m_237115_(type.translation()));

                if (i + 1 != uniqueSlotTypes.size()) {
                    slotsComponent.m_7220_(Component.m_237113_(", ").m_130940_(ChatFormatting.GRAY));
                }
            }

            validSlotTypes.addAll(validUniqueSlots);
        }

        var slotTranslationKey = "slot.tooltip." + ((validSlotTypes.size() > 1 && !allSlots) ? "plural" : "singular");

        slotInfoComponent.m_7220_(
                Component.m_237115_(Accessories.translation(slotTranslationKey))
                        .m_130940_(ChatFormatting.GRAY)
                        .m_7220_(slotsComponent.m_130940_(ChatFormatting.BLUE))
        );

        tooltip.add(slotInfoComponent);

        var slotSpecificModifiers = new HashMap<SlotType, AccessoryAttributeBuilder>();
        AccessoryAttributeBuilder defaultModifiers = null;

        boolean allDuplicates = true;

        for (var slotType : validSlotTypes) {
            var reference = SlotReference.of(entity, slotType.name(), 0);

            var builder = AccessoriesAPI.getAttributeModifiers(stack, reference, true);

            slotSpecificModifiers.put(slotType, builder);

            if (defaultModifiers == null) {
                defaultModifiers = builder;
            } else if (allDuplicates) {
                // TODO: ! WARNING ! THIS MAY NOT WORK?
                allDuplicates = defaultModifiers.equals(builder);
            }
        }

        var slotTypeToTooltipInfo = new HashMap<SlotType, List<Component>>();

        if (allDuplicates) {
            if (!defaultModifiers.isEmpty()) {
                var attributeTooltip = new ArrayList<Component>();

                addAttributeTooltip(defaultModifiers.getAttributeModifiers(false), attributeTooltip);

                slotTypeToTooltipInfo.put(null, attributeTooltip);
            }
        } else {
            for (var slotModifiers : slotSpecificModifiers.entrySet()) {
                var slotType = slotModifiers.getKey();
                var modifiers = slotModifiers.getValue();

                if (modifiers.isEmpty()) continue;

                var attributeTooltip = new ArrayList<Component>();

                addAttributeTooltip(modifiers.getAttributeModifiers(false), attributeTooltip);

                slotTypeToTooltipInfo.put(slotType, attributeTooltip);
            }
        }

        var extraAttributeTooltips = new HashMap<SlotType, List<Component>>();
        List<Component> defaultExtraAttributeTooltip = null;

        boolean allDuplicatesExtras = true;

        for (var slotType : validSlotTypes) {
            var extraAttributeTooltip = new ArrayList<Component>();
            accessory.getAttributesTooltip(stack, slotType, extraAttributeTooltip, tooltipType);

            extraAttributeTooltips.put(slotType, extraAttributeTooltip);

            if (defaultExtraAttributeTooltip == null) {
                defaultExtraAttributeTooltip = extraAttributeTooltip;
            } else if (allDuplicatesExtras) {
                allDuplicatesExtras = extraAttributeTooltip.equals(defaultExtraAttributeTooltip);
            }
        }

        if (allDuplicatesExtras) {
            slotTypeToTooltipInfo.computeIfAbsent(null, s -> new ArrayList<>())
                    .addAll(defaultExtraAttributeTooltip);
        } else {
            extraAttributeTooltips.forEach((slotType, components) -> {
                slotTypeToTooltipInfo.computeIfAbsent(slotType, s -> new ArrayList<>())
                        .addAll(components);
            });
        }

        if(slotTypeToTooltipInfo.containsKey(null)) {
            var anyTooltipInfo = slotTypeToTooltipInfo.get(null);

            if (anyTooltipInfo.size() > 0) {
                tooltip.add(CommonComponents.f_237098_);

                tooltip.add(
                        Component.m_237115_(Accessories.translation("tooltip.attributes.any"))
                                .m_130940_(ChatFormatting.GRAY)
                );

                tooltip.addAll(anyTooltipInfo);
            }

            slotTypeToTooltipInfo.remove(null);
        }

        if(!slotTypeToTooltipInfo.isEmpty()) {
            for (var entry : slotTypeToTooltipInfo.entrySet()) {
                var tooltipData = entry.getValue();

                if(tooltipData.size() == 0) continue;

                tooltip.add(CommonComponents.f_237098_);

                tooltip.add(
                        Component.m_237110_(
                                Accessories.translation("tooltip.attributes.slot"),
                                Component.m_237115_(entry.getKey().translation()).m_130940_(ChatFormatting.BLUE)
                        ).m_130940_(ChatFormatting.GRAY)
                );

                tooltip.addAll(entry.getValue());
            }
        }
    }

    private static void addAttributeTooltip(Multimap<Attribute, AttributeModifier> multimap, List<Component> tooltip) {
        if (multimap.isEmpty()) return;

        for (Map.Entry<Attribute, AttributeModifier> entry : multimap.entries()) {
            AttributeModifier attributeModifier = entry.getValue();
            double d = attributeModifier.m_22218_();

            if (attributeModifier.m_22217_() == AttributeModifier.Operation.MULTIPLY_BASE
                    || attributeModifier.m_22217_() == AttributeModifier.Operation.MULTIPLY_TOTAL) {
                d *= 100.0;
            } else if (entry.getKey().equals(Attributes.f_22278_)) {
                d *= 10.0;
            }

            var key = entry.getKey();

            if (d > 0.0) {
                tooltip.add(
                        Component.m_237110_(
                                        "attribute.modifier.plus." + attributeModifier.m_22217_().m_22235_(),
                                        ItemStack.f_41584_.format(d),
                                        Component.m_237115_(key.m_22087_())
                                )
                                .m_130940_(ChatFormatting.BLUE)
                );
            } else if (d < 0.0) {
                d *= -1.0;
                tooltip.add(
                        Component.m_237110_(
                                        "attribute.modifier.take." + attributeModifier.m_22217_().m_22235_(),
                                        ItemStack.f_41584_.format(d),
                                        Component.m_237115_(key.m_22087_())
                                )
                                .m_130940_(ChatFormatting.RED)
                );
            }
        }
    }

    public static List<ItemStack> onDeath(LivingEntity entity, DamageSource source) {
        var capability = AccessoriesCapability.get(entity);

        var droppedStacks = new ArrayList<ItemStack>();

        var gamerules = entity.m_9236_().m_46469_();

        var keepInv = gamerules.m_46170_(GameRules.f_46133_).m_46223_() || gamerules.m_46170_(Accessories.RULE_KEEP_ACCESSORY_INVENTORY).m_46223_();

        if (capability != null) {
            for (var containerEntry : capability.getContainers().entrySet()) {
                var slotType = containerEntry.getValue().slotType();

                var slotDropRule = slotType != null ? slotType.dropRule() : DropRule.DEFAULT;

                var container = containerEntry.getValue();

                var stacks = container.getAccessories();
                var cosmeticStacks = container.getCosmeticAccessories();

                for (int i = 0; i < container.getSize(); i++) {
                    var reference = SlotReference.of(entity, container.getSlotName(), i);

                    var stack = dropStack(slotDropRule, entity, stacks, reference, source, keepInv);
                    if (stack != null) droppedStacks.add(stack);

                    var cosmeticStack = dropStack(slotDropRule, entity, cosmeticStacks, reference, source, keepInv);
                    if (cosmeticStack != null) droppedStacks.add(cosmeticStack);
                }
            }

            var result = OnDeathCallback.EVENT.invoker().shouldDrop(TriState.DEFAULT, entity, capability, source, droppedStacks);

            if (!result.orElse(true)) return List.of();
        }

        return droppedStacks;
    }

    @Nullable
    private static ItemStack dropStack(DropRule dropRule, LivingEntity entity, ExpandedSimpleContainer container, SlotReference reference, DamageSource source, boolean keepInvEnabled) {
        var stack = container.m_8020_(reference.slot());
        var accessory = AccessoriesAPI.getAccessory(stack);

        if (accessory != null && dropRule == DropRule.DEFAULT) {
            dropRule = accessory.getDropRule(stack, reference, source);
        }

        if (accessory instanceof AccessoryNest holdable) {
            var dropRuleToStacks = holdable.getDropRules(stack, reference, source);

            for (int i = 0; i < dropRuleToStacks.size(); i++) {
                var rulePair = dropRuleToStacks.get(i);

                var innerStack = rulePair.right();

                var result = OnDropCallback.getAlternativeRule(rulePair.left(), innerStack, reference, source);

                var breakInnerStack = (result == DropRule.DEFAULT && EnchantmentHelper.m_44924_(innerStack))
                        || (result == DropRule.DESTROY);

                if (breakInnerStack) {
                    holdable.setInnerStack(stack, i, ItemStack.f_41583_);
                    // TODO: Do we call break here for the accessory?

                    container.m_6836_(reference.slot(), stack);
                }
            }
        }

        var result = OnDropCallback.getAlternativeRule(dropRule, stack, reference, source);

        boolean dropStack = true;

        if (result == DropRule.DESTROY) {
            container.m_6836_(reference.slot(), ItemStack.f_41583_);
            dropStack = false;
            // TODO: Do we call break here for the accessory?
        } else if (result == DropRule.KEEP) {
            dropStack = false;
        } else if (result == DropRule.DEFAULT) {
            if (keepInvEnabled) {
                dropStack = false;
            } else if (EnchantmentHelper.m_44924_(stack)) {
                container.m_6836_(reference.slot(), ItemStack.f_41583_);
                dropStack = false;
                // TODO: Do we call break here for the accessory?
            }
        }

        container.setPreviousItem(reference.slot(), ItemStack.f_41583_);

        if (!dropStack) return null;

        container.m_6836_(reference.slot(), ItemStack.f_41583_);

        return stack;
    }

    public static InteractionResultHolder<ItemStack> attemptEquipFromUse(Player player, InteractionHand hand) {
        var stack = player.m_21120_(hand);

        var capability = AccessoriesCapability.get(player);

        if(capability != null && !player.m_5833_() && !stack.m_41619_()) {
            var equipControl = capability.getHolder().equipControl();

            var shouldAttemptEquip = false;

            if(equipControl == PlayerEquipControl.MUST_CROUCH && player.m_6144_()) {
                shouldAttemptEquip = true;
            } else if(equipControl == PlayerEquipControl.MUST_NOT_CROUCH && !player.m_6144_()) {
                shouldAttemptEquip = true;
            }

            if (shouldAttemptEquip) {
                var accessory = AccessoriesAPI.getOrDefaultAccessory(stack);

                var equipReference = capability.canEquipAccessory(stack, true);

                if (equipReference != null && accessory.canEquipFromUse(stack)) {
                    accessory.onEquipFromUse(stack, equipReference.left());

                    var newHandStack = stack.m_41777_();

                    var possibleSwappedStack = equipReference.second().equipStack(newHandStack);

                    if(possibleSwappedStack.isPresent()) {
                        var swappedStack = possibleSwappedStack.get();

                        if (newHandStack.m_41619_()) {
                            newHandStack = swappedStack;
                        } else if(ItemStack.m_150942_(newHandStack, swappedStack) && (newHandStack.m_41613_() + swappedStack.m_41613_()) <= newHandStack.m_41741_()) {
                            newHandStack.m_41769_(swappedStack.m_41613_());
                        } else {
                            player.m_36356_(swappedStack);
                        }
                    }

                    return InteractionResultHolder.m_19090_(newHandStack);
                }
            }
        }

        return InteractionResultHolder.m_19098_(stack);
    }

    public static final TagKey<EntityType<?>> EQUIPMENT_MANAGEABLE = TagKey.m_203882_(Registries.f_256939_, Accessories.of("equipment_manageable"));

    public static InteractionResult attemptEquipOnEntity(Player player, InteractionHand hand, Entity entity) {
        var stack = player.m_21120_(hand);

        if(!(entity instanceof LivingEntity targetEntity) || !entity.m_6095_().m_204039_(EQUIPMENT_MANAGEABLE)) return InteractionResult.PASS;

        var targetCapability = AccessoriesCapability.get(targetEntity);

        var canModify = AllowEntityModificationCallback.EVENT.invoker().allowModifications(targetEntity, player, null).orElse(false);

        if (canModify && targetCapability != null && !player.m_5833_()) {
            if (player.m_6144_()) {
                var accessory = AccessoriesAPI.getOrDefaultAccessory(stack);

                var equipReference = targetCapability.canEquipAccessory(stack, true);

                if (equipReference != null && accessory.canEquipFromUse(stack)) {
                    if(!stack.m_41619_()) accessory.onEquipFromUse(stack, equipReference.left());

                    var newHandStack = stack.m_41777_();

                    var possibleSwappedStack = equipReference.second().equipStack(newHandStack);

                    if(possibleSwappedStack.isPresent()) {
                        var swappedStack = possibleSwappedStack.get();

                        if (newHandStack.m_41619_()) {
                            newHandStack = swappedStack;
                        } else if(ItemStack.m_150942_(newHandStack, swappedStack) && (newHandStack.m_41613_() + swappedStack.m_41613_()) <= newHandStack.m_41741_()) {
                            newHandStack.m_41769_(swappedStack.m_41613_());
                        } else {
                            player.m_36356_(swappedStack);
                        }
                    }

                    player.m_21008_(hand, newHandStack);

                    return InteractionResult.SUCCESS;
                }
            }
        }

        return InteractionResult.PASS;
    }

//    public static void setupItems(AddDataComponentCallback callback) {
//        AccessoriesAPI.getAllAccessories().forEach((item, accessory) -> {
//            var builder = AccessoryItemAttributeModifiers.builder();
//
//            accessory.getStaticModifiers(item, builder);
//
//            if(!builder.isEmpty()) {
//                callback.addTo(item, AccessoriesDataComponents.ATTRIBUTES, builder.build());
//            }
//        });
//    }

//    public interface AddDataComponentCallback {
//        <T> void addTo(Item item, DataComponentType<T> componentType, T component);
//    }
}
