package io.wispforest.accessories.api;

import com.google.common.collect.Multimap;
import io.wispforest.accessories.api.caching.ItemStackBasedPredicate;
import io.wispforest.accessories.api.caching.ItemStackPredicate;
import io.wispforest.accessories.api.core.Accessory;
import io.wispforest.accessories.api.equip.EquipAction;
import io.wispforest.accessories.api.equip.EquipCheck;
import io.wispforest.accessories.api.equip.EquipmentChecking;
import io.wispforest.accessories.api.slot.*;
import io.wispforest.accessories.data.SlotTypeLoader;
import io.wispforest.accessories.impl.caching.AccessoriesHolderLookupCache;
import io.wispforest.accessories.impl.core.AccessoriesHolderImpl;
import io.wispforest.accessories.pond.AccessoriesAPIAccess;
import it.unimi.dsi.fastutil.Pair;
import org.apache.commons.lang3.function.TriFunction;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Predicate;
import net.minecraft.class_1263;
import net.minecraft.class_1309;
import net.minecraft.class_1322;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;

public interface AccessoriesCapability extends AccessoriesStorageLookup {

    /**
     * @return The {@link AccessoriesCapability} Bound to the given living entity if present
     */
    @Nullable
    static AccessoriesCapability get(@NotNull class_1309 livingEntity){
        return ((AccessoriesAPIAccess) livingEntity).accessoriesCapability();
    }

    static Optional<AccessoriesCapability> getOptionally(@NotNull class_1309 livingEntity){
        return Optional.ofNullable(get(livingEntity));
    }

    static Collection<SlotType> getUsedSlotsFor(class_1657 player) {
        return getUsedSlotsFor(player, player.method_31548());
    }

    static Collection<SlotType> getUsedSlotsFor(class_1309 entity, class_1263 container) {
        var capability = entity.accessoriesCapability();

        return (capability != null) ? capability.getUsedSlotsFor(container) : Set.of();
    }

    //--

    /**
     * @return The entity bound to the given {@link AccessoriesCapability} instance
     */
    class_1309 entity();

    //--

    /**
     * Method used to clear all containers bound to the given {@link class_1309}
     */
    void reset(boolean loadedFromTag);

    /**
     * @return A Map containing all the {@link AccessoriesContainer}s with their {@link SlotType#name()} as the key
     */
    Map<String, AccessoriesContainer> getContainers();

    /**
     * @return a given {@link AccessoriesContainer} if found on the given {@link class_1309} tied to the Capability or null if not
     */
    @Nullable
    default AccessoriesContainer getContainer(SlotType slotType){
        return getContainers().get(slotType.name());
    }

    @Nullable
    default AccessoriesContainer getContainer(SlotTypeReference reference){
        return getContainers().get(reference.slotName());
    }

    default Collection<SlotType> getUsedSlotsFor(class_1263 container) {
        var entity = this.entity();
        var slots = new HashSet<SlotType>();

        for (int i = 0; i < container.method_5439(); i++) {
            var stack = container.method_5438(i);

            if (stack.method_7960()) continue;

            slots.addAll(SlotPredicateRegistry.getValidSlotTypes(entity, stack));
        }

        for (var ref : this.getAllEquipped()) {
            slots.addAll(SlotPredicateRegistry.getValidSlotTypes(entity, ref.stack()));
        }

        slots.addAll(SlotTypeLoader.getUsedSlotsByRegistryItem(entity));

        return slots;
    }

    void updateContainers();

    //--

    /**
     * Attempts to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference to where it was equipped. The given passed stack <b>will</b> be adjusted passed on
     * the amount of room that can be found within the found container.
     *
     * @param stack The given stack attempting to be equipped
     */
    @Nullable
    default SlotReference attemptToEquipAccessory(class_1799 stack) {
        var result = attemptToEquipAccessory(stack, false);

        return result != null ? result.first() : null;
    }

    /**
     * Attempts to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference to where it was equipped and an {@link Optional} of the previous stack if swapped for
     * the passed stack. The given passed stack <b>will</b> be adjusted passed on the amount of room that
     * can be found within the found container.
     *
     * @param stack The given stack attempting to be equipped
     */
    @Nullable
    default Pair<SlotReference, Optional<class_1799>> attemptToEquipAccessory(class_1799 stack, boolean allowSwapping) {
        var result = canEquipAccessory(stack, allowSwapping, (slotStack, slotReference) -> true);

        return result != null ? Pair.of(result.first(), result.second().equipStack(stack)) : null;
    }

    default Pair<SlotReference, EquipAction> canEquipAccessory(class_1799 stack, boolean allowSwapping) {
        return canEquipAccessory(stack, allowSwapping, (slotStack, slotReference) -> true);
    }

    /**
     * Attempts to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference to where it can be equipped and a function to attempt equipping of the item which
     * may return an {@link Optional} of the previous stack if allowing for swapping.
     * <p>
     * Info: The passed stack will not be mutated in any way! Such only occurs on call of the possible
     * returned function.
     *
     * @param stack The given stack attempting to be equipped
     */
    @Nullable
    Pair<SlotReference, EquipAction> canEquipAccessory(class_1799 stack, boolean allowSwapping, EquipCheck extraCheck);

    /**
     * @return The first {@link class_1799} formatted within {@link SlotEntryReference} that matches the given {@link class_1792}.
     */
    @Nullable
    default SlotEntryReference getFirstEquipped(class_1792 item){
        return getFirstEquipped(item, EquipmentChecking.ACCESSORIES_ONLY);
    }

    /**
     * @return The first {@link class_1799} formatted within {@link SlotEntryReference} that matches the given {@link class_1792}
     * with the given {@link EquipmentChecking} useful for detecting Cosmetic overrides for rendering.
     */
    @Nullable
    default SlotEntryReference getFirstEquipped(class_1792 item, EquipmentChecking check){
        return getFirstEquipped(ItemStackBasedPredicate.ofItem(item), check);
    }

    /**
     * @return The first {@link class_1799} formatted within {@link SlotEntryReference} that matches the given {@link Predicate}.
     */
    @Nullable
    default SlotEntryReference getFirstEquipped(Predicate<class_1799> predicate) {
        return getFirstEquipped(predicate, EquipmentChecking.ACCESSORIES_ONLY);
    }

    /**
     * @return The first {@link class_1799} formatted within {@link SlotEntryReference} that matches the given {@link Predicate}
     * with the given {@link EquipmentChecking} useful for detecting Cosmetic overrides for rendering.
     */
    @Nullable
    default SlotEntryReference getFirstEquipped(Predicate<class_1799> predicate, EquipmentChecking check) {
        return getFirstEquipped(ItemStackBasedPredicate.ofPredicate(predicate), check);
    }

    @Nullable
    default SlotEntryReference getFirstEquipped(ItemStackBasedPredicate predicate, EquipmentChecking check) {
        var cache = AccessoriesHolderImpl.getHolder(this).getLookupCache();

        if (cache != null && !(predicate instanceof ItemStackPredicate)) return cache.firstEquipped(predicate, check);

        return AccessoriesStorageLookupUtils.getFirstEquipped(this.getContainers(), (path, stack) -> new SlotEntryReference(this.entity(), path, stack),
                predicate,
                check
        );
    }

    /**
     * @return A list of all {@link class_1799}'s formatted within {@link SlotEntryReference} matching the given {@link class_1792}
     */
    default List<SlotEntryReference> getEquipped(class_1792 item){
        return getEquipped(ItemStackBasedPredicate.ofItem(item));
    }

    /**
     * @return A list of all {@link SlotEntryReference}'s formatted within {@link SlotEntryReference} matching the passed predicate
     */
    default List<SlotEntryReference> getEquipped(Predicate<class_1799> predicate){
        return getEquipped(ItemStackBasedPredicate.ofPredicate(predicate));
    }

    default List<SlotEntryReference> getEquipped(ItemStackBasedPredicate predicate) {
        return getAllEquipped().stream().filter(reference -> predicate.test(reference.stack())).toList();
    }

    @Override
    default List<SlotEntryReference> getAllEquipped() {
        var cache = AccessoriesHolderImpl.getHolder(this).getLookupCache();

        if (cache != null) return cache.getAllEquipped();

        return AccessoriesStorageLookupUtils.getAllEquipped(this.getContainers(), (path, stack) -> new SlotEntryReference(this.entity(), path, stack));
    }

    //--

    /**
     * Add map containing slot attributes to the given capability based on the keys used referencing specific slots
     * with being lost on reload
     * @param modifiers Slot Attribute Modifiers
     */
    void addTransientSlotModifiers(Multimap<String, class_1322> modifiers);

    /**
     * Add slot attribute attributes to the given capability based on the keys used referencing specific slots
     * with being persistent on a reload
     * @param modifiers Slot Attribute Modifiers
     */
    void addPersistentSlotModifiers(Multimap<String, class_1322> modifiers);

    /**
     * Add slot attribute attributes to the given capability based on the keys
     * @param modifiers Slot Attribute Modifiers
     */
    void removeSlotModifiers(Multimap<String, class_1322> modifiers);

    /**
     * Get all modifiers from the given containers bound to the capability
     */
    Multimap<String, class_1322> getSlotModifiers();

    /**
     * Remove all modifiers from the given containers bound to the capability
     */
    void clearSlotModifiers();

    /**
     * Remove all cached modifiers from the given containers bound to the capability
     */
    void clearCachedSlotModifiers();

    //-- Deprecated meaning to be removed within the future

    /**
     * Used to attempt to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference and list within a pair. The given list may contain the overflow that could not fit based
     * on the containers max stack size.
     * <p>
     * <b>WARNING: THE GIVEN STACK PASSED WILL NOT BE MUTATED AT ALL!</b>
     *
     * @param stack          The given stack attempting to be equipped
     */
    @Deprecated
    @Nullable
    default Pair<SlotReference, List<class_1799>> equipAccessory(class_1799 stack){
        return equipAccessory(stack, false);
    }

    /**
     * Used to attempt to equip a given stack within any available {@link AccessoriesContainer} returning a
     * reference and list within a pair. The given list may contain the overflow that could not fit based
     * on the containers max stack size and the old stack found if swapping was allowed.
     * <p>
     * <b>WARNING: THE GIVEN STACK PASSED WILL NOT BE MUTATED AT ALL!</b>
     *
     * @param stack          The given stack attempting to be equipped
     * @param allowSwapping  If the given call can attempt to swap accessories
     */
    @Deprecated
    default Pair<SlotReference, List<class_1799>> equipAccessory(class_1799 stack, boolean allowSwapping) {
        var stackCopy = stack.method_7972();

        var result = attemptToEquipAccessory(stackCopy, allowSwapping);

        if(result == null) return null;

        var returnStacks = new ArrayList<class_1799>();

        if(!stackCopy.method_7960()) returnStacks.add(stackCopy);

        result.second().ifPresent(returnStacks::add);

        return Pair.of(result.first(), returnStacks);
    }

    @Nullable
    @Deprecated
    default Pair<SlotReference, List<class_1799>> equipAccessory(class_1799 stack, boolean allowSwapping, TriFunction<Accessory, class_1799, SlotReference, Boolean> additionalCheck) {
        return equipAccessory(stack, allowSwapping);
    }

    /**
     * @deprecated Use {@link #isAnotherEquipped(class_1799, SlotReference, class_1792)}
     */
    @Deprecated(forRemoval = true)
    default boolean isAnotherEquipped(SlotReference slotReference, class_1792 item) {
        return isAnotherEquipped(slotReference.getStack() /* <- DO NOT DO THIS! */, slotReference, item);
    }

    /**
     * @deprecated Use {@link #isAnotherEquipped(class_1799, SlotReference, Predicate)}
     */
    @Deprecated(forRemoval = true)
    default boolean isAnotherEquipped(SlotReference slotReference, Predicate<class_1799> predicate) {
        return isAnotherEquipped(slotReference.getStack() /* <- DO NOT DO THIS! */, slotReference, predicate);
    }
}
