package io.wispforest.accessories.api.attributes;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Multimap;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.AccessoriesLoaderInternals;
import io.wispforest.accessories.api.attributes.AttributeModificationData.AllowedType;
import io.wispforest.accessories.api.slot.SlotPath;
import io.wispforest.accessories.api.slot.SlotReference;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

import java.util.*;
import java.util.Map.Entry;
import net.minecraft.class_1320;
import net.minecraft.class_1322;
import net.minecraft.class_2960;
import net.minecraft.class_6880;

import static io.wispforest.accessories.api.attributes.AttributeModificationData.AllowedType;

/**
 * Builder used to collect the attribute modifications from a given Accessory with the ability
 * to specified if an Attribute modification can be stacked or is exclusive to one version
 */
public final class AccessoryAttributeBuilder {

    private static final Logger LOGGER = LogUtils.getLogger();

    private final Map<class_6880<class_1320>, Map<class_2960, AttributeModificationData>> exclusiveAttributes;
    private final Multimap<class_6880<class_1320>, AttributeModificationData> stackedAttributes;

    private final SlotPath slotPath;

    @ApiStatus.Internal
    public AccessoryAttributeBuilder(SlotPath slotPath, @Nullable AccessoryAttributeBuilder parentBuilder) {
        this.slotPath = slotPath;

        if (parentBuilder != null) {
            this.exclusiveAttributes = parentBuilder.exclusiveAttributes;
            this.stackedAttributes = parentBuilder.stackedAttributes;
        } else {
            this.exclusiveAttributes = new HashMap<>();
            this.stackedAttributes = LinkedHashMultimap.create();
        }
    }

    @ApiStatus.Internal
    public AccessoryAttributeBuilder(SlotPath slotPath) {
        this(slotPath, null);
    }

    @ApiStatus.Internal
    public AccessoryAttributeBuilder(String slotName, int slot) {
        this(SlotPath.of(slotName, slot));
    }

    @ApiStatus.Internal
    public AccessoryAttributeBuilder() {
        this(SlotPath.of("", 0));
    }

    //--

    /**
     * Adds a given attribute modifier as an exclusive modifier meaning that only one instance should ever exist
     */
    public AccessoryAttributeBuilder addExclusive(class_6880<class_1320> attribute, class_2960 location, double amount, class_1322.class_1323 operation) {
        return this.addExclusive(attribute, location, amount, operation, false);
    }

    /**
     * Adds a given attribute modifier as a stackable modifier meaning variants based on slot position is allowed. This is done by post process
     * step of appending slot information when adding to the living entity
     */
    public AccessoryAttributeBuilder addStackable(class_6880<class_1320> attribute, class_2960 location, double amount, class_1322.class_1323 operation) {
        return this.addStackable(attribute, location, amount, operation, false);
    }

    /**
     * Adds a given attribute modifier as an exclusive modifier meaning that only one instance should ever exist
     */
    public AccessoryAttributeBuilder addExclusive(class_6880<class_1320> attribute, class_2960 location, double amount, class_1322.class_1323 operation, boolean usedInSlotValidation) {
        return this.addExclusive(attribute, new class_1322(location, amount, operation), usedInSlotValidation);
    }

    /**
     * Adds a given attribute modifier as a stackable modifier meaning variants based on slot position is allowed. This is done by post process
     * step of appending slot information when adding to the living entity
     */
    public AccessoryAttributeBuilder addStackable(class_6880<class_1320> attribute, class_2960 location, double amount, class_1322.class_1323 operation, boolean usedInSlotValidation) {
        return this.addStackable(attribute, new class_1322(location, amount, operation), usedInSlotValidation);
    }

    public AccessoryAttributeBuilder addExclusive(class_6880<class_1320> attribute, class_1322 modifier) {
        return this.addExclusive(attribute, modifier, false);
    }

    public AccessoryAttributeBuilder addStackable(class_6880<class_1320> attribute, class_1322 modifier) {
        return this.addStackable(attribute, modifier, false);
    }

    private final Set<class_2960> previouslyWarnedLocations = new HashSet<>();

    /**
     * Adds a given attribute modifier as an exclusive modifier meaning that only one instance should ever exist
     */
    public AccessoryAttributeBuilder addExclusive(class_6880<class_1320> attribute, class_1322 modifier, boolean usedInSlotValidation) {
        var id = modifier.comp_2447();

        var innerMap = this.exclusiveAttributes.computeIfAbsent(attribute, attributeHolder -> new HashMap<>());

        if(AccessoriesLoaderInternals.isDevelopmentEnv() && innerMap.containsKey(id) && !this.previouslyWarnedLocations.contains(id)) {
            LOGGER.warn("A given Modifier was found to have a duplicate location but was added as exclusive, was such on purpose as such will not stack with the other: {}", id);

            this.previouslyWarnedLocations.add(id);
        }

        innerMap.putIfAbsent(id, new AttributeModificationData(attribute, modifier, usedInSlotValidation));

        return this;
    }

    /**
     * Adds a given attribute modifier as a stackable modifier meaning variants based on slot position is allowed. This is done by post process
     * step of appending slot information when adding to the living entity
     */
    public AccessoryAttributeBuilder addStackable(class_6880<class_1320> attribute, class_1322 modifier, boolean usedInSlotValidation) {
        this.stackedAttributes.put(attribute, new AttributeModificationData(this.slotPath.createString(), attribute, modifier, usedInSlotValidation));

        return this;
    }

    //--

    @Nullable
    public AttributeModificationData getExclusive(class_6880<class_1320> attribute, class_2960 location) {
        var innerMap = this.exclusiveAttributes.get(attribute);

        if(innerMap == null) return null;

        return innerMap.get(location);
    }

    public Collection<AttributeModificationData> getStacks(class_6880<class_1320> attribute, class_2960 location) {
        return this.stackedAttributes.get(attribute).stream().filter(data -> data.modifier().comp_2447().equals(location)).toList();
    }

    @Nullable
    public AttributeModificationData removeExclusive(class_6880<class_1320> attribute, class_2960 location) {
        var innerMap = this.exclusiveAttributes.get(attribute);

        if(innerMap == null) return null;

        return innerMap.remove(location);
    }

    public Collection<AttributeModificationData> removeStacks(class_6880<class_1320> attribute, class_2960 location) {
        Set<AttributeModificationData> removedData = new HashSet<>();

        for (var data : List.copyOf(this.stackedAttributes.get(attribute))) {
            if(!data.modifier().comp_2447().equals(location)) continue;

            removedData.add(data);

            this.stackedAttributes.remove(attribute, data);
        }

        return removedData;
    }

    //--

    public Multimap<String, class_1322> getSlotModifiers() {
        return getSlotModifiers(false);
    }

    public Multimap<String, class_1322> getSlotModifiers(boolean usedWithinSlotPredicate) {
        var map = LinkedHashMultimap.<String, class_1322>create();

        this.exclusiveAttributes.forEach((attribute, innerMap) -> {
            innerMap.forEach((location, uniqueInstance) -> {
                if (uniqueInstance.isValid(AllowedType.SLOT, usedWithinSlotPredicate)) {
                    map.put(((SlotAttribute) uniqueInstance.attribute().comp_349()).slotName(), uniqueInstance.modifier());
                }
            });
        });

        this.stackedAttributes.forEach((location, stackedInstance) -> {
            if (stackedInstance.isValid(AllowedType.SLOT, usedWithinSlotPredicate)) {
                map.put(((SlotAttribute) stackedInstance.attribute().comp_349()).slotName(), stackedInstance.modifier());
            }
        });

        return map;
    }

    public Multimap<class_6880<class_1320>, class_1322> getAttributeModifiers(boolean removeSlotAttributes) {
        return getAttributeModifiers(removeSlotAttributes, false);
    }

    public Multimap<class_6880<class_1320>, class_1322> getAttributeModifiers(boolean removeSlotAttributes, boolean usedWithinSlotPredicate) {
        var map = LinkedHashMultimap.<class_6880<class_1320>, class_1322>create();

        this.exclusiveAttributes.forEach((attribute, innerMap) -> {
            innerMap.forEach((location, uniqueInstance) -> {
                if (uniqueInstance.isValid(removeSlotAttributes, usedWithinSlotPredicate)) {
                    map.put(uniqueInstance.attribute(), uniqueInstance.modifier());
                }
            });
        });

        this.stackedAttributes.forEach((location, stackedInstance) -> {
            if (stackedInstance.isValid(removeSlotAttributes, usedWithinSlotPredicate)) {
                map.put(stackedInstance.attribute(), stackedInstance.modifier());
            }
        });

        return map;
    }

    public boolean isEmpty() {
        return this.exclusiveAttributes.isEmpty() && this.stackedAttributes.isEmpty();
    }

    public Map<class_6880<class_1320>, Map<class_2960, AttributeModificationData>> exclusiveAttributes() {
        return ImmutableMap.copyOf(this.exclusiveAttributes);
    }

    public Multimap<class_6880<class_1320>, AttributeModificationData> stackedAttributes() {
        return ImmutableMultimap.copyOf(this.stackedAttributes);
    }

    public AccessoryAttributeBuilder addFrom(AccessoryAttributeBuilder builder) {
        builder.exclusiveAttributes.forEach(this.exclusiveAttributes::putIfAbsent);
        this.stackedAttributes.putAll(builder.stackedAttributes);

        return this;
    }

    @Override
    public boolean equals(Object obj) {
        if(!(obj instanceof AccessoryAttributeBuilder otherBuilder)) return false;
        if(!areMapsEqual(this.stackedAttributes, otherBuilder.stackedAttributes)) return false;

        return this.exclusiveAttributes.equals(otherBuilder.exclusiveAttributes);
    }

    private static <K, V> boolean areMapsEqual(Multimap<K, V> multimap1, Multimap<K, V> multimap2) {
        for (var entry : multimap1.asMap().entrySet()) {
            if(!entry.getValue().equals(multimap2.get(entry.getKey()))) return false;
        }

        return true;
    }

    //--

    // slotPath          = {slot_name}/{slot_index}[{nested_layer_info}]
    // nested_layer_info = /nest_{layer_index}_{slot_index}
    @Deprecated
    public static String createSlotPath(SlotReference ref) {
        return ref.createString();
    }

    @Deprecated
    public static String createSlotPath(String slotname, int slot) {
        return SlotPath.createBaseSlotPath(slotname, slot);
    }
}
