package io.wispforest.accessories.api.core;

import io.wispforest.accessories.api.SoundEventData;
import io.wispforest.accessories.api.attributes.AccessoryAttributeBuilder;
import io.wispforest.accessories.api.components.AccessoriesDataComponents;
import io.wispforest.accessories.api.components.AccessoryItemAttributeModifiers;
import io.wispforest.accessories.api.components.AccessoryMobEffectsComponent;
import io.wispforest.accessories.api.components.AccessoryStackSettings;
import io.wispforest.accessories.api.events.DropRule;
import io.wispforest.accessories.api.slot.SlotReference;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.impl.AccessoryAttributeLogic;
import io.wispforest.accessories.impl.event.AccessoriesEventHandler;
import io.wispforest.accessories.mixin.LivingEntityAccessor;
import io.wispforest.accessories.networking.client.AccessoryBreak;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.MustBeInvokedByOverriders;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import net.minecraft.class_1282;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_1890;
import net.minecraft.class_2561;
import net.minecraft.class_3417;
import net.minecraft.class_5244;
import net.minecraft.class_6880;
import net.minecraft.class_9334;
import net.minecraft.class_9701;

/**
 * Main interface for implementing accessory functionality within the main API.
 */
public interface Accessory {

    /**
     * Called on every tick of the wearing {@link class_1309} on both client and server.
     * <br/><br/>
     * Contains the code for handling {@link AccessoryMobEffectsComponent} which can be disabled
     * by not calling override
     *
     * @param stack the stack being ticked
     * @param reference the slot the accessory is in
     */
    default void tick(class_1799 stack, SlotReference reference){}

    /**
     * Called when the accessory is equipped
     *
     * @param stack the stack being equipped
     * @param reference the slot the accessory is in
     */
    default void onEquip(class_1799 stack, SlotReference reference){}

    /**
     * Called when the accessory is unequipped
     * <br><br>
     * Note: Due to how stack transfer can occur from more than just {@link net.minecraft.class_1263} interface,
     * the stack maybe a defensive copy, or only one of the stacks found to be unequipped meaning issues could arise in
     * cases that you need to remove data from the stack.
     *
     * @param stack the stack being unequipped
     * @param reference the slot the accessory is in
     */
    default void onUnequip(class_1799 stack, SlotReference reference){}

    /**
     * @param stack the stack to be equipped
     * @param reference the slot the accessory is in
     * @return whether the given stack can be equipped
     */
    default boolean canEquip(class_1799 stack, SlotReference reference){
        return true;
    }

    /**
     * @param stack the stack to be unequipped
     * @param reference the slot the accessory is in
     * @return whether the given stack can be unequipped
     */
    @MustBeInvokedByOverriders
    default boolean canUnequip(class_1799 stack, SlotReference reference){
        if(class_1890.method_60142(stack, class_9701.field_51656)) {
            return reference.entity() instanceof class_1657 player && player.method_68878();
        }

        return true;
    }

    /**
     * Helper method used to fill the passed {@link AccessoryAttributeBuilder} for every call from
     * {@link AccessoryAttributeLogic#getAttributeModifiers}
     *
     * @param stack     The Stack attempting to be unequipped
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     * @param builder   The builder to which attributes are to be added
     */
    default void getDynamicModifiers(class_1799 stack, SlotReference reference, AccessoryAttributeBuilder builder){
        getModifiers(stack, reference, builder);
    }

    /**
     * Helper method used to fill the passed {@link AccessoryItemAttributeModifiers.Builder} when called to modify
     * the given default {@link AccessoriesDataComponents#ATTRIBUTES} right before Registry freeze occurs. This is
     * useful for attributes that are not meant to be changed and remain mostly static.
     *
     * @param item      The item to which the given attributes will be defaulted for
     * @param builder   The builder to which attributes are to be added
     */
    default void getStaticModifiers(class_1792 item, AccessoryItemAttributeModifiers.Builder builder){}

    /**
     * Returns the following drop rule for the given Item
     *
     * @param stack     The Stack being prepared for dropping
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     * @param source    The specific {@link class_1282} that lead to the drop rule evaluation
     */
    @MustBeInvokedByOverriders
    default DropRule getDropRule(class_1799 stack, SlotReference reference, class_1282 source){
        return stack.method_58695(AccessoriesDataComponents.STACK_SETTINGS, AccessoryStackSettings.DEFAULT)
                .dropRule();
    }

    //--

    /**
     * Method called when equipping the given accessory from hotbar by right-clicking
     *
     * @param stack The Stack being prepared for dropping
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     */
    @MustBeInvokedByOverriders
    default void onEquipFromUse(class_1799 stack, SlotReference reference){
        var sound = getEquipSound(stack, reference);

        if(sound == null) return;

        reference.entity().method_5783(sound.event().comp_349(), sound.volume(), sound.pitch());
    }

    /**
     * Returns the equipping sound from use for a given stack
     *
     * @param stack The Stack being prepared for dropping
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     */
    @Nullable
    default SoundEventData getEquipSound(class_1799 stack, SlotReference reference){
        var equipSound = stack.method_57826(class_9334.field_54196) ? stack.method_58694(class_9334.field_54196).comp_3175() : class_3417.field_14883;

        return new SoundEventData(equipSound, 1.0f, 1.0f);
    }

    @Deprecated(forRemoval = true)
    default boolean canEquipFromUse(class_1799 stack){
        try {
            return canEquipFromUse(stack, null);
        } catch (NullPointerException e) {
            return false;
        }
    }

    /**
     * Returns whether the given stack can be equipped from use
     *
     * @param stack The Stack attempted to be equipped
     */
    @MustBeInvokedByOverriders
    default boolean canEquipFromUse(class_1799 stack, SlotReference reference){
        if (stack.method_57826(AccessoriesDataComponents.STACK_SETTINGS)) {
            return stack.method_58694(AccessoriesDataComponents.STACK_SETTINGS).canEquipFromUse();
        }

        if (stack.method_57826(class_9334.field_54196)) {
            return stack.method_58694(class_9334.field_54196).comp_3213();
        }

        return true;
    }

    default boolean canEquipFromDispenser(class_1799 stack, SlotReference reference){
        if (stack.method_57826(AccessoriesDataComponents.STACK_SETTINGS)) {
            return stack.method_58694(AccessoriesDataComponents.STACK_SETTINGS).canEquipFromDispenser();
        }

        if (stack.method_57826(class_9334.field_54196)) {
            return stack.method_58694(class_9334.field_54196).comp_3178();
        }

        return true;
    }

    /**
     * Method used to render client based particles when {@link SlotReference#breakStack()} is
     * called on the server and the {@link AccessoryBreak} packet is received
     *
     * @param stack The Stack that broke
     * @param reference The reference to the targeted {@link class_1309}, slot and index
     */
    default void onBreak(class_1799 stack, SlotReference reference) {
        ((LivingEntityAccessor) reference.entity()).accessors$breakItem(stack);
    }

    /**
     * @return Return the max stack amount allowed when equipping a given stack into an accessories inventory
     */
    default int maxStackSize(class_1799 stack){
        var data = stack.method_58695(AccessoriesDataComponents.STACK_SETTINGS, AccessoryStackSettings.DEFAULT);

        if(data.useStackSize()) return stack.method_7914();

        return Math.min(Math.max(data.sizeOverride(), 1), stack.method_7914());
    }

    //--

    /**
     * Method used to add tooltip info for attribute like data based on a given slot type
     *
     * @param stack The Stack being referenced
     * @param type The SlotType being referenced
     * @param tooltips Final list containing the tooltip info
     * @param tooltipContext Current tooltip context
     * @param tooltipType Current tooltipFlag
     */
    default void getAttributesTooltip(class_1799 stack, SlotType type, List<class_2561> tooltips, class_1792.class_9635 tooltipContext, class_1836 tooltipType){
        getAttributesTooltip(stack, type, tooltips);

        var component = stack.method_58695(AccessoriesDataComponents.STACK_SETTINGS, AccessoryStackSettings.DEFAULT)
                .slotBasedTooltips()
                .get(type.name());

        if (component != null && !component.equals(class_5244.field_39003)) {
            tooltips.add(component);
        }
    }

    /**
     * Method used to add any additional tooltip information to a given {@link Accessory} tooltip after
     * {@link AccessoriesEventHandler#addEntityBasedTooltipData} if called and at the
     * end of the {@link AccessoriesEventHandler#getTooltipData} call.
     *
     * <p>
     *     Do note that <b>if the given entity</b> is found to not be null, the list passed
     *     contains all tooltip info allowing for positioning before or after the tooltip info
     * </p>
     *
     * @param stack The Stack being referenced
     * @param tooltips Final list containing the tooltip info
     * @param tooltipContext Current tooltip context
     * @param tooltipType Current tooltipFlag
     */
    default void getExtraTooltip(class_1799 stack, List<class_2561> tooltips, class_1792.class_9635 tooltipContext, class_1836 tooltipType){
        getExtraTooltip(stack, tooltips);

        var component = stack.method_58695(AccessoriesDataComponents.STACK_SETTINGS, AccessoryStackSettings.DEFAULT)
                .extraTooltip();

        if (!component.equals(class_5244.field_39003)) {
            tooltips.add(component);
        }
    }

    //--

    /**
     * @deprecated Use {@link #getDynamicModifiers} instead
     */
    @ApiStatus.ScheduledForRemoval(inVersion = "1.22")
    @Deprecated(forRemoval = true)
    default void getModifiers(class_1799 stack, SlotReference reference, AccessoryAttributeBuilder builder){}

    /**
     * @deprecated Use {@link #getAttributesTooltip(class_1799, SlotType, List, class_1792.class_9635, class_1836)}
     */
    @Deprecated(forRemoval = true)
    @ApiStatus.ScheduledForRemoval(inVersion = "1.22")
    default void getAttributesTooltip(class_1799 stack, SlotType type, List<class_2561> tooltips){}

    /**
     * @deprecated Use {@link #getExtraTooltip(class_1799, List, class_1792.class_9635, class_1836)}
     */
    @Deprecated(forRemoval = true)
    @ApiStatus.ScheduledForRemoval(inVersion = "1.22")
    default void getExtraTooltip(class_1799 stack, List<class_2561> tooltips){}
}