package io.wispforest.accessories.api.core;

import io.wispforest.accessories.api.attributes.AccessoryAttributeBuilder;
import io.wispforest.accessories.api.components.AccessoriesDataComponents;
import io.wispforest.accessories.api.components.AccessoryNestContainerContents;
import io.wispforest.accessories.api.events.DropRule;
import io.wispforest.accessories.api.events.SlotStateChange;
import io.wispforest.accessories.api.slot.SlotEntryReference;
import io.wispforest.accessories.api.slot.SlotPath;
import io.wispforest.accessories.api.slot.SlotReference;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.impl.AccessoryNestUtils;
import io.wispforest.accessories.pond.stack.PatchedDataComponentMapExtension;
import it.unimi.dsi.fastutil.Pair;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Consumer;
import java.util.function.Function;
import net.minecraft.class_1282;
import net.minecraft.class_1309;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_2561;

/**
 * An {@link Accessory} that contains and delegates to other accessories in some way
 */
// TODO: POSSIBLY LOOK INTO METHOD OF INDICATING WHEN A SET CALL IS REQUIRED OR FROM ANOTHER MOD TO ALLOW IMMUTABLE NESTS?
public interface AccessoryNest extends Accessory {

    /**
     * @return all inner accessory stacks
     */
    default List<class_1799> getInnerStacks(class_1799 holderStack) {
        var data = holderStack.method_58694(AccessoriesDataComponents.NESTED_ACCESSORIES);

        return data == null ? List.of() : data.accessories();
    }

    /**
     * Sets a given stack at the specified index for the passed holder stack
     *
     * @param holderStack The given HolderStack
     * @param index       The target index
     * @param newStack    The new stack replacing the given index
     */
    default boolean setInnerStack(class_1799 holderStack, int index, class_1799 newStack) {
        if(!AccessoryNest.isAccessoryNest(holderStack)) return false;
        if(AccessoryNest.isAccessoryNest(newStack) && !this.allowDeepRecursion()) return false;

        holderStack.method_57368(
                AccessoriesDataComponents.NESTED_ACCESSORIES,
                new AccessoryNestContainerContents(List.of()),
                contents -> contents.setStack(index, newStack));

        return true;
    }

    /**
     * By default, accessory nests can only go one layer deep as it's hard to track the stack modifications any further
     *
     * @return Whether this implementation of the Accessory nest allows for further nesting of other Nests
     */
    default boolean allowDeepRecursion() {
        // TODO: MAKE DATA ADJUSTABLE?
        return false;
    }

    default List<Pair<DropRule, class_1799>> getDropRules(class_1799 stack, SlotReference reference, class_1282 source) {
        var innerRules = new ArrayList<Pair<DropRule, class_1799>>();

        var innerStacks = getInnerStacks(stack);

        for (int i = 0; i < innerStacks.size(); i++) {
            var innerStack = innerStacks.get(i);

            var rule = AccessoryRegistry.getAccessoryOrDefault(innerStack).getDropRule(innerStack, SlotPath.cloneWithInnerIndex(reference, i), source);

            innerRules.add(Pair.of(rule, innerStack));
        }

        return innerRules;
    }

    //--

    /**
     * Method used to perform some action on a possible {@link AccessoryNest} and return a result from that action or a default value if none found
     *
     * @param holderStack   Potential stack linked to a AccessoryNest
     * @param slotReference The primary SlotReference used from the given call
     * @param func          Action being done
     * @param defaultValue  Default value if stack is not a AccessoryNest
     */
    static <T> T attemptFunction(class_1799 holderStack, SlotReference slotReference, Function<Map<SlotEntryReference, Accessory>, T> func, T defaultValue){
        var data = AccessoryNestUtils.getData(holderStack);

        if(data == null) return defaultValue;

        var t = func.apply(data.getMap(slotReference));

        checkIfChangesOccurred(holderStack, null, data);

        return t;
    }

    /**
     * Method used to perform some action on a possible {@link AccessoryNest} and return a result from that action or a default value if none found
     *
     * @param holderStack   Potential stack linked to a AccessoryNest
     * @param livingEntity Potential Living Entity involved with any stack changes
     * @param func          Action being done
     * @param defaultValue  Default value if stack is not a AccessoryNest
     */
    static <T> T attemptFunction(class_1799 holderStack, @Nullable class_1309 livingEntity, Function<Map<class_1799, Accessory>, T> func, T defaultValue){
        var data = AccessoryNestUtils.getData(holderStack);

        if(data == null) return defaultValue;

        var t = func.apply(data.getMap());

        checkIfChangesOccurred(holderStack, livingEntity, data);

        return t;
    }

    /**
     * Method used to perform some action on a possible {@link AccessoryNest}
     *
     * @param holderStack   Potential stack linked to a AccessoryNest
     * @param slotReference Potential Living Entity involved with any stack changes
     * @param consumer      Action being done
     */
    static void attemptConsumer(class_1799 holderStack, SlotReference slotReference, Consumer<Map<SlotEntryReference, Accessory>> consumer){
        var data = AccessoryNestUtils.getData(holderStack);

        if(data == null) return;

        consumer.accept(data.getMap(slotReference));

        checkIfChangesOccurred(holderStack, slotReference.entity(), data);
    }

    /**
     * Method used to perform some action on a possible {@link AccessoryNest}
     *
     * @param holderStack  Potential stack linked to a AccessoryNest
     * @param livingEntity Potential Living Entity involved with any stack changes
     * @param consumer     Action being done
     */
    static void attemptConsumer(class_1799 holderStack, @Nullable class_1309 livingEntity, Consumer<Map<class_1799, Accessory>> consumer) {
        var data = AccessoryNestUtils.getData(holderStack);

        if (data == null) return;

        consumer.accept(data.getMap());

        checkIfChangesOccurred(holderStack, livingEntity, data);
    }

    static boolean checkIfChangesOccurred(class_1799 holderStack, @Nullable class_1309 livingEntity, AccessoryNestContainerContents data) {
        boolean hasChangeOccurred = false;

        var accessories = data.accessories();

        for (int i = 0; i < accessories.size(); i++) {
            var stack = accessories.get(i);

            if(stack.method_57353() instanceof PatchedDataComponentMapExtension extension && extension.accessories$hasChanged()){
                hasChangeOccurred = true;
            } else if(data.slotChanges().containsKey(i)) {
                hasChangeOccurred = true;
            } else if(AccessoryRegistry.getAccessoryOrDefault(stack) instanceof AccessoryNest) {
                var innerData = AccessoryNestUtils.getData(stack);

                if(innerData != null) {
                    hasChangeOccurred = checkIfChangesOccurred(stack, livingEntity, innerData);

                    if(hasChangeOccurred) break;
                }
            }

            if(hasChangeOccurred) {
                data.slotChanges().putIfAbsent(i, SlotStateChange.MUTATION);
            }
        }

        if(hasChangeOccurred) {
            var nest = (AccessoryNest) AccessoryRegistry.getAccessoryOrDefault(holderStack);

            holderStack.method_57379(AccessoriesDataComponents.NESTED_ACCESSORIES, data);

            nest.onStackChanges(holderStack, data, livingEntity);
        }

        return hasChangeOccurred;
    }

    //--

    static boolean isAccessoryNest(class_1799 holderStack) {
        return AccessoryRegistry.getAccessoryOrDefault(holderStack) instanceof AccessoryNest;
    }

    /**
     * Check and handle any inner stack changes that may have occurred from an action performed on the stacks within the nest
     *
     * @param holderStack  HolderStack containing the nest of stacks
     * @param data         StackData linked to the given HolderStack
     * @param livingEntity Potential Living Entity involved with any stack changes
     */
    default void onStackChanges(class_1799 holderStack, AccessoryNestContainerContents data, @Nullable class_1309 livingEntity){}

    //--

    @Override
    default void onBreak(class_1799 stack, SlotReference reference) {
        attemptConsumer(stack, reference, map -> map.forEach((entryRef, accessory) -> accessory.onBreak(entryRef.stack(), entryRef.reference())));
    }

    @Override
    default boolean canEquipFromUse(class_1799 stack) {
        return attemptFunction(stack, null, (Map<class_1799, Accessory> map) -> {
            for (var entry : map.entrySet()) {
                if(!entry.getValue().canEquipFromUse(entry.getKey())) return false;
            }

            return true;
        }, true);
    }

    @Override
    default void onEquipFromUse(class_1799 stack, SlotReference reference) {
        attemptConsumer(stack, reference, map -> map.forEach((entryRef, accessory) -> accessory.onEquipFromUse(entryRef.stack(), entryRef.reference())));
    }

    @Override
    default void tick(class_1799 stack, SlotReference reference) {
        attemptConsumer(stack, reference, map -> map.forEach((entryRef, accessory) -> accessory.tick(entryRef.stack(), entryRef.reference())));
    }

    @Override
    default void onEquip(class_1799 stack, SlotReference reference) {
        attemptConsumer(stack, reference, map -> map.forEach((entryRef, accessory) -> accessory.onEquip(entryRef.stack(), entryRef.reference())));
    }

    @Override
    default void onUnequip(class_1799 stack, SlotReference reference) {
        attemptConsumer(stack, reference, map -> map.forEach((entryRef, accessory) -> accessory.onUnequip(entryRef.stack(), entryRef.reference())));
    }

    @Override
    default boolean canEquip(class_1799 stack, SlotReference reference) {
        return attemptFunction(stack, reference, map -> {
            MutableBoolean canEquip = new MutableBoolean(true);

            map.forEach((entryRef, accessory) -> canEquip.setValue(canEquip.booleanValue() && accessory.canEquip(entryRef.stack(), entryRef.reference())));

            return canEquip.getValue();
        }, false);
    }

    @Override
    default boolean canUnequip(class_1799 stack, SlotReference reference) {
        return attemptFunction(stack, reference, map -> {
            MutableBoolean canUnequip = new MutableBoolean(true);

            map.forEach((entryRef, accessory) -> canUnequip.setValue(canUnequip.booleanValue() && accessory.canUnequip(entryRef.stack(), entryRef.reference())));

            return canUnequip.getValue();
        }, true);
    }

    @Override
    default void getDynamicModifiers(class_1799 stack, SlotReference reference, AccessoryAttributeBuilder builder) {
        attemptConsumer(stack, reference, innerMap -> innerMap.forEach((entryRef, accessory) -> {
            accessory.getDynamicModifiers(entryRef.stack(), entryRef.reference(), new AccessoryAttributeBuilder(entryRef.reference(), builder));
        }));
    }

    @Override
    default void getAttributesTooltip(class_1799 stack, SlotType type, List<class_2561> tooltips, class_1792.class_9635 tooltipContext, class_1836 tooltipType) {
        attemptConsumer(stack, (class_1309) null, map -> map.forEach((stack1, accessory) -> accessory.getAttributesTooltip(stack1, type, tooltips, tooltipContext, tooltipType)));
    }

    @Override
    default void getExtraTooltip(class_1799 stack, List<class_2561> tooltips, class_1792.class_9635 tooltipContext, class_1836 tooltipType) {
        attemptConsumer(stack, (class_1309) null, map -> map.forEach((stack1, accessory) -> accessory.getExtraTooltip(stack1, tooltips, tooltipContext, tooltipType)));
    }
}
