package io.wispforest.accessories.api.slot;

import com.google.common.collect.ImmutableMap;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.Accessories;
import io.wispforest.accessories.api.core.Accessory;
import io.wispforest.accessories.data.SlotTypeLoader;
import io.wispforest.accessories.impl.slot.ExtraSlotTypeProperties;
import io.wispforest.accessories.impl.slot.StrictMode;
import net.fabricmc.fabric.api.event.Event;
import net.fabricmc.fabric.api.event.EventFactory;
import net.minecraft.class_1299;
import net.minecraft.class_1799;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_7877;
import org.apache.commons.lang3.function.TriFunction;
import org.jetbrains.annotations.ApiStatus;
import org.slf4j.Logger;

import java.util.*;

/**
 * Utilities for adding custom slots that are rendered outside the accessories screen with options for restricting
 * modification via data pack.
 */
public class UniqueSlotHandling {

    private static final Logger LOGGER = LogUtils.getLogger();

    /**
     * Event for registration of unique slots.
     */
    public static final Event<RegistrationCallback> EVENT = EventFactory.createArrayBacked(RegistrationCallback.class, (invokers) -> factory -> {
        for (var invoker : invokers) invoker.registerSlots(factory);
    });

    public static boolean isUniqueSlot(String slotType) {
        return slotType.split(":").length > 1;
    }

    public static boolean isUniqueGroup(String group, boolean isClient) {
        return (isClient ? GROUPS_CLIENT : GROUPS_SERVER).contains(group);
    }

    public interface RegistrationCallback {
        /**
         * Invoked when unique slots are being loaded.
         *
         * @param factory the factory for registering unique slots
         */
        void registerSlots(UniqueSlotBuilderFactory factory);
    }

    public interface UniqueSlotBuilderFactory {
        /**
         * Starts building a new unique slot.
         */
        UniqueSlotBuilder create(class_2960 location, int amount);
    }

    /**
     * Builder Object used to create unique slots with the base required info and some other extra properties that can
     * be adjusted.
     */
    public interface UniqueSlotBuilder {
        /**
         * Sets the unique slot's slot predicates.
         * <p>
         * By default, the tag-based slot predicate is used.
         */
        UniqueSlotBuilder slotPredicates(class_2960... locations);

        /**
         * Sets the list of entity types that will have this slot.
         */
        UniqueSlotBuilder validTypes(class_1299<?>... types);

        /**
         * Controls whether modification of this slot via a data pack is disallowed.
         */
        default UniqueSlotBuilder strictMode(boolean value) {
            return strictMode(value ? StrictMode.FULL : StrictMode.NONE);
        }

        /**
         * Controls whether modification of this slot via a data pack is disallowed.
         */
        UniqueSlotBuilder strictMode(StrictMode mode);

        /**
         * Controls whether this slot can be resized later (e.g. with a data pack or attribute)
         */
        UniqueSlotBuilder allowResizing(boolean value);

        /**
         * Controls whether accessories can be equipped from use into this slot.
         * <p>
         * A value of {@code false} overrides {@link Accessory#canEquipFromUse(class_1799)}.
         */
        UniqueSlotBuilder allowEquipFromUse(boolean value);

        UniqueSlotBuilder allowTooltipInfo(boolean value);

        /**
         * Builds and registers the unique slot.
         * @return a reference to the unique slot type
         */
        SlotTypeReference build();
    }

    public static void bootStrapDataGen(class_7877 registryBuilder) {
        registryBuilder.method_46777(class_5321.method_29180(Accessories.of("unique_slots_registry_bootstrap")), bootstrapContext -> {
            UniqueSlotHandling.gatherUniqueSlots((location, integer, resourceLocations) -> location::toString);
        });
    }

    //--

    @ApiStatus.Internal
    public static void gatherUniqueSlots(TriFunction<class_2960, Integer, Collection<class_2960>, SlotTypeReference> slotRegistration) {
        GROUPS_SERVER.clear();
        SLOT_TO_ENTITIES.clear();

        UniqueSlotBuilderFactory eventRegistration = (location, amount) -> new ServerUniqueSlotBuilder(location, amount, slotRegistration);

        EVENT.invoker().registerSlots(eventRegistration);
    }

    @ApiStatus.Internal
    public static void buildClientSlotReferences() {
        UniqueSlotBuilderFactory eventRegistration = (location, amount) -> new UniqueSlotBuilder() {
            @Override public UniqueSlotBuilder slotPredicates(class_2960... locations) { return this; }
            @Override public UniqueSlotBuilder validTypes(class_1299<?>... types) { return this; }
            @Override public UniqueSlotBuilder strictMode(StrictMode value) { return this; }
            @Override public UniqueSlotBuilder allowResizing(boolean value) { return this; }
            @Override public UniqueSlotBuilder allowEquipFromUse(boolean value) { return this; }
            @Override public UniqueSlotBuilder allowTooltipInfo(boolean value) { return this; }

            @Override
            public SlotTypeReference build() {
                var name = location.toString();

                var slotType = SlotTypeLoader.INSTANCE.getSlotType(true, name);

                if(slotType == null) {
                    LOGGER.error("Unable to get the given unique slot as the slot has been not been synced to the client! [Name: {}]", name);
                }

                return () -> name;
            }
        };

        EVENT.invoker().registerSlots(eventRegistration);
    }

    private static final class ServerUniqueSlotBuilder implements UniqueSlotBuilder {
        private final class_2960 location;
        private final int amount;
        private Set<class_2960> slotPredicates = Set.of(Accessories.of("tag"));
        private Set<class_1299<?>> validTypes = Set.of();

        private StrictMode mode = StrictMode.FULL;
        private boolean allowResizing = false;
        private boolean allowEquipFromUse = true;
        private boolean allowTooltipInfo = true;

        private final TriFunction<class_2960, Integer, Collection<class_2960>, SlotTypeReference> slotRegistration;

        ServerUniqueSlotBuilder(class_2960 location, int amount, TriFunction<class_2960, Integer, Collection<class_2960>, SlotTypeReference> slotRegistration){
            this.location = location;
            this.amount = amount;

            this.slotRegistration = slotRegistration;
        }

        @Override
        public ServerUniqueSlotBuilder slotPredicates(class_2960... locations) {
            this.slotPredicates = Set.of(locations);

            return this;
        }

        @Override
        public ServerUniqueSlotBuilder validTypes(class_1299<?>... types) {
            this.validTypes = Set.of(types);

            return this;
        }

        public ServerUniqueSlotBuilder strictMode(StrictMode mode) {
            this.mode = mode;

            return this;
        }

        @Override
        public ServerUniqueSlotBuilder allowResizing(boolean value) {
            this.allowResizing = value;

            return this;
        }

        @Override
        public ServerUniqueSlotBuilder allowEquipFromUse(boolean value) {
            this.allowEquipFromUse = value;

            return this;
        }

        public UniqueSlotBuilder allowTooltipInfo(boolean value) {
            this.allowTooltipInfo = value;

            return this;
        }

        /**
         * Builds and registers the unique slot.
         * @return a reference to the unique slot type
         */
        public SlotTypeReference build() {
            var slotTypeRef = this.slotRegistration.apply(location, amount, slotPredicates);

            SLOT_TO_ENTITIES.put(slotTypeRef.slotName(), Set.copyOf(this.validTypes));

            ExtraSlotTypeProperties.getProperties(false)
                    .put(slotTypeRef.slotName(), new ExtraSlotTypeProperties(this.allowResizing, this.mode, this.allowEquipFromUse, this.allowTooltipInfo));

            return slotTypeRef;
        }
    }

    private static final Map<String, Set<class_1299<?>>> SLOT_TO_ENTITIES = new HashMap<>();

    private static final Set<String> GROUPS_SERVER = new LinkedHashSet<>();
    private static final Set<String> GROUPS_CLIENT = new LinkedHashSet<>();

    @ApiStatus.Internal
    public static Map<String, Set<class_1299<?>>> getSlotToEntities() {
        return ImmutableMap.copyOf(SLOT_TO_ENTITIES);
    }

    @ApiStatus.Internal
    public static void addGroup(String group) {
        GROUPS_SERVER.add(group);
    }

    @ApiStatus.Internal
    public static Set<String> getGroups(boolean isClient) {
        return isClient ? GROUPS_CLIENT : GROUPS_SERVER;
    }

    @ApiStatus.Internal
    public static void setClientGroups(Collection<String> set) {
        GROUPS_CLIENT.clear();
        GROUPS_CLIENT.addAll(set);
    }
}
