package io.wispforest.accessories.commands;

import com.mojang.brigadier.StringReader;
import com.mojang.brigadier.arguments.ArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import com.mojang.brigadier.suggestion.Suggestions;
import com.mojang.brigadier.suggestion.SuggestionsBuilder;
import com.mojang.datafixers.util.Either;
import io.wispforest.accessories.api.AccessoriesCapability;
import io.wispforest.accessories.api.AccessoriesContainer;
import io.wispforest.accessories.api.slot.SlotPath;
import io.wispforest.accessories.data.EntitySlotLoader;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_2168;
import net.minecraft.class_2172;
import net.minecraft.class_2186;
import net.minecraft.class_2561;
import net.minecraft.class_9010;
import net.minecraft.class_9348;
import net.minecraft.class_9349;

public record AccessoriesMixedSlotArgument(String entityArgumentName) implements ArgumentType<Either<SlotPath, Integer>> {

    private static final Collection<String> EXAMPLES = Arrays.asList("back/1", "charm/1", "feet.1", "weapon");

    private static final DynamicCommandExceptionType ERROR_UNKNOWN_SLOT = new DynamicCommandExceptionType(
            object -> class_2561.method_54159("slot.unknown", object)
    );

    private static final DynamicCommandExceptionType ERROR_ONLY_SINGLE_SLOT_ALLOWED = new DynamicCommandExceptionType(
            object -> class_2561.method_54159("slot.only_single_allowed", object)
    );

    public static AccessoriesMixedSlotArgument slot(String entityArgumentName) {
        return new AccessoriesMixedSlotArgument(entityArgumentName);
    }

    public static Either<SlotPath, Integer> getSlot(CommandContext<class_2168> context, String name) {
        return context.getArgument(name, Either.class);
    }

    @Override
    public Either<SlotPath, Integer> parse(StringReader reader) throws CommandSyntaxException {
        String string = class_9010.method_58128(reader, c -> c != ' ');
        var slotPath = SlotPath.fromString(string);

        if (slotPath != null) return Either.left(slotPath);

        var vanillaSlot = parseVanillaSlot(reader, string);

        if (vanillaSlot != null) return Either.right(vanillaSlot);

        throw ERROR_UNKNOWN_SLOT.createWithContext(reader, string);
    }

    @Nullable
    public Integer parseVanillaSlot(StringReader reader, String string) throws CommandSyntaxException {
        class_9348 slotRange = class_9349.method_58080(string);

        if (slotRange == null) {
            return null;
        } else if (slotRange.method_58077() != 1) {
            throw ERROR_ONLY_SINGLE_SLOT_ALLOWED.createWithContext(reader, string);
        } else {
            return slotRange.method_58075().getInt(0);
        }
    }

    @Override
    public <S> CompletableFuture<Suggestions> listSuggestions(CommandContext<S> commandContext, SuggestionsBuilder suggestionsBuilder) {
        S object = commandContext.getSource();

        if (object instanceof class_2168) {
            List<String> suggestions = new ArrayList<>();

            try {
                // TODO: ISSUE THIS DOSE NOT WORK ON FABRIC
                var entityTarget = class_2186.method_9313((CommandContext<class_2168>) commandContext, entityArgumentName);

                if (entityTarget instanceof class_1309 livingEntity) {
                    var capability = livingEntity.accessoriesCapability();

                    if (capability != null) {
                        suggestions.addAll(
                                EntitySlotLoader.getEntitySlots(livingEntity).values().stream().flatMap(slotType -> {
                                    var slotPaths = new ArrayList<String>();

                                    var container = capability.getContainer(slotType);

                                    if (container != null) {
                                        for (int i = 0; i < container.getSize(); i++) {
                                            slotPaths.add(SlotPath.createBaseSlotPath(slotType, i));
                                        }
                                    }

                                    return slotPaths.stream();
                                }).toList()
                        );
                    }
                }
            } catch (Exception e) {}

            suggestions.addAll(class_9349.method_58088().toList());

            return class_2172.method_9265(suggestions, suggestionsBuilder);
        } else {
            return object instanceof class_2172 sharedSuggestionProvider
                    ? sharedSuggestionProvider.method_9261(commandContext)
                    : Suggestions.empty();
        }
    }

    @Override
    public Collection<String> getExamples() {
        return EXAMPLES;
    }
}
