package io.wispforest.accessories.data;

import com.mojang.logging.LogUtils;
import io.wispforest.accessories.Accessories;
import io.wispforest.accessories.api.slot.SlotType;
import io.wispforest.accessories.api.slot.UniqueSlotHandling;
import io.wispforest.accessories.data.api.EndecDataLoader;
import io.wispforest.accessories.data.api.SyncedDataHelper;
import io.wispforest.accessories.impl.core.AccessoriesHolderImpl;
import io.wispforest.accessories.impl.slot.ExtraSlotTypeProperties;
import io.wispforest.accessories.impl.slot.StrictMode;
import io.wispforest.accessories.pond.ReplaceableJsonResourceReloadListener;
import io.wispforest.accessories.utils.CollectionUtils;
import io.wispforest.endec.Endec;
import io.wispforest.endec.StructEndec;
import io.wispforest.endec.impl.StructEndecBuilder;
import it.unimi.dsi.fastutil.Pair;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

import java.util.*;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import net.minecraft.class_1299;
import net.minecraft.class_1309;
import net.minecraft.class_1937;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import net.minecraft.class_7924;

/**
 * Resource Reload in which handles the loading of {@link SlotType}'s bindings
 * to the targeted {@link class_1299} though a {@link class_6862} or {@link class_2960}
 */
public class EntitySlotLoader extends EndecDataLoader<EntitySlotLoader.RawEnityBinding> implements SyncedDataHelper<SequencedMap<class_1299<?>, List<String>>> {

    private static final Logger LOGGER = LogUtils.getLogger();

    public static final EntitySlotLoader INSTANCE = new EntitySlotLoader();

    private Map<class_6862<class_1299<?>>, Map<String, SlotType>> tagToBoundSlots = new HashMap<>();
    private Map<class_1299<?>, Map<String, SlotType>> entityToBoundSlots = new HashMap<>();

    private SequencedMap<class_1299<?>, SequencedMap<String, SlotType>> server = new LinkedHashMap<>();
    private SequencedMap<class_1299<?>, SequencedMap<String, SlotType>> client = new LinkedHashMap<>();

    protected EntitySlotLoader() {
        super(Accessories.of("entity_slot_loader"), "accessories/entity", RawEnityBinding.ENDEC, class_3264.field_14190, Set.of(SlotTypeLoader.INSTANCE.getId()));

        ReplaceableJsonResourceReloadListener.toggleValue(this);
    }

    //--

    /**
     * @return The valid {@link SlotType}'s for given {@link class_1309} based on its {@link class_1299}
     */
    public static Map<String, SlotType> getEntitySlots(class_1309 livingEntity){
        return getEntitySlots(livingEntity.method_73183(), livingEntity.method_5864());
    }

    /**
     * @return The valid {@link SlotType}'s for given {@link class_1299}
     */
    public static Map<String, SlotType> getEntitySlots(class_1937 level, class_1299<?> entityType){
        var map = EntitySlotLoader.INSTANCE.getSlotTypes(level.method_8608(), entityType);

        return map != null ? map : Map.of();
    }

    //--

    @Nullable
    public final Map<String, SlotType> getSlotTypes(boolean isClientSide, class_1299<?> entityType){
        return this.getEntitySlotData(isClientSide).get(entityType);
    }

    @ApiStatus.Internal
    public final Map<class_1299<?>, Map<String, SlotType>> getEntitySlotData(boolean isClientSide){
        return (Map) (isClientSide ? this.client : this.server);
    }

    //--

    public record RawEnityBinding(Set<String> entityTargets, Set<String> slotTypes) {
        public static final StructEndec<RawEnityBinding> ENDEC = StructEndecBuilder.of(
                Endec.STRING.setOf().fieldOf("entities", RawEnityBinding::entityTargets),
                Endec.STRING.setOf().fieldOf("slots", RawEnityBinding::slotTypes),
                RawEnityBinding::new
        );
    }

    @Override
    public Endec<SequencedMap<class_1299<?>, List<String>>> syncDataEndec() {
        return Endec.map(LinkedHashMap::new,
            type -> class_7923.field_41177.method_10221(type).toString(), strType -> class_7923.field_41177.method_63535(class_2960.method_60654(strType)),
            Endec.STRING.listOf());
    }

    @Override
    public void onReceivedData(SequencedMap<class_1299<?>, List<String>> data) {
        SequencedMap<class_1299<?>, SequencedMap<String, SlotType>> entitySlotTypes = new LinkedHashMap<>();

        for (var entry : data.entrySet()) {
            var map = entry.getValue().stream()
                    .map(string -> SlotTypeLoader.INSTANCE.getSlotType(true, string))
                    .collect(CollectionUtils.toLinkedMap(SlotType::name));

            entitySlotTypes.put(entry.getKey(), map);
        }

        this.client = Collections.unmodifiableSequencedMap(entitySlotTypes);

        AccessoriesHolderImpl.clearValidationCache(true);
    }

    @Override
    public SequencedMap<class_1299<?>, List<String>> getServerData() {
        var entitySlots = new LinkedHashMap<class_1299<?>, List<String>>();

        for (var entry : server.entrySet()) {
            entitySlots.put(entry.getKey(), List.copyOf(entry.getValue().keySet()));
        }

        return entitySlots;
    }

    public void buildEntryMap() {
        var tempMap = new LinkedHashMap<class_1299<?>, SequencedMap<String, SlotType>>();

        this.tagToBoundSlots.forEach((entityTag, slots) -> {
            var entityTypes = class_7923.field_41177.method_46733(entityTag)
                    .map(holders -> holders.method_40239().map(class_6880::comp_349).collect(Collectors.toSet()))
                    .orElseGet(() -> {
                        LOGGER.warn("[EntitySlotLoader]: Unable to locate the given EntityType Tag used within a slot entry: [Location: {}]", entityTag.comp_327());
                        return Set.of();
                    });

            entityTypes.forEach(entityType -> {
                tempMap.computeIfAbsent(entityType, entityType1 -> new LinkedHashMap<>())
                        .putAll(slots);
            });
        });

        this.entityToBoundSlots.forEach((entityType, slots) -> {
            tempMap.computeIfAbsent(entityType, entityType1 -> new LinkedHashMap<>())
                    .putAll(slots);
        });

        var finishMap = new LinkedHashMap<class_1299<?>, SequencedMap<String, SlotType>>();

        tempMap.forEach((entityType, slotsBuilder) -> finishMap.put(entityType, Collections.unmodifiableSequencedMap(slotsBuilder)));

        this.server = finishMap;

        AccessoriesHolderImpl.clearValidationCache(false);

        this.tagToBoundSlots.clear();
        this.entityToBoundSlots.clear();
    }

    //--

    @Override
    protected void apply(Map<class_2960, RawEnityBinding> rawData, class_3300 resourceManager, class_3695 profiler) {
        var allSlotTypes = SlotTypeLoader.INSTANCE.getEntries(false);

        this.tagToBoundSlots.clear();
        this.entityToBoundSlots.clear();

        for (var resourceEntry : rawData.entrySet()) {
            var location = resourceEntry.getKey();
            var rawEnityBinding = resourceEntry.getValue();

            var slots = new LinkedHashMap<String, SlotType>();

            rawEnityBinding.slotTypes().stream().map(slotName -> {
                return Pair.of(slotName, allSlotTypes.get(Accessories.parseLocationOrDefault(slotName)));
            }).forEach(slotInfo -> {
                var slotType = slotInfo.right();

                if(slotType != null) {
                    if(!ExtraSlotTypeProperties.getProperty(slotInfo.left(), false).strictMode().equals(StrictMode.FULL)) {
                        slots.put(slotType.name(), slotType);
                    } else {
                        LOGGER.warn("Unable to add the given slot [{}] to the given group due to it being in strict mode! [Location: {}]", slotInfo.left(), location);
                    }
                } else if (slotType == null) {
                    LOGGER.warn("Unable to locate a given slot [{}] to add to a given entity('s) as it was not registered: [Location: {}]", slotInfo.first(), location);
                }
            });

            //--

            rawEnityBinding.entityTargets().forEach(string -> {
                if(string.contains("#")){
                    var entityTypeTagLocation = class_2960.method_12829(string.replace("#", ""));

                    var entityTypeTag = class_6862.method_40092(class_7924.field_41266, entityTypeTagLocation);

                    tagToBoundSlots.computeIfAbsent(entityTypeTag, entityTag -> new HashMap<>())
                            .putAll(slots);
                } else {
                    Optional.ofNullable(class_2960.method_12829(string))
                            .flatMap(class_7923.field_41177::method_17966)
                            .ifPresentOrElse(entityType -> {
                                entityToBoundSlots.computeIfAbsent(entityType, entityType1 -> new HashMap<>())
                                        .putAll(slots);
                            }, () -> {
                                LOGGER.warn("[EntitySlotLoader]: Unable to locate the given EntityType [{}] within the registries for a slot entry: [Location: {}]", string, location);
                            });
                }
            });
        }

        for (var entry : UniqueSlotHandling.getSlotToEntities().entrySet()) {
            var slotType = SlotTypeLoader.INSTANCE.getEntries(false).get(Accessories.parseLocationOrDefault(entry.getKey()));

            for (var entityType : entry.getValue()) {
                entityToBoundSlots.computeIfAbsent(entityType, entityType1 -> new LinkedHashMap<>())
                        .put(slotType.name(), slotType);
            }
        }
    }
}
