package io.wispforest.accessories.data.api;

import com.google.common.collect.BiMap;
import io.wispforest.endec.Endec;
import io.wispforest.endec.SerializationContext;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.IntFunction;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_3695;

public abstract class ManagedEndecDataLoader<V, D> extends EndecDataLoader<D> implements SyncedDataHelper<SequencedBiMap<class_2960, V>>, LookupDataLoader<V> {

    private final SequencedBiMap<class_2960, V> server = SequencedBiMap.of(LinkedHashMap::new);
    private final SequencedBiMap<class_2960, V> client = SequencedBiMap.of(LinkedHashMap::new);

    private final Endec<V> valueEndec;
    private final Endec<SequencedBiMap<class_2960, V>> mapEndec;

    protected ManagedEndecDataLoader(class_2960 id, String type, Endec<V> valueEndec, Endec<D> dataEndec, class_3264 packType) {
        this(id, type, valueEndec, dataEndec, packType, false);
    }

    protected ManagedEndecDataLoader(class_2960 id, String type, Endec<V> valueEndec, Endec<D> dataEndec, class_3264 packType, boolean requiresRegistries) {
        this(id, type, valueEndec, dataEndec, packType, SerializationContext.empty(), requiresRegistries);
    }

    protected ManagedEndecDataLoader(class_2960 id, String type, Endec<V> valueEndec, Endec<D> dataEndec, class_3264 packType, Set<class_2960> dependencies) {
        this(id, type, valueEndec, dataEndec, packType, SerializationContext.empty(), false, dependencies);
    }

    protected ManagedEndecDataLoader(class_2960 id, String type, Endec<V> valueEndec, Endec<D> dataEndec, class_3264 packType, SerializationContext context, boolean requiresRegistries) {
        this(id, type, valueEndec, dataEndec, packType, context, requiresRegistries, Set.of());
    }

    protected ManagedEndecDataLoader(class_2960 id, String type, Endec<V> valueEndec, Endec<D> dataEndec, class_3264 packType, SerializationContext context, boolean requiresRegistries, Set<class_2960> dependencies) {
        super(id, type, dataEndec, packType, context, requiresRegistries, dependencies);

        this.valueEndec = valueEndec;
        this.mapEndec = biMapEndec(value -> SequencedBiMap.of(LinkedHashMap::new), class_2960::toString, class_2960::method_12829, valueEndec);
    }

    public static <V, D> ManagedEndecDataLoader<V, D> of(class_2960 id, String type, Endec<V> valueEndec, Endec<D> dataEndec, class_3264 packType, Function<Map<class_2960, D>, Map<class_2960, V>> mapFrom) {
        return new ManagedEndecDataLoader<V, D>(id, type, valueEndec, dataEndec, packType){
            @Override
            public Map<class_2960, V> mapFrom(Map<class_2960, D> rawData) {
                return mapFrom.apply(rawData);
            }
        };
    }

    @Override
    public Map<class_2960, V> getEntries(boolean isClientSide) {
        return Collections.unmodifiableMap(isClientSide ? client : server);
    }

    @Override
    @Nullable
    public V getEntry(class_2960 id, boolean isClientSide) {
        return (isClientSide ? client : server).get(id);
    }

    @Override
    public class_2960 getId(V t, boolean isClientSide) {
        return (isClientSide ? client : server).inverse().get(t);
    }

    //--

    public abstract Map<class_2960, V> mapFrom(Map<class_2960, D> rawData);

    protected void onSync() {}

    @Override
    public final SequencedBiMap<class_2960, V> getServerData() {
        return this.server;
    }

    @Override
    public final Endec<SequencedBiMap<class_2960, V>> syncDataEndec() {
        return this.mapEndec;
    }

    @Override
    public final void onReceivedData(SequencedBiMap<class_2960, V> data) {
        this.client.clear();
        this.client.putAll(data);

        this.onSync();
    }

    //--

    @Override
    protected void apply(Map<class_2960, D> loadedObjects, class_3300 resourceManager, class_3695 profiler) {
        this.server.clear();
        this.server.putAll(mapFrom(loadedObjects));
    }

    @ApiStatus.Internal
    private static <K, V, M extends BiMap<K, V>> Endec<M> biMapEndec(IntFunction<M> biMapFactory, Function<K, String> keyToString, Function<String, K> stringToKey, Endec<V> valueEndec) {
        return Endec.of((ctx, serializer, map) -> {
            try (var mapState = serializer.map(ctx, valueEndec, map.size())) {
                map.forEach((k, v) -> mapState.entry(keyToString.apply(k), v));
            }
        }, (ctx, deserializer) -> {
            var mapState = deserializer.map(ctx, valueEndec);

            var map = biMapFactory.apply(mapState.estimatedSize());
            mapState.forEachRemaining(entry -> map.put(stringToKey.apply(entry.getKey()), entry.getValue()));

            return map;
        });
    }
}
