package io.wispforest.accessories.impl.core;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.AccessoriesInternals;
import io.wispforest.accessories.api.AccessoriesCapability;
import io.wispforest.accessories.api.AccessoriesContainer;
import io.wispforest.accessories.api.core.Accessory;
import io.wispforest.accessories.api.core.AccessoryRegistry;
import io.wispforest.accessories.api.equip.EquipAction;
import io.wispforest.accessories.api.equip.EquipCheck;
import io.wispforest.accessories.api.slot.SlotPredicateRegistry;
import io.wispforest.accessories.api.slot.SlotReference;
import io.wispforest.accessories.data.EntitySlotLoader;
import io.wispforest.accessories.endec.NbtMapCarrier;
import io.wispforest.accessories.impl.AccessoryAttributeLogic;
import io.wispforest.accessories.impl.slot.ExtraSlotTypeProperties;
import io.wispforest.accessories.networking.AccessoriesNetworking;
import io.wispforest.accessories.networking.client.SyncEntireContainer;
import io.wispforest.accessories.utils.InstanceEndec;
import io.wispforest.endec.SerializationContext;
import io.wispforest.endec.util.MapCarrierDecodable;
import io.wispforest.endec.util.MapCarrierEncodable;
import io.wispforest.owo.serialization.RegistriesAttribute;
import it.unimi.dsi.fastutil.Pair;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import net.minecraft.class_1309;
import net.minecraft.class_1322;
import net.minecraft.class_1799;
import net.minecraft.class_3222;

@ApiStatus.Internal
public class AccessoriesCapabilityImpl implements AccessoriesCapability, InstanceEndec {

    private static final Logger LOGGER = LogUtils.getLogger();

    private final class_1309 entity;

    public AccessoriesCapabilityImpl(class_1309 entity) {
        this.entity = entity;
    }

    @Override
    public class_1309 entity() {
        return entity;
    }

    @Override
    public Map<String, AccessoriesContainer> getContainers() {
        var holder = AccessoriesHolderImpl.getHolder(this);

        // Dirty patch to handle capability mismatch on containers when transferring it
        // TODO: Wonder if this is the best solution to the problem of desynced when data is copied
        for (var container : holder.getAllSlotContainers().values()) {
            if(this.entity == container.capability().entity()) break;

            ((AccessoriesContainerImpl) container).capability = this;
        }

        return holder.getSlotContainers();
    }

    @Override
    public void reset(boolean loadedFromTag) {
        if (this.entity.method_73183().method_8608()) return;

        var holder = ((AccessoriesHolderImpl) AccessoriesInternals.getHolder(entity));

        if (!loadedFromTag) {
            var oldContainers = Map.copyOf(holder.getAllSlotContainers());

            holder.init(this);

            var currentContainers = holder.getAllSlotContainers();

            oldContainers.forEach((s, oldContainer) -> {
                var currentContainer = currentContainers.get(s);

                currentContainer.getAccessories().setFromPrev(oldContainer.getAccessories());

                currentContainer.markChanged(false);
            });
        } else {
            holder.init(this);
        }

        if (!(this.entity instanceof class_3222 serverPlayer) || serverPlayer.field_13987 == null) return;

        var carrier = NbtMapCarrier.of();

        holder.encode(carrier, SerializationContext.attributes(RegistriesAttribute.of(this.entity.method_73183().method_30349())));

        AccessoriesNetworking.sendToTrackingAndSelf(serverPlayer, new SyncEntireContainer(serverPlayer.method_5628(), carrier));
    }

    private boolean updateContainersLock = false;

    @Override
    public void updateContainers() {
        if (updateContainersLock) return;

        boolean hasUpdateOccurred;

        var containers = this.getContainers().values();

        this.updateContainersLock = true;

        do {
            hasUpdateOccurred = false;

            for (var container : containers) {
                if (!container.hasChanged()) {
                    continue;
                }

                container.update();

                hasUpdateOccurred = true;
            }
        } while (hasUpdateOccurred);

        this.updateContainersLock = false;
    }

    @Override
    public void addTransientSlotModifiers(Multimap<String, class_1322> modifiers) {
        var containers = this.getContainers();

        for (var entry : modifiers.asMap().entrySet()) {
            if (!containers.containsKey(entry.getKey())) continue;

            var container = containers.get(entry.getKey());

            entry.getValue().forEach(container::addTransientModifier);
        }
    }

    @Override
    public void addPersistentSlotModifiers(Multimap<String, class_1322> modifiers) {
        var containers = this.getContainers();

        for (var entry : modifiers.asMap().entrySet()) {
            if (!containers.containsKey(entry.getKey())) continue;

            var container = containers.get(entry.getKey());

            entry.getValue().forEach(container::addPersistentModifier);
        }
    }

    @Override
    public void removeSlotModifiers(Multimap<String, class_1322> modifiers) {
        var containers = this.getContainers();

        for (var entry : modifiers.asMap().entrySet()) {
            if (!containers.containsKey(entry.getKey())) continue;

            var container = containers.get(entry.getKey());

            entry.getValue().forEach(modifier -> container.removeModifier(modifier.comp_2447()));
        }
    }

    @Override
    public Multimap<String, class_1322> getSlotModifiers() {
        Multimap<String, class_1322> modifiers = HashMultimap.create();

        this.getContainers().forEach((s, container) -> modifiers.putAll(s, container.getModifiers().values()));

        return modifiers;
    }

    @Override
    public void clearSlotModifiers() {
        this.getContainers().forEach((s, container) -> container.clearModifiers());
    }

    @Override
    public void clearCachedSlotModifiers() {
        var slotModifiers = HashMultimap.<String, class_1322>create();

        var containers = this.getContainers();

        containers.forEach((name, container) -> {
            var modifiers = container.getCachedModifiers();

            if (modifiers.isEmpty()) return;

            var accessories = container.getAccessories();

            for (int i = 0; i < accessories.method_5439(); i++) {
                var stack = accessories.method_5438(i);

                if (stack.method_7960()) continue;

                var slotReference = container.createReference(i);

                slotModifiers.putAll(AccessoryAttributeLogic.getAttributeModifiers(stack, slotReference).getSlotModifiers());
            }
        });

        slotModifiers.asMap().forEach((name, modifiers) -> {
            if (!containers.containsKey(name)) return;

            var container = containers.get(name);

            modifiers.forEach(container.getCachedModifiers()::remove);

            container.clearCachedModifiers();
        });
    }

    //--

    @Nullable
    public Pair<SlotReference, EquipAction> canEquipAccessory(class_1799 stack, boolean allowSwapping, EquipCheck extraCheck) {
        var accessory = AccessoryRegistry.getAccessoryOrDefault(stack);

        if (accessory == null) return null;

        var validContainers = new HashMap<String, AccessoriesContainer>();

        if (stack.method_7960() && allowSwapping) {
            var allContainers = this.getContainers();

            EntitySlotLoader.getEntitySlots(this.entity())
                    .forEach((s, slotType) -> validContainers.put(s, allContainers.get(slotType.name())));
        } else {
            // First attempt to equip an accessory within empty slot
            for (var container : this.getContainers().values()) {
                if (container.getSize() <= 0) continue;

                boolean isValid = SlotPredicateRegistry.canInsertIntoSlot(stack, container.createReference(0));

                // Prevents checking containers that will never allow for the given stack to be equipped within it
                if (!isValid || !ExtraSlotTypeProperties.getProperty(container.getSlotName(), entity.method_73183().method_8608()).allowEquipFromUse()) continue;

                if (allowSwapping) validContainers.put(container.getSlotName(), container);

                var accessories = container.getAccessories();

                for (int i = 0; i < container.getSize(); i++) {
                    var slotStack = accessories.method_5438(i);
                    var slotReference = container.createReference(i);

                    if (slotStack.method_7960()
                            && AccessoryRegistry.canUnequip(slotStack, slotReference)
                            && SlotPredicateRegistry.canInsertIntoSlot(stack, slotReference)
                            && extraCheck.isValid(slotStack, false)) {
                        return Pair.of(container.createReference(i), (newStack) -> setStack(slotReference, newStack, false));
                    }
                }
            }
        }

        // Second attempt to equip an accessory within the first slot by swapping if allowed
        for (var validContainer : validContainers.values()) {
            var accessories = validContainer.getAccessories();

            for (int i = 0; i < accessories.method_5439(); i++) {
                var slotStack = accessories.method_5438(i).method_7972();
                var slotReference = validContainer.createReference(i);

                if (slotStack.method_7960() || !AccessoryRegistry.canUnequip(slotStack, slotReference)) continue;

                if (stack.method_7960() || (SlotPredicateRegistry.canInsertIntoSlot(stack, slotReference) && extraCheck.isValid(slotStack, true))) {
                    return Pair.of(slotReference, (newStack) -> setStack(slotReference, newStack, true));
                }
            }
        }

        return null;
    }

    private Optional<class_1799> setStack(SlotReference reference, class_1799 newStack, boolean shouldSwapStacks) {
        var oldStack = reference.getStack();

        if (oldStack == null) {
            oldStack = class_1799.field_8037;
        } else {
            oldStack = oldStack.method_7972();
        }

        var accessory = AccessoryRegistry.getAccessoryOrDefault(oldStack);

        if(shouldSwapStacks) {
            var splitStack = newStack.method_7960() ? class_1799.field_8037 : newStack.method_7971(accessory.maxStackSize(newStack));

            if (!entity.method_73183().method_8608()) {
                reference.setStack(splitStack);
            }

            return Optional.of(oldStack);
        } else {
            if (!entity.method_73183().method_8608()) {
                var splitStack = newStack.method_7971(accessory.maxStackSize(newStack));

                reference.setStack(splitStack);
            }

            return Optional.empty();
        }
    }

    //--

    @Override
    public void encode(MapCarrierEncodable carrier, SerializationContext ctx) {
        AccessoriesHolderImpl.getHolder(this).encode(carrier, ctx);
    }

    @Override
    public void decode(MapCarrierDecodable carrier, SerializationContext ctx) {
        AccessoriesHolderImpl.getHolder(this).decode(carrier, ctx);
    }
}
