package io.wispforest.accessories.impl.core;

import I;
import Z;
import com.mojang.logging.LogUtils;
import io.wispforest.accessories.api.AccessoriesCapability;
import io.wispforest.accessories.api.components.AccessoriesDataComponents;
import io.wispforest.accessories.api.core.Accessory;
import io.wispforest.accessories.api.core.AccessoryRegistry;
import io.wispforest.accessories.impl.caching.AccessoriesHolderLookupCache;
import io.wispforest.accessories.utils.BaseContainer;
import io.wispforest.accessories.utils.ImmutableContainer;
import io.wispforest.accessories.utils.ItemStackMutation;
import io.wispforest.accessories.utils.ItemStackResize;
import io.wispforest.owo.util.EventSource;
import io.wispforest.owo.util.EventSource.Subscription;
import it.unimi.dsi.fastutil.ints.Int2BooleanArrayMap;
import it.unimi.dsi.fastutil.ints.Int2BooleanMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_11343;
import net.minecraft.class_1263;
import net.minecraft.class_1799;
import net.minecraft.class_2371;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;


///
/// An implementation of [BaseContainer] with overall API designed for use with [AccessoriesContainerImpl]
/// with hooks for mutation or resizing checks on stacks combined with a copy of the `previousItems` to
/// use later for checks of changes/used to rollback/remove effects from an entity.
///
public class ExpandedContainer extends BaseContainer {

    private static final Logger LOGGER = LogUtils.getLogger();

    private final AccessoriesContainerImpl container;

    private final String name;

    private final class_2371<class_1799> previousItems;

    private final Int2BooleanMap setFlags = new Int2BooleanArrayMap();

    private boolean canFlagSetCalls = true;

    private boolean newlyConstructed;

    private final Int2ObjectMap<EventSource.Subscription> currentMutationSubscriptions = new Int2ObjectOpenHashMap<>();
    private final Int2ObjectMap<EventSource.Subscription> currentResizeSubscriptions = new Int2ObjectOpenHashMap<>();

    private final Int2ObjectMap<EventSource.Subscription> currentPrevResizeSubscriptions = new Int2ObjectOpenHashMap<>();

    public ExpandedContainer(AccessoriesContainerImpl container, int size, String name) {
        this(container, size, name, true);
    }

    public ExpandedContainer(AccessoriesContainerImpl container, int size, String name, boolean toggleNewlyConstructed) {
        super(size);

        this.container = container;

        this.addListener(container);

        if(toggleNewlyConstructed) this.newlyConstructed = true;

        this.name = name;

        this.previousItems = class_2371.method_10213(size, class_1799.field_8037);
    }

    public String name() {
        return this.name;
    }

    public class_1263 toImmutable() {
        return new ImmutableContainer(this.getItems());
    }

    //--

    public boolean wasNewlyConstructed() {
        var bl = newlyConstructed;

        this.newlyConstructed = false;

        return bl;
    }

    public boolean isSlotFlagged(int slot){
        var bl = setFlags.getOrDefault(slot, false);

        if(bl) setFlags.put(slot, false);

        return bl;
    }

    void toggleFlagablity() {
        canFlagSetCalls = !canFlagSetCalls;
    }

    private void removeAllSlotSubscription(int slot) {
        removeMutationSubscription(slot);
        removeResizeSubscription(slot);

        removePrevResizeSubscription(slot);
    }

    private void removePrevResizeSubscription(int slot) {
        var subscription = currentPrevResizeSubscriptions.remove(slot);

        if (subscription != null) subscription.cancel();
    }

    private void removeResizeSubscription(int slot) {
        var subscription = currentResizeSubscriptions.remove(slot);

        if (subscription != null) subscription.cancel();
    }

    private void removeMutationSubscription(int slot) {
        var subscription = currentMutationSubscriptions.remove(slot);

        if (subscription != null) subscription.cancel();
    }

    public void setPreviousItem(int slot, class_1799 stack) {
        if(slot >= 0 && slot < this.previousItems.size()) {
            this.previousItems.set(slot, stack);

            removePrevResizeSubscription(slot);

            if (!stack.method_7960()) {
                /*
                    TODO: MAY NEED TO DEAL WITH THIS BETTER I.E. CALLING UNEQUIP BEFORE OR SOMETHING BUT IDK
                    LIKE THIS ISSUE IS DOWN TO THE FACT THAT THE COUNT CAN BE ADJUSTED WITHOUT NOTIFYING THE CONTAINER OF THE CHANGE
                    MEANING THE REFERENCE WILL BE EMPTY LEADING TO NO UNEQUIP CALL BUT IT MEANS THE STACK DOSE
                    NOT HAVE THE CORRECT STACK IF IT WAS TRANSFERRED TO ANOTHER STACK
                 */
                var stackCopy = stack.method_7972();

                this.currentPrevResizeSubscriptions.put(slot, ItemStackResize.getEvent(stack).source().subscribe((stack1, prevSize) -> {
                    var isEmpty = stack1.method_7947() <= 0;

                    if (isEmpty) {
                        this.previousItems.set(slot, stackCopy);

                        removePrevResizeSubscription(slot);
                    }
                }));
            }
        }
    }

    public class_1799 getPreviousItem(int slot) {
        return slot >= 0 && slot < this.previousItems.size()
                ? this.previousItems.get(slot)
                : class_1799.field_8037;
    }

    //--

    @Override
    public int method_58350(class_1799 itemStack) {
        var accessory = AccessoryRegistry.getAccessoryOrDefault(itemStack);

        return Math.min(super.method_58350(itemStack), accessory.maxStackSize(itemStack));
    }

    @Override
    public class_1799 method_5438(int slot) {
        if(!validIndex(slot)) return class_1799.field_8037;

        return super.method_5438(slot);
    }

    @Override
    public class_1799 method_5434(int slot, int amount) {
        if(!validIndex(slot)) return class_1799.field_8037;

        var stack = super.method_5434(slot, amount);

        if (!stack.method_7960()) {
            if (canFlagSetCalls) setFlags.put(slot, true);

            var prevStack = this.method_5438(slot);

            if (prevStack.method_7960()) {
                removeMutationSubscription(slot);
                removeResizeSubscription(slot);
            }

            this.setPreviousItem(slot, stack);
        }

        return stack;
    }

    @Override
    public class_1799 method_5441(int slot) {
        if(!validIndex(slot)) return class_1799.field_8037;

        // TODO: Concerning the flagging system, should this work for it?

        var stack = super.method_5441(slot);

        removeMutationSubscription(slot);
        removeResizeSubscription(slot);

        return stack;
    }

    @Override
    public void method_5447(int slot, class_1799 stack) {
        if(!validIndex(slot)) return;

        removeMutationSubscription(slot);
        removeResizeSubscription(slot);

        super.method_5447(slot, stack);

        if (!stack.method_7960()) {
            this.currentMutationSubscriptions.put(slot,
                    ItemStackMutation.getEvent(stack).source().subscribe((stack1, types) -> {
                        if (types.contains(AccessoriesDataComponents.ATTRIBUTES) || types.contains(AccessoriesDataComponents.NESTED_ACCESSORIES)) {
                            this.method_5431();
                        }

                        if (!this.container.capability().entity().method_73183().method_8608()) {
                            var cache = AccessoriesHolderImpl.getHolder(this.container.capability()).getLookupCache();

                            if (cache != null) cache.invalidateLookupData(this.container.getSlotName(), stack1, types);
                        }
                    })
            );

            this.currentResizeSubscriptions.put(slot,
                    ItemStackResize.getEvent(stack).source().subscribe((stack1, prevSize) -> {
                        if (stack1.method_7960()) {
                            this.method_5447(slot, class_1799.field_8037);
                        }
                    })
            );
        }

        if (canFlagSetCalls) setFlags.put(slot, true);
    }

    // Simple validation method to make sure that the given access is valid before attempting an operation
    public boolean validIndex(int slot){
        var isValid = slot >= 0 && slot < this.method_5439();

        if(!isValid && FabricLoader.getInstance().isDevelopmentEnvironment()){
            var nameInfo = (this.name != null ? "Container: " + this.name + ", " : "");

            try {
                throw new IllegalStateException("Access to a given Inventory was found to be out of the range valid for the container! [Name: " + nameInfo + " Index: " + slot + "]");
            } catch (Exception e) {
                LOGGER.debug("Full Exception: ", e);
            }
        }

        return isValid;
    }

    //--


    @Override
    public void loadItemsFromList(Collection<class_11343> slottedStacks) {
        this.container.containerListenerLock = true;

        var capability = this.container.capability();

        var prevStacks = new ArrayList<class_1799>();
        for(int i = 0; i < this.method_5439(); ++i) {
            var currentStack = this.method_5438(i);

            prevStacks.add(currentStack);

            this.method_5447(i, class_1799.field_8037);
        }

        var invalidStacks = new ArrayList<class_1799>();
        var decodedStacks = new ArrayList<class_1799>();

        for (var slottedStack : slottedStacks) {
            var stack = slottedStack.comp_4212();

            decodedStacks.add(stack);

            if (slottedStack.method_71368(this.method_5439())) {
                this.method_5447(slottedStack.comp_4211(), stack);
            } else {
                invalidStacks.add(stack);
            }
        }

        this.container.containerListenerLock = false;

        if (!capability.entity().method_73183().method_8608()) {
            if (!prevStacks.equals(decodedStacks)) {
                this.method_5431();
            }

            AccessoriesHolderImpl.getHolder(capability).invalidStacks.addAll(invalidStacks);
        }
    }

    //--

    @Override
    public Iterator<class_1799> iterator() {
        return new Iterator<>() {
            private int index = 0;

            @Override
            public boolean hasNext() {
                return index < ExpandedContainer.this.method_5439();
            }

            @Override
            public class_1799 next() {
                var stack = ExpandedContainer.this.method_5438(index);

                index++;

                return stack;
            }
        };
    }

    public void foreach(BiConsumer<Integer, class_1799> consumer) {
        var i = 0;

        for (class_1799 itemStack : this) {
            consumer.accept(i, itemStack);
            i++;
        }
    }

    public <T> T foreach(BiFunction<Integer, class_1799, @Nullable T> consumer) {
        var i = 0;

        for (class_1799 itemStack : this) {
            var result = consumer.apply(i, itemStack);

            if (result != null) return null;

            i++;
        }

        return null;
    }

    public void setFromPrev(ExpandedContainer prevContainer) {
        int i = 0;

        for (var itemStack : prevContainer) {
            prevContainer.removeMutationSubscription(i);
            this.setPreviousItem(i, itemStack);
            i++;
        }
    }

    public void copyPrev(ExpandedContainer prevContainer) {
        for (int i = 0; i < prevContainer.method_5439(); i++) {
            if(i >= this.method_5439()) continue;

            var prevItem = prevContainer.getPreviousItem(i);

            prevContainer.removeAllSlotSubscription(i);

            if(!prevItem.method_7960()) this.setPreviousItem(i, prevItem);
        }
    }
}
