package io.wispforest.accessories.impl.event;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import com.google.common.collect.UnmodifiableIterator;
import io.wispforest.accessories.Accessories;
import io.wispforest.accessories.AccessoriesClientInternals;
import io.wispforest.accessories.AccessoriesInternals;
import io.wispforest.accessories.api.AccessoriesCapability;
import io.wispforest.accessories.api.AccessoriesContainer;
import io.wispforest.accessories.api.attributes.AccessoryAttributeBuilder;
import io.wispforest.accessories.api.components.AccessoriesDataComponents;
import io.wispforest.accessories.api.components.AccessoryItemAttributeModifiers;
import io.wispforest.accessories.api.components.AccessoryItemAttributeModifiers.Builder;
import io.wispforest.accessories.api.components.AccessoryMobEffectsComponent;
import io.wispforest.accessories.api.components.AccessoryNestContainerContents;
import io.wispforest.accessories.api.core.Accessory;
import io.wispforest.accessories.api.core.AccessoryNest;
import io.wispforest.accessories.api.core.AccessoryRegistry;
import io.wispforest.accessories.api.data.AccessoriesTags;
import io.wispforest.accessories.api.events.*;
import io.wispforest.accessories.api.slot.*;
import io.wispforest.accessories.data.EntitySlotLoader;
import io.wispforest.accessories.data.SlotTypeLoader;
import io.wispforest.accessories.endec.NbtMapCarrier;
import io.wispforest.accessories.impl.AccessoryAttributeLogic;
import io.wispforest.accessories.api.core.AccessoryNestUtils;
import io.wispforest.accessories.impl.PlayerEquipControl;
import io.wispforest.accessories.impl.core.AccessoriesCapabilityImpl;
import io.wispforest.accessories.impl.core.AccessoriesContainerImpl;
import io.wispforest.accessories.impl.core.AccessoriesHolderImpl;
import io.wispforest.accessories.impl.core.ExpandedContainer;
import io.wispforest.accessories.impl.option.AccessoriesPlayerOptionsHolder;
import io.wispforest.accessories.impl.option.PlayerOptions;
import io.wispforest.accessories.impl.slot.ExtraSlotTypeProperties;
import io.wispforest.accessories.menu.variants.AccessoriesMenuBase;
import io.wispforest.accessories.misc.AccessoriesGameRules;
import io.wispforest.accessories.networking.AccessoriesNetworking;
import io.wispforest.accessories.networking.client.SyncContainerData;
import io.wispforest.accessories.networking.client.SyncEntireContainer;
import io.wispforest.accessories.networking.client.SyncPlayerOptions;
import io.wispforest.accessories.pond.AccessoriesLivingEntityExtension;
import io.wispforest.accessories.utils.AttributeUtils;
import io.wispforest.endec.SerializationContext;
import io.wispforest.owo.serialization.RegistriesAttribute;
import it.unimi.dsi.fastutil.Pair;
import net.fabricmc.fabric.api.util.TriState;
import net.minecraft.ChatFormatting;
import net.minecraft.core.Holder;
import net.minecraft.core.component.DataComponentType;
import net.minecraft.network.chat.CommonComponents;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.server.players.PlayerList;
import net.minecraft.world.Container;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.ai.attributes.Attribute;
import net.minecraft.world.entity.ai.attributes.AttributeModifier;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.TooltipFlag;
import net.minecraft.world.item.component.TooltipDisplay;
import net.minecraft.world.item.enchantment.EnchantmentEffectComponents;
import net.minecraft.world.item.enchantment.EnchantmentHelper;
import net.minecraft.world.level.GameRules;
import net.minecraft.world.level.Level;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import static io.wispforest.accessories.Accessories.ACCESSORY_EQUIPPED;
import static io.wispforest.accessories.Accessories.ACCESSORY_UNEQUIPPED;

import I;
import Z;

@ApiStatus.Internal
public class AccessoriesEventHandler {

    public static boolean dataReloadOccurred = false;

    public static void onWorldTick(Level level) {
        if (!(level instanceof ServerLevel serverLevel)) return;

        revalidatePlayersOnReload(serverLevel.getServer().getPlayerList());
    }

    public static void revalidatePlayersOnReload(PlayerList playerList) {
        if (!dataReloadOccurred) return;

        for (var player : playerList.getPlayers()) revalidatePlayer(player);

        dataReloadOccurred = false;
    }

    public static void revalidatePlayer(ServerPlayer player) {
        var capability = AccessoriesCapability.get(player);

        if (capability == null) return;

        var validSlotTypes = EntitySlotLoader.getEntitySlots(player).values();

        var holderImpl = AccessoriesHolderImpl.getHolder(capability);

        holderImpl.setValidTypes(validSlotTypes.stream().map(SlotType::name).collect(Collectors.toSet()));

        for (var container : holderImpl.getAllSlotContainers().values()) {
            var slotType = container.slotType();

            if (slotType != null && validSlotTypes.contains(slotType)) {
                var baseSize = ((AccessoriesContainerImpl) container).getBaseSize();

                if (baseSize == null || baseSize != slotType.amount()) {
                    container.markChanged();
                    container.update();
                }

                var stacks = container.getAccessories();
                var cosmeticStacks = container.getCosmeticAccessories();

                for (int i = 0; i < container.getSize(); i++) {
                    var reference = container.createReference(i);

                    handleInvalidStacks(stacks, reference, player);
                    handleInvalidStacks(cosmeticStacks, reference, player);
                }
            } else {
                // TODO: DROP CONTAINER ?!
                var stacks = container.getAccessories();
                var cosmeticStacks = container.getCosmeticAccessories();

                for (int i = 0; i < container.getSize(); i++) {
                    var reference = container.createReference(i);

                    dropAndRemoveStack(stacks, reference, player);
                    dropAndRemoveStack(cosmeticStacks, reference, player);
                }
            }
        }
    }

    private static void handleInvalidStacks(Container container, SlotReference reference, ServerPlayer player) {
        var stack = container.getItem(reference.index());

        if (stack.isEmpty()) return;

        var bl = !SlotPredicateRegistry.canInsertIntoSlot(stack, reference);

        if (bl) dropAndRemoveStack(container, reference, player);
    }

    private static void dropAndRemoveStack(Container container, SlotReference reference, ServerPlayer player) {
        var stack = container.getItem(reference.index());

        container.setItem(reference.index(), ItemStack.EMPTY);

        AccessoriesInternals.INSTANCE.giveItemToPlayer(player, stack);
    }

    public static void entityLoad(LivingEntity entity, Level level) {
        if (!level.isClientSide() || !(entity instanceof ServerPlayer serverPlayer)) return;

        SyncEntireContainer.syncToAllTrackingAndSelf(serverPlayer);
    }

    public static void onTracking(LivingEntity entity, ServerPlayer serverPlayer) {
        SyncEntireContainer.syncTo(entity, (packet) -> AccessoriesNetworking.sendToPlayer(serverPlayer, packet));
    }

    public static void dataSync(@Nullable PlayerList list, @Nullable ServerPlayer player) {
        if (list != null && !list.getPlayers().isEmpty()) {
            revalidatePlayersOnReload(list);

            // TODO: OPTIMIZE THIS?
            for (var playerEntry : list.getPlayers()) {
                var capability = AccessoriesCapability.get(playerEntry);

                if (capability == null) return;

                var carrier = NbtMapCarrier.of();

                AccessoriesHolderImpl.getHolder(capability).encode(carrier, SerializationContext.attributes(RegistriesAttribute.of(playerEntry.level().registryAccess())));

                AccessoriesNetworking.sendToTrackingAndSelf(playerEntry, new SyncEntireContainer(capability.entity().getId(), carrier));

                if (playerEntry.containerMenu instanceof AccessoriesMenuBase base) {
                    Accessories.openAccessoriesMenu(playerEntry, base.menuVariant(), base.targetEntity());
                }
            }
        } else if (player != null) {
            revalidatePlayer(player);

            var capability = AccessoriesCapability.get(player);

            if (capability == null) return;

            var carrier = NbtMapCarrier.of();

            AccessoriesHolderImpl.getHolder(capability).encode(carrier, SerializationContext.attributes(RegistriesAttribute.of(player.level().registryAccess())));

            AccessoriesNetworking.sendToPlayer(player, new SyncEntireContainer(capability.entity().getId(), carrier));

            AccessoriesNetworking.sendToPlayer(player, new SyncPlayerOptions(AccessoriesPlayerOptionsHolder.getOptions(player)));

            if (player.containerMenu instanceof AccessoriesMenuBase base) {
                Accessories.openAccessoriesMenu(player, base.menuVariant(), base.targetEntity());
            }
        }

        AccessoriesHolderImpl.clearValidationCache(false);
    }

    public static void onLivingEntityTick(LivingEntity entity) {
        if (entity.isRemoved()) return;

        var capability = AccessoriesCapability.get(entity);

        if (capability != null) {
            var dirtyStacks = new HashMap<String, ItemStack>();
            var dirtyCosmeticStacks = new HashMap<String, ItemStack>();

            var removedAttributesBuilder = new AccessoryAttributeBuilder();
            var addedAttributesBuilder = new AccessoryAttributeBuilder();

            for (var containerEntry : AccessoriesHolderImpl.getHolder(capability).getAllSlotContainers().entrySet()) {
                var container = containerEntry.getValue();

                var accessories = container.getAccessories();
                var cosmetics = container.getCosmeticAccessories();

                for (int i = 0; i < accessories.getContainerSize(); i++) {
                    var slotReference = container.createReference(i);

                    var slotId = container.getSlotName() + "/" + i;

                    var currentStack = accessories.getItem(i);

                    // TODO: Move ticking below checks?
                    if (!currentStack.isEmpty()) {
                        // TODO: Document this behavior to prevent double ticking maybe!!!
                        currentStack.inventoryTick(entity.level(), entity, null);

                        var accessory = AccessoryRegistry.getAccessoryOrDefault(currentStack);

                        if (accessory != null) {
                            accessory.tick(currentStack, slotReference);

                            AccessoryNestUtils.recursivelyConsume(currentStack, stack -> {
                                var effects = stack.get(AccessoriesDataComponents.MOB_EFFECTS);

                                if (effects != null) {
                                    effects.handleReapplyingEffects(entity, entity.level().getGameTime());
                                }
                            });
                        }
                    }

                    var lastStack = accessories.getPreviousItem(i);
                    var flagged = accessories.isSlotFlagged(i);

                    // Prevent attribute related logic on the client and if the entity
                    // is dead as such data should not be updated. Though we allow for
                    // ticking to occur at least for vanilla parity I guess.
                    if (entity.level().isClientSide() || entity.isDeadOrDying()) continue;

                    if (!ItemStack.matches(currentStack, lastStack) || flagged) {
                        if (!lastStack.isEmpty()) {
                            var removedEnchantmentBuilder = new AccessoryAttributeBuilder(slotReference);

                            // TODO: MAYBE MOVE THIS TO AccessoryAttributeLogic or something
                            EnchantmentHelper.forEachModifier(lastStack, AccessoriesInternals.INSTANCE.getInternalEquipmentSlot(), (attributeHolder, modifier) -> {
                                var namespace = modifier.id().getNamespace();
                                var splitPath = new ArrayList<>(List.of(modifier.id().getPath().split("/")));

                                splitPath.removeLast();

                                removedEnchantmentBuilder.addStackable(attributeHolder, new AttributeModifier(ResourceLocation.fromNamespaceAndPath(namespace, String.join("/", splitPath)), modifier.amount(), modifier.operation()));
                            });

                            removedAttributesBuilder.addFrom(removedEnchantmentBuilder);
                            removedAttributesBuilder.addFrom(AccessoryAttributeLogic.getAttributeModifiers(lastStack, slotReference));

                            ((AccessoriesLivingEntityExtension) entity).pushEnchantmentContext(lastStack, slotReference);
                            EnchantmentHelper.stopLocationBasedEffects(lastStack, entity, AccessoriesInternals.INSTANCE.getInternalEquipmentSlot());
                        }

                        if (!currentStack.isEmpty()) {
                            var addedEnchantmentBuilder = new AccessoryAttributeBuilder(slotReference);

                            EnchantmentHelper.forEachModifier(currentStack, AccessoriesInternals.INSTANCE.getInternalEquipmentSlot(), (attributeHolder, modifier) -> {
                                var namespace = modifier.id().getNamespace();
                                var splitPath = new ArrayList<>(List.of(modifier.id().getPath().split("/")));

                                splitPath.removeLast();

                                addedEnchantmentBuilder.addStackable(attributeHolder, new AttributeModifier(ResourceLocation.fromNamespaceAndPath(namespace, String.join("/", splitPath)), modifier.amount(), modifier.operation()));
                            });

                            addedAttributesBuilder.addFrom(addedEnchantmentBuilder);
                            addedAttributesBuilder.addFrom(AccessoryAttributeLogic.getAttributeModifiers(currentStack, slotReference));

                            ((AccessoriesLivingEntityExtension) entity).pushEnchantmentContext(currentStack, slotReference);
                            EnchantmentHelper.runLocationChangedEffects((ServerLevel) entity.level(), currentStack, entity, AccessoriesInternals.INSTANCE.getInternalEquipmentSlot());
                        }

                        AccessoryNestUtils.recursivelyConsume(lastStack, stack -> {
                            if (stack.has(AccessoriesDataComponents.MOB_EFFECTS)) {
                                stack.get(AccessoriesDataComponents.MOB_EFFECTS)
                                    .handleRemovingEffects(entity);
                            }
                        });

                        AccessoryNestUtils.recursivelyConsume(currentStack, stack -> {
                            if (stack.has(AccessoriesDataComponents.MOB_EFFECTS)) {
                                stack.get(AccessoriesDataComponents.MOB_EFFECTS)
                                    .handleApplyingConstantEffects(entity);
                            }
                        });

                        boolean equipmentChange = false;

                        /*
                         * TODO: Does item check need to exist anymore?
                         */
                        if (!ItemStack.isSameItem(currentStack, lastStack) || flagged) {
                            AccessoryRegistry.getAccessoryOrDefault(lastStack).onUnequip(lastStack, slotReference);
                            AccessoryRegistry.getAccessoryOrDefault(currentStack).onEquip(currentStack, slotReference);

                            if (entity instanceof ServerPlayer serverPlayer) {
                                if (!currentStack.isEmpty()) {
                                    ACCESSORY_EQUIPPED.trigger(serverPlayer, currentStack, slotReference, false);
                                }

                                if (!lastStack.isEmpty()) {
                                    ACCESSORY_UNEQUIPPED.trigger(serverPlayer, lastStack, slotReference, false);
                                }
                            }

                            equipmentChange = true;
                        }

                        AccessoryChangeCallback.EVENT.invoker().onChange(lastStack, currentStack, slotReference, equipmentChange ? SlotStateChange.REPLACEMENT : SlotStateChange.MUTATION);

                        container.getAccessories().setPreviousItem(i, currentStack);
                        dirtyStacks.put(slotId, currentStack.copy());

                        recursiveStackChange(slotReference, AccessoryNestUtils.getData(lastStack), AccessoryNestUtils.getData(currentStack));
                    }

                    var currentCosmeticStack = cosmetics.getItem(i);
                    var lastCosmeticStack = container.getCosmeticAccessories().getPreviousItem(i);

                    if (!ItemStack.matches(currentCosmeticStack, lastCosmeticStack)) {
                        cosmetics.setPreviousItem(i, currentCosmeticStack);
                        dirtyCosmeticStacks.put(slotId, currentCosmeticStack.copy());

                        if (entity instanceof ServerPlayer serverPlayer) {
                            if (!currentStack.isEmpty()) {
                                ACCESSORY_EQUIPPED.trigger(serverPlayer, currentStack, slotReference, true);
                            }
                            if (!lastStack.isEmpty()) {
                                ACCESSORY_UNEQUIPPED.trigger(serverPlayer, lastStack, slotReference, true);
                            }
                        }
                    }
                }
            }

            if (entity.level().isClientSide()) return;

            AttributeUtils.removeTransientAttributeModifiers(entity, removedAttributesBuilder);
            AttributeUtils.addTransientAttributeModifiers(entity, addedAttributesBuilder);

            //--
            var updatedContainers = AccessoriesHolderImpl.getHolder(capability).containersRequiringUpdates();

            capability.updateContainers();

            ContainersChangeCallback.EVENT.invoker().onChange(entity, capability, ImmutableMap.copyOf(updatedContainers));

            if (!dirtyStacks.isEmpty() || !dirtyCosmeticStacks.isEmpty() || !updatedContainers.isEmpty()) {
                var packet = SyncContainerData.of(entity, updatedContainers.keySet(), dirtyStacks, dirtyCosmeticStacks);

                AccessoriesNetworking.sendToTrackingAndSelf(entity, packet);
            }

            updatedContainers.clear();
        }

        //--

        var holder = ((AccessoriesHolderImpl) AccessoriesInternals.INSTANCE.getHolder(entity));

        // Fix for holder data not being loaded so invalid stacks can be collected
        if (holder.loadedFromTag() && capability == null) {
            var tempCapability = new AccessoriesCapabilityImpl(entity);
        }

        var invalidStacks = (holder).invalidStacks;

        if (!invalidStacks.isEmpty() && entity.level() instanceof ServerLevel serverLevel) {
            for (ItemStack invalidStack : invalidStacks) {
                if (entity instanceof ServerPlayer serverPlayer) {
                    AccessoriesInternals.INSTANCE.giveItemToPlayer(serverPlayer, invalidStack);
                } else {
                    entity.spawnAtLocation(serverLevel, invalidStack);
                }
            }

            invalidStacks.clear();
        }
    }

    private static void recursiveStackChange(SlotReference slotReference, @Nullable AccessoryNestContainerContents lastNestData, @Nullable AccessoryNestContainerContents currentNestData) {
        var currentNestChanges = (currentNestData != null)
                ? currentNestData.slotChanges()
                : new HashMap<Integer, SlotStateChange>();

        var lastInnerStacks = lastNestData != null ? List.copyOf(lastNestData.getMap(slotReference).entrySet()) : List.<Map.Entry<SlotEntryReference, Accessory>>of();
        var currentInnerStacks = currentNestData != null ? List.copyOf(currentNestData.getMap(slotReference).entrySet()) : List.<Map.Entry<SlotEntryReference, Accessory>>of();

        var maxIterationLength = Math.max(lastInnerStacks.size(), currentInnerStacks.size());

        for (int i = 0; i < maxIterationLength; i++) {
            var lastInnerEntry = (i < lastInnerStacks.size()) ? lastInnerStacks.get(i) : null;
            var currentInnerEntry = (i < currentInnerStacks.size()) ? currentInnerStacks.get(i) : null;

            var changeType = currentNestChanges.getOrDefault(i, SlotStateChange.REPLACEMENT);

            if (lastInnerEntry == null && currentInnerEntry != null) {
                var currentRef = currentInnerEntry.getKey();
                var currentInnerStack = currentRef.stack();

                onStackChange(currentRef.reference(), ItemStack.EMPTY, currentRef.stack(), changeType);

                recursiveStackChange(slotReference, null, AccessoryNestUtils.getData(currentInnerStack));
            } else if (currentInnerEntry == null && lastInnerEntry != null) {
                var lastRef = lastInnerEntry.getKey();
                var lastInnerStack = lastRef.stack();

                onStackChange(lastRef.reference(), lastRef.stack(), ItemStack.EMPTY, changeType);

                recursiveStackChange(slotReference, AccessoryNestUtils.getData(lastInnerStack), null);
            } else if (lastInnerEntry != null && currentInnerEntry != null) {
                var currentRef = currentInnerEntry.getKey();
                var lastRef = lastInnerEntry.getKey();

                var innerRef = lastRef.reference();

                var currentInnerStack = currentRef.stack();
                var lastInnerStack = lastRef.stack();

                onStackChange(innerRef, lastInnerStack, currentInnerStack, changeType);

                recursiveStackChange(slotReference, AccessoryNestUtils.getData(lastInnerStack), AccessoryNestUtils.getData(currentInnerStack));
            }
        }

        currentNestChanges.clear();
    }

    private static void onStackChange(SlotReference slotReference, ItemStack lastStack, ItemStack currentStack, SlotStateChange stateChange) {
        if (slotReference.entity() instanceof ServerPlayer serverPlayer) {
            if (!currentStack.isEmpty()) {
                ACCESSORY_EQUIPPED.trigger(serverPlayer, currentStack, slotReference, false);
            }

            if (!lastStack.isEmpty()) {
                ACCESSORY_UNEQUIPPED.trigger(serverPlayer, lastStack, slotReference, false);
            }
        }

        AccessoryChangeCallback.EVENT.invoker().onChange(lastStack, currentStack, slotReference, stateChange);
    }

    public static void getTooltipData(@Nullable LivingEntity entity, ItemStack stack, List<Component> tooltip,TooltipDisplay display, Item.TooltipContext tooltipContext, TooltipFlag tooltipType) {
        var accessory = AccessoryRegistry.getAccessoryOrDefault(stack);

        if (accessory != null) {
            // Add possible client values to tooltipFlag
            tooltipType = AccessoriesClientInternals.getInstance().createTooltipFlag(tooltipType);

            if (entity != null && AccessoriesCapability.get(entity) != null)
                addEntityBasedTooltipData(entity, accessory, stack, tooltip, display, tooltipContext, tooltipType);

            accessory.getExtraTooltip(stack, tooltip, tooltipContext, tooltipType);
        }
    }

    // TODO: Rewrite for better handling of various odd cases
    private static void addEntityBasedTooltipData(LivingEntity entity, Accessory accessory, ItemStack stack, List<Component> tooltip, TooltipDisplay display, Item.TooltipContext tooltipContext, TooltipFlag tooltipType) {
        // TODO: MAYBE DEPENDING ON ENTITY OR SOMETHING SHOW ALL VALID SLOTS BUT COLOR CODE THEM IF NOT VALID FOR ENTITY?
        // TODO: ADD BETTER HANDLING FOR POSSIBLE SLOTS THAT ARE EQUIPABLE IN BUT IS AT ZERO SIZE
        var validSlotTypes = new HashSet<>(SlotPredicateRegistry.getValidSlotTypes(entity, stack));

        if (validSlotTypes.isEmpty()) return;

        {
            final var validUniqueSlots = new HashSet<SlotType>();

            validSlotTypes.removeIf(slotType -> {
                var isUnique = UniqueSlotHandling.isUniqueSlot(slotType.name());

                if(isUnique) validUniqueSlots.add(slotType);

                return isUnique;
            });

            var sharedSlotTypes = SlotTypeLoader.INSTANCE.getEntries(entity.level()).values()
                    .stream()
                    .filter(slotType -> !UniqueSlotHandling.isUniqueSlot(slotType.name()))
                    .collect(Collectors.toSet());

            var slotInfoComponent = Component.literal("");

            var slotsComponent = Component.literal("");
            boolean allSlots = false;

            if (validSlotTypes.containsAll(sharedSlotTypes)) {
                slotsComponent.append(Component.translatable(Accessories.translationKey("slot.any")));
                allSlots = true;
            } else {
                var entitySlotTypes = Set.copyOf(EntitySlotLoader.getEntitySlots(entity).values());

                var invalidSlotsTypes = Sets.difference(entitySlotTypes, validSlotTypes);

                if (invalidSlotsTypes.size() < validSlotTypes.size()) {
                    slotsComponent.append(Component.translatable(Accessories.translationKey("slot.any")));
                    slotsComponent.append(Component.translatable(Accessories.translationKey("slot.except")).withStyle(ChatFormatting.GRAY));

                    var invalidSlotsItr = invalidSlotsTypes.iterator();

                    while(invalidSlotsItr.hasNext()) {
                        var type = invalidSlotsItr.next();

                        if (ExtraSlotTypeProperties.getProperty(type.name(), entity.level().isClientSide()).allowTooltipInfo()) {
                            slotsComponent.append(Component.translatable(type.translation()).withStyle(ChatFormatting.RED));

                            if (invalidSlotsItr.hasNext()) {
                                slotsComponent.append(Component.literal(", ").withStyle(ChatFormatting.GRAY));
                            }
                        }
                    }
                } else {
                    var validSlotsItr = validSlotTypes.iterator();

                    while(validSlotsItr.hasNext()) {
                        var type = validSlotsItr.next();

                        if (ExtraSlotTypeProperties.getProperty(type.name(), entity.level().isClientSide()).allowTooltipInfo()) {
                            slotsComponent.append(Component.translatable(type.translation()));

                            if (validSlotsItr.hasNext()) {
                                slotsComponent.append(Component.literal(", ").withStyle(ChatFormatting.GRAY));
                            }
                        }
                    }
                }
            }

            validSlotTypes.addAll (validUniqueSlots);

            final var filteredValidUniqueSlots = validUniqueSlots.stream()
                    .filter(slotType -> ExtraSlotTypeProperties.getProperty(slotType.name(), true).allowTooltipInfo())
                    .toList();

            if (!filteredValidUniqueSlots.isEmpty()) {
                var uniqueItr = filteredValidUniqueSlots.iterator();

                while(uniqueItr.hasNext()) {
                    var type = uniqueItr.next();

                    slotsComponent.append(Component.translatable(type.translation()));

                    if(uniqueItr.hasNext()) {
                        slotsComponent.append(Component.literal(", ").withStyle(ChatFormatting.GRAY));
                    }
                }
            }

            if(!slotsComponent.getSiblings().isEmpty()) {
                var slotTranslationKey = "slot.tooltip." + ((validSlotTypes.size() > 1 && !allSlots) ? "plural" : "singular");

                slotInfoComponent.append(
                        Component.translatable(Accessories.translationKey(slotTranslationKey))
                                .withStyle(ChatFormatting.GRAY)
                                .append(slotsComponent.withStyle(ChatFormatting.BLUE))
                );

                tooltip.add(slotInfoComponent);
            }
        }

        var slotSpecificModifiers = new HashMap<SlotType, AccessoryAttributeBuilder>();
        AccessoryAttributeBuilder defaultModifiers = null;

        boolean allDuplicates = true;

        for (var slotType : validSlotTypes) {
            var reference = SlotReference.of(entity, slotType.name(), 0);

            var builder = AccessoryAttributeLogic.getAttributeModifiers(stack, reference, true);

            if (builder.isEmpty()) continue;

            slotSpecificModifiers.put(slotType, builder);

            if (defaultModifiers == null) {
                defaultModifiers = builder;
            } else if (allDuplicates) {
                // TODO: ! WARNING ! THIS MAY NOT WORK?
                allDuplicates = defaultModifiers.equalWithoutPaths(builder);
            }
        }

        var slotTypeToTooltipInfo = new HashMap<SlotType, List<Component>>();

        if (allDuplicates) {
            if (defaultModifiers != null && !defaultModifiers.isEmpty()) {
                var attributeTooltip = new ArrayList<Component>();

                addAttributeTooltip(entity, stack, defaultModifiers.getAttributeModifiers(false), attributeTooltip, display, tooltipContext, tooltipType);

                slotTypeToTooltipInfo.put(null, attributeTooltip);
            }
        } else {
            for (var slotModifiers : slotSpecificModifiers.entrySet()) {
                var slotType = slotModifiers.getKey();
                var modifiers = slotModifiers.getValue();

                var attributeTooltip = new ArrayList<Component>();

                addAttributeTooltip(entity, stack, modifiers.getAttributeModifiers(false), attributeTooltip, display, tooltipContext, tooltipType);

                slotTypeToTooltipInfo.put(slotType, attributeTooltip);
            }
        }

        var extraAttributeTooltips = new HashMap<SlotType, List<Component>>();
        List<Component> defaultExtraAttributeTooltip = null;

        boolean allDuplicatesExtras = true;

        for (var slotType : validSlotTypes) {
            var extraAttributeTooltip = new ArrayList<Component>();
            accessory.getAttributesTooltip(stack, slotType, extraAttributeTooltip, tooltipContext, tooltipType);

            extraAttributeTooltips.put(slotType, extraAttributeTooltip);

            if (defaultExtraAttributeTooltip == null) {
                defaultExtraAttributeTooltip = extraAttributeTooltip;
            } else if (allDuplicatesExtras) {
                allDuplicatesExtras = extraAttributeTooltip.equals(defaultExtraAttributeTooltip);
            }
        }

        if (allDuplicatesExtras) {
            if (defaultExtraAttributeTooltip != null) {
                slotTypeToTooltipInfo.computeIfAbsent(null, s -> new ArrayList<>())
                    .addAll(defaultExtraAttributeTooltip);
            }
        } else {
            extraAttributeTooltips.forEach((slotType, components) -> {
                slotTypeToTooltipInfo.computeIfAbsent(slotType, s -> new ArrayList<>())
                        .addAll(components);
            });
        }

        if (slotTypeToTooltipInfo.containsKey(null)) {
            var anyTooltipInfo = slotTypeToTooltipInfo.get(null);

            if (!anyTooltipInfo.isEmpty()) {
                tooltip.add(CommonComponents.EMPTY);

                tooltip.add(
                        Component.translatable(Accessories.translationKey("tooltip.attributes.any"))
                                .withStyle(ChatFormatting.GRAY)
                );

                tooltip.addAll(anyTooltipInfo);
            }

            slotTypeToTooltipInfo.remove(null);
        }

        if (!slotTypeToTooltipInfo.isEmpty()) {
            for (var entry : slotTypeToTooltipInfo.entrySet()) {
                var tooltipData = entry.getValue();

                if (tooltipData.isEmpty()) continue;

                tooltip.add(CommonComponents.EMPTY);

                tooltip.add(
                        Component.translatable(
                                Accessories.translationKey("tooltip.attributes.slot"),
                                Component.translatable(entry.getKey().translation()).withStyle(ChatFormatting.BLUE)
                        ).withStyle(ChatFormatting.GRAY)
                );

                tooltip.addAll(entry.getValue());
            }
        }
    }

    private static void addAttributeTooltip(LivingEntity entity, ItemStack stack, Multimap<Holder<Attribute>, AttributeModifier> multimap, List<Component> tooltip, TooltipDisplay display, Item.TooltipContext context, TooltipFlag flag) {
        if (multimap.isEmpty()) return;

        AccessoriesInternals.INSTANCE.addAttributeTooltips((entity instanceof Player player ? player : null), stack, multimap, tooltip::add, display, context, flag);
    }

    @Nullable
    public static Collection<ItemStack> onDeath(LivingEntity entity, DamageSource source) {
        var capability = AccessoriesCapability.get(entity);

        if (capability == null) return List.of();

        var droppedStacks = new ArrayList<ItemStack>();

        var gamerules = ((ServerLevel) entity.level()).getGameRules();

        var keepInv = gamerules.getRule(GameRules.RULE_KEEPINVENTORY).get() || gamerules.getRule(AccessoriesGameRules.RULE_KEEP_ACCESSORY_INVENTORY).get();

        for (var containerEntry : AccessoriesHolderImpl.getHolder(capability).getAllSlotContainers().entrySet()) {
            var slotType = containerEntry.getValue().slotType();

            var slotDropRule = slotType != null ? slotType.dropRule() : DropRule.DEFAULT;

            var container = containerEntry.getValue();

            var stacks = container.getAccessories();
            var cosmeticStacks = container.getCosmeticAccessories();

            for (int i = 0; i < container.getSize(); i++) {
                var reference = SlotReference.of(entity, container.getSlotName(), i);

                var stack = dropStack(slotDropRule, entity, stacks, reference, source, keepInv);
                if (stack != null) droppedStacks.add(stack);

                var cosmeticStack = dropStack(slotDropRule, entity, cosmeticStacks, reference, source, keepInv);
                if (cosmeticStack != null) droppedStacks.add(cosmeticStack);
            }
        }

        var result = OnDeathCallback.EVENT.invoker().shouldDrop(TriState.DEFAULT, entity, capability, source, droppedStacks);

        if (!result.orElse(true)) return null;

        return droppedStacks;
    }

    @Nullable
    private static ItemStack dropStack(DropRule dropRule, LivingEntity entity, ExpandedContainer container, SlotReference reference, DamageSource source, boolean keepInvEnabled) {
        var stack = container.getItem(reference.index());

        if (stack.isEmpty()) return null;

        var accessory = AccessoryRegistry.getAccessoryOrDefault(stack);

        if (accessory != null && dropRule == DropRule.DEFAULT) {
            dropRule = accessory.getDropRule(stack, reference, source);
        }

        if (accessory instanceof AccessoryNest holdable) {
            var dropRuleToStacks = holdable.getDropRules(stack, reference, source);

            for (int i = 0; i < dropRuleToStacks.size(); i++) {
                var rulePair = dropRuleToStacks.get(i);

                var innerStack = rulePair.right();

                var result = OnDropCallback.getAlternativeRule(rulePair.left(), innerStack, reference, source);

                var breakInnerStack = (result == DropRule.DEFAULT && EnchantmentHelper.has(innerStack, EnchantmentEffectComponents.PREVENT_EQUIPMENT_DROP))
                        || (result == DropRule.DESTROY);

                if (breakInnerStack) {
                    holdable.setInnerStack(stack, i, ItemStack.EMPTY);
                    // TODO: Do we call break here for the accessory?

                    container.setItem(reference.index(), stack);
                }
            }
        }

        var result = OnDropCallback.getAlternativeRule(dropRule, stack, reference, source);

        boolean dropStack = true;
        boolean keepingStack = false;

        if (result == DropRule.DESTROY) {
            container.setItem(reference.index(), ItemStack.EMPTY);
            dropStack = false;
            // TODO: Do we call break here for the accessory?
        } else if (result == DropRule.KEEP) {
            dropStack = false;
            keepingStack = true;
        } else if (result == DropRule.DEFAULT) {
            if (keepInvEnabled) {
                dropStack = false;

                keepingStack = true;
            } else if (EnchantmentHelper.has(stack, EnchantmentEffectComponents.PREVENT_EQUIPMENT_DROP)) {
                container.setItem(reference.index(), ItemStack.EMPTY);
                dropStack = false;
                // TODO: Do we call break here for the accessory?
            }
        }

        // Used to indicate within the Accessories system when the player becomes alive that we need to
        // equip the accessory again to trigger equip call and properly add back Attributes
        if (keepingStack) {
            container.setPreviousItem(reference.index(), ItemStack.EMPTY);
        }

        if (!dropStack) return null;

        container.setItem(reference.index(), ItemStack.EMPTY);

        return stack;
    }

    public static InteractionResult attemptEquipFromUse(Player player, InteractionHand hand) {
        var stack = player.getItemInHand(hand);

        var capability = AccessoriesCapability.get(player);

        if (capability != null && !player.isSpectator() && !stack.isEmpty()) {
            var equipControl = AccessoriesPlayerOptionsHolder.getOptions(player).getDefaultedData(PlayerOptions.EQUIP_CONTROL);

            var shouldAttemptEquip = false;

            if (equipControl == PlayerEquipControl.MUST_CROUCH && player.isShiftKeyDown()) {
                shouldAttemptEquip = true;
            } else if (equipControl == PlayerEquipControl.MUST_NOT_CROUCH && !player.isShiftKeyDown()) {
                shouldAttemptEquip = true;
            }

            if (shouldAttemptEquip) {
                var accessory = AccessoryRegistry.getAccessoryOrDefault(stack);

                var equipReference = capability.canEquipAccessory(stack, true);

                if (equipReference != null && accessory.canEquipFromUse(stack, equipReference.left())) {
                    accessory.onEquipFromUse(stack, equipReference.left());

                    var newHandStack = stack.copy();

                    var possibleSwappedStack = equipReference.second().equipStack(newHandStack);

                    if (possibleSwappedStack.isPresent()) {
                        var swappedStack = possibleSwappedStack.get();

                        if (newHandStack.isEmpty()) {
                            newHandStack = swappedStack;
                        } else if (ItemStack.isSameItemSameComponents(newHandStack, swappedStack) && (newHandStack.getCount() + swappedStack.getCount()) <= newHandStack.getMaxStackSize()) {
                            newHandStack.grow(swappedStack.getCount());
                        } else {
                            player.addItem(swappedStack);
                        }
                    }

                    return InteractionResult.SUCCESS.heldItemTransformedTo(newHandStack);
                }
            }
        }

        return InteractionResult.PASS;
    }

    public static InteractionResult attemptEquipOnEntity(Player player, InteractionHand hand, Entity entity) {
        var stack = player.getItemInHand(hand);

        if (!(entity instanceof LivingEntity targetEntity) || !entity.getType().is(AccessoriesTags.EQUIPMENT_MANAGEABLE))
            return InteractionResult.PASS;

        var targetCapability = AccessoriesCapability.get(targetEntity);

        var canModify = AllowEntityModificationCallback.EVENT.invoker().allowModifications(targetEntity, player, null).orElse(false);

        if (canModify && targetCapability != null && !player.isSpectator()) {
            if (player.isShiftKeyDown()) {
                var accessory = AccessoryRegistry.getAccessoryOrDefault(stack);

                var equipReference = targetCapability.canEquipAccessory(stack, true);

                if (equipReference != null && accessory.canEquipFromUse(stack, equipReference.left())) {
                    if (!stack.isEmpty()) accessory.onEquipFromUse(stack, equipReference.left());

                    var newHandStack = stack.copy();

                    var possibleSwappedStack = equipReference.second().equipStack(newHandStack);

                    if (possibleSwappedStack.isPresent()) {
                        var swappedStack = possibleSwappedStack.get();

                        if (newHandStack.isEmpty()) {
                            newHandStack = swappedStack;
                        } else if (ItemStack.isSameItemSameComponents(newHandStack, swappedStack) && (newHandStack.getCount() + swappedStack.getCount()) <= newHandStack.getMaxStackSize()) {
                            newHandStack.grow(swappedStack.getCount());
                        } else {
                            player.addItem(swappedStack);
                        }
                    }

                    return InteractionResult.SUCCESS.heldItemTransformedTo(newHandStack);
                }
            }
        }

        return InteractionResult.PASS;
    }

    public static void setupItems(AddDataComponentCallback callback) {
        AccessoryRegistry.getAllAccessories().forEach((item, accessory) -> {
            var builder = AccessoryItemAttributeModifiers.builder();

            accessory.getStaticModifiers(item, builder);

            if (!builder.isEmpty()) {
                callback.addTo(item, AccessoriesDataComponents.ATTRIBUTES, builder.build());
            }
        });
    }

    public interface AddDataComponentCallback {
        <T> void addTo(Item item, DataComponentType<T> componentType, T component);
    }
}
