/*
 * Decompiled with CFR 0.152.
 */
package haider.plugin.healthGUI.display;

import haider.plugin.healthGUI.HealthGUI;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import net.kyori.adventure.text.Component;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.attribute.Attribute;
import org.bukkit.attribute.AttributeInstance;
import org.bukkit.entity.ArmorStand;
import org.bukkit.entity.Entity;
import org.bukkit.entity.EntityType;
import org.bukkit.entity.LivingEntity;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitTask;

public class Display3DManager {
    private final HealthGUI plugin;
    private final Map<UUID, ArmorStand> entityDisplays;
    private final Map<UUID, Long> lastUpdateTime;
    private final Map<UUID, String> lastDisplayText;
    private final Map<UUID, Double> lastHealthValue;
    private final Map<UUID, Location> lastEntityLocations;
    private final Map<UUID, Long> lastLocationUpdate;
    private final Map<UUID, Long> lastCreationAttempt;
    private BukkitTask movementTrackingTask;
    private BukkitTask proximityCheckTask;
    private static final double DISPLAY_HEIGHT_OFFSET = 2.3;
    private static final long UPDATE_INTERVAL_MS = 50L;
    private static final int MAX_DISPLAY_DISTANCE = 50;
    private static final int AUTO_DISPLAY_DISTANCE = 20;
    private static final double MOVEMENT_THRESHOLD = 0.1;

    public Display3DManager(HealthGUI plugin) {
        this.plugin = plugin;
        this.entityDisplays = new ConcurrentHashMap<UUID, ArmorStand>();
        this.lastUpdateTime = new ConcurrentHashMap<UUID, Long>();
        this.lastDisplayText = new ConcurrentHashMap<UUID, String>();
        this.lastHealthValue = new ConcurrentHashMap<UUID, Double>();
        this.lastEntityLocations = new ConcurrentHashMap<UUID, Location>();
        this.lastLocationUpdate = new ConcurrentHashMap<UUID, Long>();
        this.lastCreationAttempt = new ConcurrentHashMap<UUID, Long>();
        this.startMovementTrackingTask();
        this.startProximityCheckingTask();
        plugin.getLogger().info("\u2713 Enhanced 3D Health Display Manager initialized with movement tracking and auto-display");
    }

    private void startMovementTrackingTask() {
        this.movementTrackingTask = Bukkit.getScheduler().runTaskTimer((Plugin)this.plugin, () -> this.updateDisplayPositions(), 0L, 1L);
    }

    private void startProximityCheckingTask() {
        this.proximityCheckTask = Bukkit.getScheduler().runTaskTimer((Plugin)this.plugin, () -> this.checkPlayerProximity(), 0L, 20L);
    }

    private void checkPlayerProximity() {
        if (!this.plugin.getConfigManager().isShow3DHealthDisplay()) {
            return;
        }
        for (Player player : Bukkit.getOnlinePlayers()) {
            if (player == null || !player.isOnline()) continue;
            for (Entity entity : player.getNearbyEntities(20.0, 20.0, 20.0)) {
                LivingEntity livingEntity;
                if (!(entity instanceof LivingEntity) || entity.isDead() || (livingEntity = (LivingEntity)entity) instanceof Player || this.hasActiveDisplay(livingEntity) || this.hasNearbyHealthDisplay(livingEntity)) continue;
                this.createOrUpdate3DDisplay(livingEntity, player);
            }
        }
        this.cleanupDistantDisplays();
    }

    private void cleanupDistantDisplays() {
        this.entityDisplays.entrySet().removeIf(entry -> {
            UUID entityId = (UUID)entry.getKey();
            ArmorStand display = (ArmorStand)entry.getValue();
            if (display == null || display.isDead()) {
                this.cleanupEntityData(entityId);
                return true;
            }
            Entity entity = Bukkit.getEntity((UUID)entityId);
            if (entity == null || entity.isDead()) {
                display.remove();
                this.cleanupEntityData(entityId);
                return true;
            }
            boolean hasNearbyPlayer = false;
            for (Player player : Bukkit.getOnlinePlayers()) {
                double distance;
                if (player == null || !player.isOnline() || !player.getWorld().equals((Object)entity.getWorld()) || !((distance = entity.getLocation().distance(player.getLocation())) <= 20.0)) continue;
                hasNearbyPlayer = true;
                break;
            }
            if (!hasNearbyPlayer) {
                display.remove();
                this.cleanupEntityData(entityId);
                return true;
            }
            return false;
        });
    }

    private void updateDisplayPositions() {
        if (this.entityDisplays.isEmpty()) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        this.entityDisplays.entrySet().removeIf(entry -> {
            UUID entityId = (UUID)entry.getKey();
            ArmorStand display = (ArmorStand)entry.getValue();
            if (display == null || display.isDead()) {
                this.cleanupEntityData(entityId);
                return true;
            }
            Entity entity = Bukkit.getEntity((UUID)entityId);
            if (!(entity instanceof LivingEntity) || entity.isDead()) {
                display.remove();
                this.cleanupEntityData(entityId);
                return true;
            }
            LivingEntity livingEntity = (LivingEntity)entity;
            Location currentLocation = livingEntity.getLocation();
            Location lastLocation = this.lastEntityLocations.get(entityId);
            if (lastLocation == null || currentLocation.distance(lastLocation) >= 0.1) {
                Location newDisplayLocation = this.calculateDisplayLocation(livingEntity);
                this.smoothTeleport(display, newDisplayLocation);
                this.lastEntityLocations.put(entityId, currentLocation.clone());
                this.lastLocationUpdate.put(entityId, currentTime);
            }
            return false;
        });
    }

    private void removeAllDisplaysForEntity(LivingEntity entity) {
        UUID entityId = entity.getUniqueId();
        Location entityLocation = entity.getLocation();
        ArmorStand trackedDisplay = this.entityDisplays.remove(entityId);
        if (trackedDisplay != null && !trackedDisplay.isDead()) {
            trackedDisplay.remove();
        }
        this.cleanupEntityData(entityId);
        try {
            for (Entity nearbyEntity : entity.getWorld().getNearbyEntities(entityLocation, 8.0, 8.0, 8.0)) {
                double distance;
                ArmorStand armorStand;
                if (!(nearbyEntity instanceof ArmorStand) || (armorStand = (ArmorStand)nearbyEntity).isVisible() || !armorStand.isCustomNameVisible() || armorStand.customName() == null || !((distance = armorStand.getLocation().distance(entityLocation)) < 4.0)) continue;
                armorStand.remove();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void cleanupEntityData(UUID entityId) {
        this.lastUpdateTime.remove(entityId);
        this.lastDisplayText.remove(entityId);
        this.lastHealthValue.remove(entityId);
        this.lastEntityLocations.remove(entityId);
        this.lastLocationUpdate.remove(entityId);
        this.lastCreationAttempt.remove(entityId);
    }

    public void show3DHealthDisplay(LivingEntity entity, Player viewer) {
        if (!this.plugin.getConfigManager().isShow3DHealthDisplay()) {
            return;
        }
        if (entity == null || viewer == null || entity.isDead()) {
            return;
        }
        if (entity.getLocation().distance(viewer.getLocation()) > 50.0) {
            this.hide3DHealthDisplay(entity);
            return;
        }
        this.createOrUpdate3DDisplay(entity, viewer);
    }

    private void createOrUpdate3DDisplay(LivingEntity entity, Player viewer) {
        UUID entityId = entity.getUniqueId();
        long currentTime = System.currentTimeMillis();
        Long lastAttempt = this.lastCreationAttempt.get(entityId);
        if (lastAttempt != null && currentTime - lastAttempt < 1000L) {
            return;
        }
        String healthText = this.generateHealthText(entity, viewer);
        ArmorStand display = this.entityDisplays.get(entityId);
        if (display != null) {
            if (display.isDead()) {
                this.entityDisplays.remove(entityId);
                this.cleanupEntityData(entityId);
                display = null;
            } else {
                this.updateExistingDisplay(display, entity, healthText);
                this.lastDisplayText.put(entityId, healthText);
                this.lastUpdateTime.put(entityId, currentTime);
                this.lastHealthValue.put(entityId, entity.getHealth());
                this.lastEntityLocations.put(entityId, entity.getLocation().clone());
                this.lastLocationUpdate.put(entityId, currentTime);
                return;
            }
        }
        this.lastCreationAttempt.put(entityId, currentTime);
        this.removeAllDisplaysForEntity(entity);
        display = this.createNewDisplay(entity, healthText);
        if (display != null) {
            this.entityDisplays.put(entityId, display);
            this.lastDisplayText.put(entityId, healthText);
            this.lastUpdateTime.put(entityId, currentTime);
            this.lastHealthValue.put(entityId, entity.getHealth());
            this.lastEntityLocations.put(entityId, entity.getLocation().clone());
            this.lastLocationUpdate.put(entityId, currentTime);
        }
    }

    private ArmorStand createNewDisplay(LivingEntity entity, String healthText) {
        try {
            Location displayLocation = this.calculateDisplayLocation(entity);
            UUID entityId = entity.getUniqueId();
            if (this.entityDisplays.containsKey(entityId)) {
                return null;
            }
            for (Entity nearbyEntity : entity.getWorld().getNearbyEntities(displayLocation, 3.0, 3.0, 3.0)) {
                ArmorStand existingStand;
                if (!(nearbyEntity instanceof ArmorStand) || (existingStand = (ArmorStand)nearbyEntity).isVisible() || !existingStand.isCustomNameVisible()) continue;
                return null;
            }
            ArmorStand armorStand = (ArmorStand)entity.getWorld().spawnEntity(displayLocation, EntityType.ARMOR_STAND);
            this.setupArmorStandProperties(armorStand);
            Component textComponent = this.plugin.getMiniMessage().deserialize((Object)healthText);
            armorStand.customName(textComponent);
            return armorStand;
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to create 3D health display: " + e.getMessage());
            e.printStackTrace();
            return null;
        }
    }

    private void setupArmorStandProperties(ArmorStand armorStand) {
        armorStand.setVisible(false);
        armorStand.setCustomNameVisible(true);
        armorStand.setGravity(false);
        armorStand.setCanPickupItems(false);
        armorStand.setBasePlate(false);
        armorStand.setArms(false);
        armorStand.setSmall(true);
        armorStand.setMarker(true);
        armorStand.setSilent(true);
        armorStand.setInvulnerable(true);
        armorStand.setCollidable(false);
        armorStand.setCanMove(false);
    }

    private void updateExistingDisplay(ArmorStand display, LivingEntity entity, String healthText) {
        try {
            String lastText;
            Location newLocation = this.calculateDisplayLocation(entity);
            Location currentDisplayLocation = display.getLocation();
            if (this.needsPositionUpdate(currentDisplayLocation, newLocation)) {
                this.smoothTeleport(display, newLocation);
            }
            if ((lastText = this.lastDisplayText.get(entity.getUniqueId())) == null || !lastText.equals(healthText)) {
                Component textComponent = this.plugin.getMiniMessage().deserialize((Object)healthText);
                display.customName(textComponent);
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to update 3D health display: " + e.getMessage());
            this.hide3DHealthDisplay(entity);
        }
    }

    private boolean needsPositionUpdate(Location current, Location target) {
        if (current.getWorld() != target.getWorld()) {
            return true;
        }
        double distance = current.distance(target);
        return distance >= 0.1;
    }

    private void smoothTeleport(ArmorStand display, Location targetLocation) {
        Location smoothLocation = targetLocation.clone();
        Location currentLoc = display.getLocation();
        smoothLocation.setYaw(currentLoc.getYaw());
        smoothLocation.setPitch(currentLoc.getPitch());
        display.teleport(smoothLocation);
    }

    private Location calculateDisplayLocation(LivingEntity entity) {
        Location entityLocation = entity.getLocation().clone();
        double heightOffset = 2.3;
        switch (entity.getType()) {
            case ENDER_DRAGON: {
                heightOffset = 4.0;
                break;
            }
            case WITHER: 
            case GIANT: {
                heightOffset = 3.5;
                break;
            }
            case HORSE: 
            case COW: 
            case PIG: {
                heightOffset = 2.0;
                break;
            }
            case CHICKEN: 
            case RABBIT: {
                heightOffset = 1.5;
                break;
            }
            default: {
                heightOffset = 2.3;
            }
        }
        entityLocation.add(0.0, heightOffset, 0.0);
        entityLocation.add(0.5, 0.0, 0.5);
        return entityLocation;
    }

    private String generateHealthText(LivingEntity entity, Player viewer) {
        double currentHealth = entity.getHealth();
        AttributeInstance maxHealthAttribute = entity.getAttribute(Attribute.GENERIC_MAX_HEALTH);
        double maxHealth = maxHealthAttribute != null ? maxHealthAttribute.getValue() : 20.0;
        double healthPercentage = currentHealth / maxHealth * 100.0;
        String entityName = this.getEntityDisplayName(entity);
        String healthBar = this.plugin.getConfigManager().createHealthBar(currentHealth, maxHealth);
        String healthColor = this.plugin.getConfigManager().getHealthColorByPercentage(healthPercentage);
        String heartSymbol = this.plugin.getConfigManager().getHeartSymbol();
        StringBuilder displayText = new StringBuilder();
        if (this.plugin.getConfigManager().isShowEntityName()) {
            displayText.append("<white><bold>").append(entityName).append("</bold></white>");
            displayText.append(" ");
        }
        if (this.plugin.getConfigManager().isShowHeartSymbol()) {
            displayText.append("<red>").append(heartSymbol).append("</red>");
            displayText.append(" ");
        }
        if (this.plugin.getConfigManager().isShowHealthValues()) {
            displayText.append("<").append(healthColor).append(">").append(String.format("%.1f", currentHealth)).append("/").append(String.format("%.1f", maxHealth)).append("</").append(healthColor).append(">");
            displayText.append(" ");
        }
        if (this.plugin.getConfigManager().isShowPercentage()) {
            displayText.append("<gray>(</gray>").append("<").append(healthColor).append(">").append(String.format("%.0f", healthPercentage)).append("%").append("</").append(healthColor).append(">").append("<gray>)</gray>");
            displayText.append(" ");
        }
        if (this.plugin.getConfigManager().isShowHealthBar()) {
            displayText.append(healthBar);
        }
        return displayText.toString().trim();
    }

    private String getEntityDisplayName(LivingEntity entity) {
        if (entity instanceof Player) {
            return ((Player)entity).getName();
        }
        Component customName = entity.customName();
        if (customName != null) {
            return (String)this.plugin.getMiniMessage().serialize(customName);
        }
        String typeName = entity.getType().name();
        return this.formatEntityTypeName(typeName);
    }

    private String formatEntityTypeName(String typeName) {
        return typeName.substring(0, 1).toUpperCase() + typeName.substring(1).toLowerCase().replace("_", " ");
    }

    public void hide3DHealthDisplay(LivingEntity entity) {
        if (entity == null) {
            return;
        }
        UUID entityId = entity.getUniqueId();
        ArmorStand display = this.entityDisplays.remove(entityId);
        if (display != null && !display.isDead()) {
            display.remove();
        }
        this.cleanupEntityData(entityId);
    }

    public void hideAllDisplaysForPlayer(Player player) {
        for (ArmorStand display : this.entityDisplays.values()) {
            double distance;
            if (display != null && !display.isDead() && display.getLocation().getWorld() == player.getWorld() && !((distance = display.getLocation().distance(player.getLocation())) <= 50.0)) continue;
        }
    }

    public void cleanupAllDisplays() {
        if (this.movementTrackingTask != null && !this.movementTrackingTask.isCancelled()) {
            this.movementTrackingTask.cancel();
        }
        if (this.proximityCheckTask != null && !this.proximityCheckTask.isCancelled()) {
            this.proximityCheckTask.cancel();
        }
        for (ArmorStand display : this.entityDisplays.values()) {
            if (display == null || display.isDead()) continue;
            display.remove();
        }
        this.entityDisplays.clear();
        this.lastUpdateTime.clear();
        this.lastDisplayText.clear();
        this.lastHealthValue.clear();
        this.lastEntityLocations.clear();
        this.lastLocationUpdate.clear();
        this.lastCreationAttempt.clear();
        this.plugin.getLogger().info("\u2713 All 3D health displays cleaned up");
    }

    public void cleanupInvalidDisplays() {
        this.entityDisplays.entrySet().removeIf(entry -> {
            UUID entityId = (UUID)entry.getKey();
            ArmorStand display = (ArmorStand)entry.getValue();
            if (display == null || display.isDead()) {
                this.cleanupEntityData(entityId);
                return true;
            }
            Entity entity = Bukkit.getEntity((UUID)entityId);
            if (entity == null || entity.isDead()) {
                display.remove();
                this.cleanupEntityData(entityId);
                return true;
            }
            return false;
        });
    }

    public int getActiveDisplaysCount() {
        return this.entityDisplays.size();
    }

    public boolean hasActiveDisplay(LivingEntity entity) {
        if (entity == null) {
            return false;
        }
        ArmorStand display = this.entityDisplays.get(entity.getUniqueId());
        return display != null && !display.isDead();
    }

    private boolean hasNearbyHealthDisplay(LivingEntity entity) {
        if (entity == null) {
            return false;
        }
        Location entityLocation = entity.getLocation();
        for (Entity nearbyEntity : entity.getWorld().getNearbyEntities(entityLocation, 5.0, 5.0, 5.0)) {
            double distance;
            ArmorStand armorStand;
            if (!(nearbyEntity instanceof ArmorStand) || (armorStand = (ArmorStand)nearbyEntity).isVisible() || !armorStand.isCustomNameVisible() || armorStand.customName() == null || !((distance = armorStand.getLocation().distance(entityLocation)) < 4.0)) continue;
            return true;
        }
        return false;
    }

    public void restart3DSystem() {
        this.cleanupAllDisplays();
        this.entityDisplays.clear();
        this.lastUpdateTime.clear();
        this.lastDisplayText.clear();
        this.lastHealthValue.clear();
        this.lastEntityLocations.clear();
        this.lastLocationUpdate.clear();
        this.lastCreationAttempt.clear();
        if (this.movementTrackingTask == null || this.movementTrackingTask.isCancelled()) {
            this.startMovementTrackingTask();
        }
        if (this.proximityCheckTask == null || this.proximityCheckTask.isCancelled()) {
            this.startProximityCheckingTask();
        }
        this.plugin.getLogger().info("\u2713 3D Health Display System restarted successfully");
    }

    public Map<String, Object> getSystemStatus() {
        HashMap<String, Object> status = new HashMap<String, Object>();
        status.put("activeDisplays", this.getActiveDisplaysCount());
        status.put("enabled", this.plugin.getConfigManager().isShow3DHealthDisplay());
        status.put("maxDistance", 50);
        status.put("autoDisplayDistance", 20);
        status.put("updateInterval", 50L);
        status.put("memoryUsage", this.entityDisplays.size() + this.lastUpdateTime.size() + this.lastDisplayText.size() + this.lastEntityLocations.size() + this.lastLocationUpdate.size());
        status.put("movementTrackingActive", this.movementTrackingTask != null && !this.movementTrackingTask.isCancelled());
        status.put("proximityCheckingActive", this.proximityCheckTask != null && !this.proximityCheckTask.isCancelled());
        status.put("trackedEntityMovements", this.lastEntityLocations.size());
        return status;
    }
}

